/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        OrthogPolyApproximation
//- Description:  Class for Polynomial Chaos Approximation
//-               
//- Owner:        Mike Eldred

#ifndef ORTHOG_POLY_APPROXIMATION_H
#define ORTHOG_POLY_APPROXIMATION_H

#include "BasisPolyApproximation.H"
#include "BasisPolynomial.H"
#include "DataModel.H"


namespace Dakota {

class ProblemDescDB;


/// Derived approximation class for orthogonal polynomials (global
/// approximation).

/** The OrthogPolyApproximation class provides a global approximation
    based on orthogonal polynomials.  It is used primarily for polynomial
    chaos expansions (for stochastic finite element approaches to
    uncertainty quantification). */

class OrthogPolyApproximation: public BasisPolyApproximation
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  OrthogPolyApproximation();
  /// standard constructor
  OrthogPolyApproximation(const ProblemDescDB& problem_db,
			  const size_t& num_acv);
  /// destructor
  ~OrthogPolyApproximation();

  //
  //- Heading: Member functions
  //

  /// set numExpansionTerms
  void expansion_terms(const int& exp_terms);
  /// get numExpansionTerms
  const int& expansion_terms() const;

  /// set basisTypes
  void basis_types(const ShortArray& basis);
  /// get basisTypes
  const ShortArray& basis_types() const;

  /// pass alpha_stat parameters to JACOBI polynomial bases
  void jacobi_alphas(const RealDenseVector& alphas);
  /// pass beta_stat parameters to JACOBI polynomial bases
  void jacobi_betas(const RealDenseVector& betas);
  /// pass alpha_stat parameters to GENERALIZED_LAGUERRE polynomial bases
  void generalized_laguerre_alphas(const RealDenseVector& alphas);

  /// calculate number of chaos coefficients and expansion order
  /// (numExpansionTerms and approxOrder) based on user input
  void resolve_inputs();
  /// initialize polynomialBasis, multiIndex, et al.
  void allocate_arrays();

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  int min_coefficients() const;

  /// find the coefficients for the expansion of multivariate
  /// orthogonal polynomials
  void find_coefficients();

  /// print the coefficients for the expansion
  void print_coefficients(ostream& s) const;

  /// retrieve the response PCE value for a given parameter vector
  const Real& get_value(const RealVector& x);
  /// retrieve the response PCE gradient for a given parameter vector
  /// and default DVV
  const RealBaseVector& get_gradient(const RealVector& x);
  /// retrieve the response PCE gradient for a given parameter vector
  /// and given DVV
  const RealBaseVector& get_gradient(const RealVector& x, const UIntArray& dvv);

  /// return the mean of the PCE, treating all variables as random
  const Real& get_mean();
  /// return the mean of the PCE for a given parameter vector,
  /// treating a subset of the variables as random
  const Real& get_mean(const RealVector& x);
  /// return the gradient of the PCE mean for a given parameter
  /// vector, treating all variables as random
  const RealBaseVector& get_mean_gradient();
  /// return the gradient of the PCE mean for a given parameter vector
  /// and given DVV, treating a subset of the variables as random
  const RealBaseVector& get_mean_gradient(const RealVector& x,
					  const UIntArray& dvv);

  /// return the variance of the PCE, treating all variables as random
  const Real& get_variance();
  /// return the variance of the PCE for a given parameter vector,
  /// treating a subset of the variables as random
  const Real& get_variance(const RealVector& x);
  /// return the gradient of the PCE variance for a given parameter
  /// vector, treating all variables as random
  const RealBaseVector& get_variance_gradient();
  /// return the gradient of the PCE variance for a given parameter
  /// vector and given DVV, treating a subset of the variables as random
  const RealBaseVector& get_variance_gradient(const RealVector& x,
					      const UIntArray& dvv);

  /// returns the norm-squared of a particular multivariate polynomial,
  /// treating all variables as random
  const Real& norm_squared(size_t expansion_index);
  /// returns the norm-squared of a particular multivariate polynomial,
  /// treating a subset of the variables as random
  const Real& norm_squared_random(size_t expansion_index);

private:

  //
  //- Heading: Member functions
  //

  /// calculate complete basis of multivariate orthogonal polynomials
  /// evaluated at a particular parameter set
  const RealVector& get_multivariate_polynomials(const RealVector& xi);
  // calculate a particular multivariate orthogonal polynomial
  // evaluated at a particular parameter set
  //Real get_multivariate_polynomial(const RealVector& x, int term);

  /// computes the chaosCoeffs via numerical integration
  /// (expCoeffsSolnApproach is QUADRATURE or SPARSE_GRID)
  void integration();
  /// computes the chaosCoeffs via linear regression
  /// (expCoeffsSolnApproach is REGRESSION)
  void regression();
  /// computes the chaosCoeffs via averaging of samples
  /// (expCoeffsSolnApproach is SAMPLING)
  void expectation();

  /// cross-validates alternate gradient expressions
  void gradient_check();

  //
  //- Heading: Data
  //

  /// number of terms in Polynomial Chaos expansion (length of chaosCoeffs)
  int numExpansionTerms;

  /// array of basis types for each one-dimensional orthogonal polynomial:
  /// HERMITE, LEGENDRE, LAGUERRE, JACOBI, or GENERALIZED_LAGUERRE
  ShortArray basisTypes;

  /// array of one-dimensional basis polynomial objects which are used in
  /// constructing the multivariate orthogonal/interpolation polynomials
  Array<BasisPolynomial> polynomialBasis;

  /// numExpansionTerms-by-numVars array for identifying the orders of
  /// the one-dimensional orthogonal polynomials contributing to each
  /// of the multivariate orthogonal polynomials
  UShort2DArray multiIndex;

  /// a basis set of multivariate orthogonal polynomials evaluated at
  /// a particular xi (returned by get_multivariate_polynomials())
  RealVector wienerAskeyChaos;

  /// norm-squared of one of the multivariate polynomial basis functions
  Real multiPolyNormSq;

  /// the orthogonal polynomial values (Psi_j) at each of the simulation
  /// sample points (num_pts array of numExpansionTerms vectors)
  RealVectorArray chaosSamples;
};


inline OrthogPolyApproximation::OrthogPolyApproximation(): numExpansionTerms(0)
{ }


inline OrthogPolyApproximation::~OrthogPolyApproximation()
{ }


inline void OrthogPolyApproximation::expansion_terms(const int& exp_terms)
{ numExpansionTerms = exp_terms; }


inline const int& OrthogPolyApproximation::expansion_terms() const
{ return numExpansionTerms; }


inline void OrthogPolyApproximation::basis_types(const ShortArray& basis)
{
  if (basis.length() != numVars) {
    Cerr << "Error: incoming basis array length (" << basis.length()
	 << ") does not match number of variables (" << numVars
	 << ") in OrthogPolyApproximation::basis_types()." << endl;
    abort_handler(-1);
  }
  if (basisTypes != basis) {
    basisTypes = basis;
    if (polynomialBasis.empty())
      polynomialBasis.reshape(numVars);
    for (size_t i=0; i<numVars; i++)
      polynomialBasis[i] = BasisPolynomial(basisTypes[i]);
  }
}


inline const ShortArray& OrthogPolyApproximation::basis_types() const
{ return basisTypes; }


inline void OrthogPolyApproximation::
jacobi_alphas(const RealDenseVector& alphas)
{
  size_t i, num_alphas = alphas.length(), cntr = 0;
  if (basisTypes.count(JACOBI) != num_alphas) {
    Cerr << "Error: mismatch in array length in OrthogPolyApproximation::"
	 << "jacobi_alphas()." << endl;
    abort_handler(-1);
  }
  for (i=0; i<numVars; i++)
    if (basisTypes[i] == JACOBI)
      polynomialBasis[i].alpha_stat(alphas[cntr++]);
}


inline void OrthogPolyApproximation::jacobi_betas(const RealDenseVector& betas)
{
  size_t i, num_betas = betas.length(), cntr = 0;
  if (basisTypes.count(JACOBI) != num_betas) {
    Cerr << "Error: mismatch in array length in OrthogPolyApproximation::"
	 << "jacobi_betas()." << endl;
    abort_handler(-1);
  }
  for (i=0; i<numVars; i++)
    if (basisTypes[i] == JACOBI)
      polynomialBasis[i].beta_stat(betas[cntr++]);
}


inline void OrthogPolyApproximation::
generalized_laguerre_alphas(const RealDenseVector& alphas)
{
  size_t i, num_alphas = alphas.length(), cntr = 0;
  if (basisTypes.count(GENERALIZED_LAGUERRE) != num_alphas) {
    Cerr << "Error: mismatch in array length in OrthogPolyApproximation::"
	 << "generalized_laguerre_alphas()." << endl;
    abort_handler(-1);
  }
  for (i=0; i<numVars; i++)
    if (basisTypes[i] == GENERALIZED_LAGUERRE)
      polynomialBasis[i].alpha_stat(alphas[cntr++]);
}

} // namespace Dakota

#endif
