/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        CommandShell
//- Description:
//-
//-
//- Owner:        Bill Bohnhoff
//- Version: $Id: CommandShell.H 3989 2006-09-27 13:59:30Z slbrow $

#ifndef COMMAND_SHELL_H
#define COMMAND_SHELL_H

#include "data_types.h"


namespace Dakota {

/// Utility class which defines convenience operators for spawning
/// processes with system calls.

/** The CommandShell class wraps the C system() utility and defines
    convenience operators for building a command string and then
    passing it to the shell. */

class CommandShell
{
public:

  //
  //- Heading: Constructor and destructor
  //

  CommandShell();  ///< constructor
  ~CommandShell(); ///< destructor

  //
  //- Heading: Operator overloaded functions
  //

  /// adds string to unixCommand
  CommandShell& operator<<(const char* string);

  /// allows passing of the flush function to the shell using <<
  CommandShell& operator<<(CommandShell& (*f)(CommandShell&));

  /// "flushes" the shell; i.e. executes the unixCommand
  CommandShell& flush();

  void asynch_flag(const bool flag);          ///< set the asynchFlag
  bool asynch_flag() const;                   ///< get the asynchFlag

  void suppress_output_flag(const bool flag); ///< set the suppressOutputFlag
  bool suppress_output_flag() const;          ///< get the suppressOutputFlag

private:

  //
  //- Heading: Data members
  //

  /// the command string that is constructed through one or more <<
  /// insertions and then executed by flush
  String unixCommand;

  /// flags nonblocking operation (background system calls)
  bool asynchFlag;
  /// flags suppression of shell output (no command echo)
  bool suppressOutputFlag;
};


/// convenient shell manipulator function to "flush" the shell
CommandShell& flush(CommandShell& shell);

inline void CommandShell::asynch_flag(const bool flag)
{ asynchFlag = flag; }
 
inline bool CommandShell::asynch_flag() const
{ return asynchFlag; }

inline void CommandShell::suppress_output_flag(const bool flag)
{ suppressOutputFlag = flag; }
 
inline bool CommandShell::suppress_output_flag() const
{ return suppressOutputFlag; }

} // namespace Dakota

#endif
