/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Approximation
//- Description:  Abstract base class for approximations
//-               
//- Owner:        Mike Eldred

#ifndef DAKOTA_APPROXIMATION_H
#define DAKOTA_APPROXIMATION_H

#include "data_types.h"

namespace Dakota {

class ProblemDescDB;
class SurrogateDataPointRep;


/// Container class encapsulating basic parameter and response data
/// for defining a "truth" data point.

/** A list of these data points is contained in each Approximation
    instance (Approximation::currentPoints) and provides the data to
    build the approximation.  A handle-body idiom is used to avoid
    excessive data copying overhead. */

class SurrogateDataPoint
{
public:

  //
  //- Heading: Constructors, destructor, and operators
  //

  /// default constructor
  SurrogateDataPoint();
  /// standard constructor
  SurrogateDataPoint(const RealVector& x, const Real& fn_val,
		     const RealBaseVector& fn_grad, const RealMatrix& fn_hess);
  /// copy constructor
  SurrogateDataPoint(const SurrogateDataPoint& sdp);
  /// destructor
  ~SurrogateDataPoint();

  /// assignment operator
  SurrogateDataPoint& operator=(const SurrogateDataPoint& sdp);
  /// equality operator
  bool operator==(const SurrogateDataPoint& sdp) const;

  //
  //- Heading: member functions
  //

  const RealVector&     continuous_variables() const; ///< return continuousVars
  const Real&           response_function()    const; ///< return responseFn
  const RealBaseVector& response_gradient()    const; ///< return responseGrad
  const RealMatrix&     response_hessian()     const; ///< return responseHess

  /// function to check sdpRep (does this handle contain a body)
  bool is_null() const;

private:

  //
  //- Heading: Private data members
  //
 
  /// pointer to the body (handle-body idiom)
  SurrogateDataPointRep* sdpRep;
};


/// The representation of a surrogate data point.  This representation,
/// or body, may be shared by multiple SurrogateDataPoint handle instances.

/** The SurrogateDataPoint/SurrogateDataPointRep pairs utilize a
    handle-body idiom (Coplien, Advanced C++). */

class SurrogateDataPointRep
{
  //
  //- Heading: Friends
  //

  /// the handle class can access attributes of the body class directly
  friend class SurrogateDataPoint;

private:

  //
  //- Heading: Private member functions
  //

  /// constructor
  SurrogateDataPointRep(const RealVector& x, const Real& fn_val,
			const RealBaseVector& fn_grad,
			const RealMatrix& fn_hess);
  /// destructor
  ~SurrogateDataPointRep();

  //
  //- Heading: Private data members
  //

  RealVector     continuousVars; ///< continuous variables
  Real           responseFn;     ///< truth response function value
  RealBaseVector responseGrad;   ///< truth response function gradient
  RealMatrix     responseHess;   ///< truth response function Hessian

  /// number of handle objects sharing sdpRep
  int referenceCount;
};


/// Base class for the approximation class hierarchy.

/** The Approximation class is the base class for the response data
    fit approximation class hierarchy in DAKOTA.  One instance of an
    Approximation must be created for each function to be approximated
    (a vector of Approximations is contained in
    ApproximationInterface).  For memory efficiency and enhanced
    polymorphism, the approximation hierarchy employs the
    "letter/envelope idiom" (see Coplien "Advanced C++", p. 133), for
    which the base class (Approximation) serves as the envelope and
    one of the derived classes (selected in
    Approximation::get_approximation()) serves as the letter. */

class Approximation
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  /// default constructor
  Approximation();
  /// standard constructor for envelope
  Approximation(ProblemDescDB& problem_db, const size_t& num_vars);
   /// alternate constructor
  Approximation(const String& approx_type, short approx_order,
		const size_t& num_vars);
  /// copy constructor
  Approximation(const Approximation& approx);

  /// destructor
  virtual ~Approximation();

  /// assignment operator
  Approximation operator=(const Approximation& approx);

  //
  //- Heading: Virtual functions
  //

  /// retrieve the approximate function value for a given parameter vector
  virtual const Real& get_value(const RealVector& x);
  /// retrieve the approximate function gradient for a given parameter vector
  virtual const RealBaseVector& get_gradient(const RealVector& x);
  /// retrieve the approximate function Hessian for a given parameter vector
  virtual const RealMatrix& get_hessian(const RealVector& x);
  /// retrieve the variance of the predicted value for a given parameter vector
  virtual const Real& get_variance(const RealVector& x);
  /// retrieve the diagnostic metric for the diagnostic type specified
  virtual const Real& get_diagnostic(const String& metric_type);

  /// return the coefficient array computed by find_coefficients()
  virtual const RealVector& approximation_coefficients() const;
  /// set the coefficient array from external sources, rather than
  /// computing with find_coefficients()
  virtual void approximation_coefficients(const RealVector& approx_coeffs);

  /// print the coefficient array computed in find_coefficients()
  virtual void print_coefficients(ostream& s) const;

  /// return the minimum number of samples (unknowns) required to
  /// build the derived class approximation type in numVars dimensions
  virtual int min_coefficients() const;

  /// return the recommended number of samples (unknowns) required to
  /// build the derived class approximation type in numVars dimensions
  virtual int recommended_coefficients() const;

  /// return the number of constraints to be enforced via anchorPoint
  virtual int num_constraints() const;

  /// clear current build data in preparation for next build
  virtual void clear_current();

  /// check if diagnostics are available for this approximation type
  virtual const bool diagnostics_available();

  //
  //- Heading: Member functions
  //

  /// return the minimum number of samples required to build the approximation
  /// type in numVars dimensions. Uses *_coefficients() and num_constraints().
  int min_samples(bool constraint_flag) const;

  /// return the recommended number of samples to build the approximation type
  /// in numVars dimensions (default same as min_samples)
  int recommended_samples(bool constraint_flag) const;

  /// return the number of variables used in the approximation
  int num_variables() const;

  // set approxOrder
  //void approximation_order(short approx_order);

  /// return currentPoints
  const List<SurrogateDataPoint>& current_points() const;
  /// return anchorPoint
  const SurrogateDataPoint& anchor_point() const;

  /// populates/replaces anchorPoint
  void update(const Variables& vars, const Response& response,
	      const int& fn_index);
  /// populates/replaces anchorPoint
  void update(const RealVector& c_vars, const Real& fn_val,
	      const RealBaseVector& fn_grad, const RealMatrix& fn_hess);
  /// populates/replaces currentPoints
  void update(const VariablesArray& vars_array,
	      const ResponseArray& resp_array, const int& fn_index);

  /// appends one additional entry to currentPoints
  void append(const Variables& vars, const Response& response,
	      const int& fn_index);
  /// appends one additional entry to currentPoints
  void append(const RealVector& c_vars, const Real& fn_val,
	      const RealBaseVector& fn_grad, const RealMatrix& fn_hess);
  /// appends multiple additional entries to currentPoints
  void append(const VariablesArray& vars_array,
	      const ResponseArray& resp_array, const int& fn_index);

  /// builds the approximation by invoking find_coefficients()
  void build();

  /// queries the status of anchorPoint
  bool anchor() const;

  /// clear all build data (current and history) to restore original state
  void clear_all();

  /// set approximation lower and upper bounds (currently only used by graphics)
  void set_bounds(const RealVector& lower, const RealVector& upper);

  /// render the approximate surface using the 3D graphics (2 variable
  /// problems only)
  void draw_surface();

  /// returns approxRep for access to derived class member functions
  /// that are not mapped to the top Approximation level
  Approximation* approx_rep() const;

protected:

  //
  //- Heading: Constructors
  //

  /// constructor initializes the base class part of letter classes
  /// (BaseConstructor overloading avoids infinite recursion in the
  /// derived class constructors - Coplien, p. 139)
  Approximation(BaseConstructor, const ProblemDescDB& problem_db,
		const size_t& num_vars);

  //
  //- Heading: Virtual functions
  //

  /// calculate the data fit coefficients using currentPoints and anchorPoint
  virtual void find_coefficients();

  //
  //- Heading: Data
  //

  /// flag signaling the use of gradient data in global approximation builds as
  /// indicated by the user's \c use_gradients specification.  This setting
  /// cannot be inferred from the responses spec., since we may need gradient
  /// support for evaluating gradients at a single point (e.g., the center of a
  /// trust region), but not require gradient evaluations at every point.
  bool useGradsFlag;

  /// output verbosity level: {SILENT,QUIET,NORMAL,VERBOSE,DEBUG}_OUTPUT
  short outputLevel;

  /// number of variables in the approximation
  int numVars;

  /// approximation type identifier
  String approxType;
  /// order of approximation (used by regression polynomials,
  /// orthogonal polynomials, and Taylor series)
  short approxOrder;

  /// value of the approximation returned by get_value()
  Real approxValue;
  /// gradient of the approximation returned by get_gradient()
  RealBaseVector approxGradient;
  /// Hessian of the approximation returned by get_hessian()
  RealMatrix approxHessian;
  /// value of the approximation returned by get_variance()
  Real approxVariance;
  /// value of the diagnostic returned by get_diagnostic()
  Real approxDiagnostic;

  /// list of samples used to build the approximation.  These sample points
  /// are fit approximately (e.g., using least squares regression).
  List<SurrogateDataPoint> currentPoints;

  /// a special sample (often at the center of the approximation region)
  /// for which exact matching is enforced (e.g., using equality-constrained
  /// least squares regression).
  SurrogateDataPoint anchorPoint;

private:

  //
  //- Heading: Member functions
  //

  /// Used only by the standard envelope constructor to initialize
  /// approxRep to the appropriate derived type.
  Approximation* get_approx(ProblemDescDB& problem_db, const size_t& num_vars);

  /// Used only by the alternate envelope constructor to initialize
  /// approxRep to the appropriate derived type.
  Approximation* get_approx(const String& approx_type, short approx_order,
			    const size_t& num_vars);

  /// add a new data point by either appending to currentPoints or assigning
  /// to anchorPoint, as dictated by anchor_flag.  Uses add_point() and
  /// add_anchor().
  void add(const Variables& vars, const Response& response,
	   const int& fn_index, bool anchor_flag);

  /// add a new data point by appending to currentPoints
  void add_point(const RealVector& x, const Real& fn_val,
		 const RealBaseVector& fn_grad, const RealMatrix& fn_hess);

  /// add a new data point by assigning to anchorPoint
  void add_anchor(const RealVector& x, const Real& fn_val,
		  const RealBaseVector& fn_grad, const RealMatrix& fn_hess);

  //
  //- Heading: Data
  //

  /// approximation lower bounds (used only by 3D graphics)
  RealVector approxLowerBounds;
  /// approximation upper bounds (used only by 3D graphics)
  RealVector approxUpperBounds;

  /// pointer to the letter (initialized only for the envelope)
  Approximation* approxRep;
  /// number of objects sharing approxRep
  int referenceCount;
};


inline SurrogateDataPointRep::
SurrogateDataPointRep(const RealVector& x,           const Real& fn_val,
		      const RealBaseVector& fn_grad, const RealMatrix& fn_hess):
  referenceCount(1)
{
  continuousVars = x;       responseFn   = fn_val;
  responseGrad   = fn_grad; responseHess = fn_hess;
}


inline SurrogateDataPointRep::~SurrogateDataPointRep()
{ }


inline SurrogateDataPoint::SurrogateDataPoint(): sdpRep(NULL)
{ }


inline SurrogateDataPoint::
SurrogateDataPoint(const RealVector& x,           const Real& fn_val,
		   const RealBaseVector& fn_grad, const RealMatrix& fn_hess):
  sdpRep(new SurrogateDataPointRep(x, fn_val, fn_grad, fn_hess))
{ }


inline SurrogateDataPoint::SurrogateDataPoint(const SurrogateDataPoint& sdp)
{
  // Increment new (no old to decrement)
  sdpRep = sdp.sdpRep;
  if (sdpRep) // Check for an assignment of NULL
    sdpRep->referenceCount++;
}


inline SurrogateDataPoint::~SurrogateDataPoint()
{
  if (sdpRep) { // Check for NULL
    --sdpRep->referenceCount; // decrement
    if (sdpRep->referenceCount == 0)
      delete sdpRep;
  }
}


inline SurrogateDataPoint& SurrogateDataPoint::
operator=(const SurrogateDataPoint& sdp)
{
  // Decrement old
  if (sdpRep) // Check for NULL
    if ( --sdpRep->referenceCount == 0 ) 
      delete sdpRep;
  // Increment new
  sdpRep = sdp.sdpRep;
  if (sdpRep) // Check for an assignment of NULL
    sdpRep->referenceCount++;
  return *this;
}


inline bool SurrogateDataPoint::operator==(const SurrogateDataPoint& sdp) const
{
  return ( sdpRep->continuousVars == sdp.sdpRep->continuousVars &&
	   sdpRep->responseFn     == sdp.sdpRep->responseFn     &&
	   sdpRep->responseGrad   == sdp.sdpRep->responseGrad   &&
	   sdpRep->responseHess   == sdp.sdpRep->responseHess ) ? true : false;
}


inline const RealVector& SurrogateDataPoint::continuous_variables() const
{ return sdpRep->continuousVars; }


inline const Real& SurrogateDataPoint::response_function() const
{ return sdpRep->responseFn; }


inline const RealBaseVector& SurrogateDataPoint::response_gradient() const
{ return sdpRep->responseGrad; }


inline const RealMatrix& SurrogateDataPoint::response_hessian() const
{ return sdpRep->responseHess; }


inline bool SurrogateDataPoint::is_null() const
{ return (sdpRep) ? false : true; }


inline bool Approximation::anchor() const
{
  return (approxRep) ? !approxRep->anchorPoint.is_null() : 
                       !anchorPoint.is_null();
}


inline int Approximation::num_variables() const
{ return (approxRep) ? approxRep->numVars : numVars; }


inline const List<SurrogateDataPoint>& Approximation::current_points() const
{ return (approxRep) ? approxRep->currentPoints : currentPoints; }


inline const SurrogateDataPoint& Approximation::anchor_point() const
{ return (approxRep) ? approxRep->anchorPoint : anchorPoint; }


inline void Approximation::
set_bounds(const RealVector& lower, const RealVector& upper)
{
  if (approxRep) {
    approxRep->approxLowerBounds = lower;
    approxRep->approxUpperBounds = upper;
  }
  else {
    approxLowerBounds = lower;
    approxUpperBounds = upper;
  }
}


inline Approximation* Approximation::approx_rep() const
{ return approxRep; }


// private convenience fn -> approxRep forward not needed.
inline void Approximation::
add_point(const RealVector& x,           const Real& fn_val,
	  const RealBaseVector& fn_grad, const RealMatrix& fn_hess)
{
  SurrogateDataPoint sdp(x, fn_val, fn_grad, fn_hess);
  currentPoints.insert(sdp);
}


// private convenience fn -> approxRep forward not needed.
inline void Approximation::
add_anchor(const RealVector& x,           const Real& fn_val,
	   const RealBaseVector& fn_grad, const RealMatrix& fn_hess)
{ anchorPoint = SurrogateDataPoint(x, fn_val, fn_grad, fn_hess); }


inline void Approximation::
update(const Variables& vars, const Response& response, const int& fn_index)
{
  if (approxRep)
    approxRep->add(vars, response, fn_index, true);
  else
    add(vars, response, fn_index, true); // replace anchorPoint
}


inline void Approximation::
update(const RealVector& c_vars,      const Real& fn_val,
       const RealBaseVector& fn_grad, const RealMatrix& fn_hess)
{
  if (approxRep)
    approxRep->add_anchor(c_vars, fn_val, fn_grad, fn_hess);
  else
    add_anchor(c_vars, fn_val, fn_grad, fn_hess); // replace anchorPoint
}


inline void Approximation::
append(const Variables& vars, const Response& response, const int& fn_index)
{
  if (approxRep)
    approxRep->add(vars, response, fn_index, false);
  else
    add(vars, response, fn_index, false); // append new point to currentPoints
}


inline void Approximation::
append(const RealVector& c_vars,      const Real& fn_val,
       const RealBaseVector& fn_grad, const RealMatrix& fn_hess)
{
  if (approxRep)
    approxRep->add_point(c_vars, fn_val, fn_grad, fn_hess);
  else
    add_point(c_vars, fn_val, fn_grad, fn_hess); // append new pt to currentPts
}


//inline void Approximation::approximation_order(short approx_order)
//{
//  if (approxRep) // envelope fwd to letter
//    approxRep->approxOrder = approx_order;
//  else
//    approxOrder = approx_order;
//}


/** Clears out any history (e.g., TANA3Approximation use for a
    different response function in NonDReliability). */
inline void Approximation::clear_all()
{
  if (approxRep) // envelope fwd to letter
    approxRep->clear_all();
  else { // not virtual: base class implementation
    if (!anchorPoint.is_null())
      anchorPoint = SurrogateDataPoint();
    currentPoints.clear();
  }
}


/** Redefined by TANA3Approximation to clear current data but preserve
    history. */
inline void Approximation::clear_current()
{
  if (approxRep) // envelope fwd to letter
    approxRep->clear_current();
  else // default implementation
    clear_all();
}

} // namespace Dakota

#endif
