#!/bin/csh -f
#
# A csh script to checkout Dakota (and external packages) to create source
# tar file(s) for internal or external distribution.
#
# --- external arguments/variables used ---
# $1 version or other tag for source tarball(s) created (e.g. votd, 4_0)
# $2 SVN revision of DAKOTA to checkout (trunk, tags/Version_4_1)
# $3 optional: distribution tarballs to build (default external only)
#    1: ext            (omits PSUADE, NLPQL, DOT, NPSOL) [base]
#    2: int_web        (omits PSUADE, NLPQL, DOT)        [base + NPSOL]
#    4: int_supervised (omits PSUADE, NLPQL)             [base + NPSOL + DOT]
#    8: int_full       (entire respository) 
#    This is bit-wise: add any of these to create multiple tarballs
#
# $TEST_SQA   script directory (passed to top-level scripts)
# $DEV_SVNROOT (needed by checkout_script.csh)
# -----------------------------------------

# -------------------------------
# required command-line arguments
# -------------------------------
if ( $#argv < 2 ) then
  echo "Usage:   extract_src.csh tar_tag dakota_svn_ver distro"
  echo "         distro is 1 (ext), 2 (int_web), 4 (int_supervised),"
  echo "                   8 (int_full), or any sum of these." 
  echo "Example: extract_src.csh votd trunk (external VOTD)"
  echo "Example: extract_src.csh 4_1 tags/Version_4_1 4 (SCICO 4_1)"
  exit
endif
set tar_tag = $1     
set dakota_ver = $2

# ---------------------------------
# command-line option: distribution
# ---------------------------------
if ( $#argv > 2 ) then
  set distro = $3 
  if ( $distro < 1 || $distro > 15 ) then
    echo "Invalid distro option $distro specified. Exiting."
    exit
  endif
else 
  echo "Using default: external distribution only."
  set distro = 1
endif

echo "Running $0 on `uname -n` at `date`"
echo "*** Creating Dakota source tarball(s) ***"

# ----------------------
# perform clean checkout
# ----------------------
set base = $cwd
$TEST_SQA/checkout_script.csh $base $dakota_ver export

# -------
# cleanup
# -------
cd Dakota
# remove all autotools-generated directories
find . -name autom4te.cache | xargs rm -rf
# remove other unneeded directories
\rm -rf packages/tnt docs/sqa test/sqa/v*
#\rm -rf methods/DOT/dbl_prec/*
cd ..

# --------------------
# create src tar files
# --------------------

# handle full internal case first: entire Dakota checkout
if ( $distro & 8 ) then
  tar cvf Dakota_${tar_tag}_int_full.src.tar Dakota
  echo "*** Dakota_${tar_tag}_int_full.src.tar for developer use complete."
endif

# if any of ext (1), int_web (2), int_supervised (4) needed,
# first add base components to Dakota_tmp.src.tar
if ( $distro & 7 ) then

  # always use explict tar commands to tar only the files wanted
  # NOTE: for excluded packages that can be configured, empty directories are
  # better than no directories, since configuring for a package that does not
  # exist can cause make problems.

  tar cvf Dakota_tmp.src.tar Dakota/INSTALL
  tar rvf Dakota_tmp.src.tar Dakota/LICENSE
  tar rvf Dakota_tmp.src.tar Dakota/COPYRIGHT
  tar rvf Dakota_tmp.src.tar Dakota/README
  tar rvf Dakota_tmp.src.tar Dakota/Makefile.*
  tar rvf Dakota_tmp.src.tar Dakota/examples/Makefile.*
  tar rvf Dakota_tmp.src.tar Dakota/examples/README
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Makefile.*
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Abaqus
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Cask/filters
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Nastran/cylinder
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Qtran/scripts
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Salinas/aff_all_fns.C
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Salinas/dakota_aff_lhs.in
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Salinas/salinas_script_asv
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/Rsmas
  tar rvf Dakota_tmp.src.tar Dakota/examples/script_interfaces/generic
  tar rvf Dakota_tmp.src.tar Dakota/examples/linked_interfaces
  tar rvf Dakota_tmp.src.tar Dakota/examples/methods
  # TODO: add examples and more applications to internal distros
  tar rvf Dakota_tmp.src.tar Dakota/examples/parallelism
  tar rvf Dakota_tmp.src.tar Dakota/examples/platforms
  tar rvf Dakota_tmp.src.tar Dakota/examples/tutorial
  tar rvf Dakota_tmp.src.tar Dakota/packages
  tar rvf Dakota_tmp.src.tar Dakota/methods/Makefile.am
  tar rvf Dakota_tmp.src.tar Dakota/methods/Makefile.in
  tar rvf Dakota_tmp.src.tar Dakota/methods/acro
  tar rvf Dakota_tmp.src.tar Dakota/methods/CONMIN
  tar rvf Dakota_tmp.src.tar Dakota/methods/DDACE
  tar rvf Dakota_tmp.src.tar Dakota/methods/Evidence
  tar rvf Dakota_tmp.src.tar Dakota/methods/FSUDace
  tar rvf Dakota_tmp.src.tar Dakota/methods/JEGA
  tar rvf Dakota_tmp.src.tar Dakota/methods/LHS
  tar rvf Dakota_tmp.src.tar Dakota/methods/NLPQL/Makefile.*
  tar rvf Dakota_tmp.src.tar Dakota/methods/NLPQL/aclocal.m4
  tar rvf Dakota_tmp.src.tar Dakota/methods/NLPQL/config*
  tar rvf Dakota_tmp.src.tar Dakota/methods/NLPQL/install-sh
  tar rvf Dakota_tmp.src.tar Dakota/methods/NLPQL/missing
  tar rvf Dakota_tmp.src.tar Dakota/methods/NCSUOpt
  tar rvf Dakota_tmp.src.tar Dakota/methods/NL2SOL
  tar rvf Dakota_tmp.src.tar Dakota/methods/OPTPP
  tar rvf Dakota_tmp.src.tar Dakota/methods/PSUADE/Makefile.*
  tar rvf Dakota_tmp.src.tar Dakota/methods/PSUADE/aclocal.m4
  tar rvf Dakota_tmp.src.tar Dakota/methods/PSUADE/config*
  tar rvf Dakota_tmp.src.tar Dakota/methods/PSUADE/psuade_config.h.in
  tar rvf Dakota_tmp.src.tar Dakota/methods/PSUADE/ltmain.sh
  tar rvf Dakota_tmp.src.tar Dakota/platform
  tar rvf Dakota_tmp.src.tar Dakota/aclocal.m4
  tar rvf Dakota_tmp.src.tar Dakota/config.guess
  tar rvf Dakota_tmp.src.tar Dakota/config.sub
  tar rvf Dakota_tmp.src.tar Dakota/configure
  tar rvf Dakota_tmp.src.tar Dakota/configure.ac
  tar rvf Dakota_tmp.src.tar Dakota/dakota_config.h.in
  tar rvf Dakota_tmp.src.tar Dakota/depcomp
  tar rvf Dakota_tmp.src.tar Dakota/install-sh
  tar rvf Dakota_tmp.src.tar Dakota/ltmain.sh
  tar rvf Dakota_tmp.src.tar Dakota/m4/*
  tar rvf Dakota_tmp.src.tar Dakota/missing
  tar rvf Dakota_tmp.src.tar Dakota/docs
  tar rvf Dakota_tmp.src.tar Dakota/src
  tar rvf Dakota_tmp.src.tar Dakota/test

endif
# save this tarball if needed
if ( $distro & 1 ) then
  # if creating further tarballs, copy
  if ( $distro & 6 ) then
    cp Dakota_tmp.src.tar Dakota_${tar_tag}_ext_web.src.tar
  else
    mv Dakota_tmp.src.tar Dakota_${tar_tag}_ext_web.src.tar
  endif

  # add NPSOL stubs to finished tar
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/NPSOL/aclocal.m4
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/NPSOL/config*
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/NPSOL/install-sh
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/NPSOL/Makefile.*
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/NPSOL/m4

  # add DOT stubs to finished tar
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/aclocal.m4
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/config*
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/install-sh
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/missing
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/Makefile.*
  tar rvf Dakota_${tar_tag}_ext_web.src.tar Dakota/methods/DOT/dbl_prec/Makefile.*

  echo "*** Dakota_${tar_tag}_ext_web.src.tar for external web complete."

endif

# if either of int_web (2) or int_supervised (4) needed, add NPSOL
if ( $distro & 6 ) then
  tar rvf Dakota_tmp.src.tar Dakota/methods/NPSOL
endif
# save this tarball if needed
if ( $distro & 2 ) then
  # if creating further tarballs, copy
  if ( $distro & 4 ) then
    cp Dakota_tmp.src.tar Dakota_${tar_tag}_int_web.src.tar
  else
    mv Dakota_tmp.src.tar Dakota_${tar_tag}_int_web.src.tar
  endif

  # add DOT stubs to finished tar
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/aclocal.m4
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/config*
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/install-sh
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/missing
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/Makefile.*
  tar rvf Dakota_${tar_tag}_int_web.src.tar Dakota/methods/DOT/dbl_prec/Makefile.*

  echo "*** Dakota_${tar_tag}_int_web.src.tar for internal web complete."

endif

# if int_supervised (4) needed, add DOT
if ( $distro & 4 ) then
  tar rvf Dakota_tmp.src.tar Dakota/methods/DOT
  mv Dakota_tmp.src.tar Dakota_${tar_tag}_int_supervised.src.tar

  echo "*** Dakota_${tar_tag}_int_supervised.src.tar for internal supervised complete."

endif

if ( $distro & 7 ) then
  # Clean up this Dakota area, since the tar file is now the master
  # (and the Dakota working directory and tar file will differ)
  \rm -rf Dakota
endif

echo "*** $0 finished ***"
