#
# A python module of classes for reading in experimental study files
# and experimental results files.
#
# Note: experimental results are associated with the Experiment object.
#

import re
import math
import os
import types
import sys
from xml.dom import minidom, Node
from sets import Set

#
# BUGS/ISSUES
#
# 1. These classes parse an XML file.  Should we test the overall syntax
# 	of the file first?  I think there are XML files to do that if we
#	have an XML style sheet.
#
# 2. These class extract just a subset of the XML information that might
#	be in a file.  If 'extra' information is provided, is that an error?
#	Should we capture that 'extra' information in a standard manner?
#	If so, we should probably have these classes inherit from a base
#	class that defines generic XML information (attributes, text, etc)
#	
# 3. The semantics of how this class differs from the Pickled XML used in
#	EXACT may be unclear to the user.  In both cases, we refer to how
#	these classes are 'loaded' from XML.
#
# 4. I made a design decision to try to make these classes naturally follow
#	from the XML specification used with 'runexp'.  Thus, we need
#	to reconcile how these scripts relate to the EXACT EXIST database.
#
# 5. Should we rename 'analyze' to 'analysis'?  That seems a better naming
#	convention.  But perhaps the question is whether the 'analyze' block
#	contains multiple tests, or whether we should have a separate
#	'analysis' block for each test/analysis that we want to do... Hmmm.
#
# 6. Make I/O controlable at a global level, to enable a 'quiet' mode.
#
# 7. The XML specification of a 'results' file should be changed to make it
#	more specific to the results for a _single_ experiment.
#
# 8. Do we need to have the experimental trial store the type of the 
#    data elements?
#
# 9. When an experiment is run, the *results.xml file is generated based on
#	the name of the experiment file.  This is an error, as it should
#	be generated based on the experimental study name (which could default
#	to the prefix of the experimental study filename).  This 
#	discrepancy confuses this Python code, which does not necessarily
#	know about the XML that was used to generate the XML parse tree.
#
# 10. What convention should we use for function/method names?  Should we
#	use
#		level_names()
#	or
#		levelNames()
#	??
#
# 11. How are we going to organize these objects and functions into the exact
#	package?  (More fundamentally, is EXACT a package or module?)
#	Similarly, are these examples of how EXACT works, or are we going to
#	rework EXACT to make these the core experimental objects?
#
# 12. What formating style should be used with Python?  I've put two spaces
#	between method declarations in classes, because one didn't seem like enough...
#
# 13. What python statistics package should we integrate for experimental testing?
#	I think that simple summary statistics are available in some packages, but for
#	more powerful tests we need R.
#
# 14. Process warning information from config.xml and build.xml files.
#
# 15. Consider treating the scenario key as a tuple, which can be hashed.
#

def get_text(node):
  nodetext = ""
  for child in node.childNodes:
    if child.nodeType == Node.TEXT_NODE:
       nodetext = nodetext + child.nodeValue
  nodetext.strip()
  return nodetext


def median(mylist):
  mylist.sort()
  if (len(mylist) == 0):
     return -1
  elif (len(mylist) == 1):
     return mylist[0]
  elif (divmod(len(mylist),2)[1] == 1):
     return mylist[(len(mylist)-1)/2]
  ndx = len(mylist)/2
  return (mylist[ndx-1]+mylist[ndx])/2


#
# This class illustrates how we can define objects to perform
# analysis.  The 'validation' analysis is the simplest form.  It verifies
# that the specified measurement has a given value (within a specified
# tolerance).
#
class ValidationAnalysis(object):
    def __init__(self):
	"""
	An object that can analyze an experiment to validate one
	or more values in the experiment.
	"""
	self.measurement=None
	self.value=None
	self.tolerance=None
	self.results={}
	#
	# sense = 1  if minimizing
	# sense = -1 if maximizing
	# sense = 0  if looking for an absolute tolerance level
	#
	self.sense=1.0

    def analyze(self, results):
	"""
	Analyze an experiment.
	"""
	for combination in results:
	  status = "Pass"
	  for trial in combination:
	    if not self.validate(trial[self.measurement]):
	       status = "Fail"
	       break
	  if status == "Pass":
	     sys.stdout.write(".")
	  else:
	     sys.stdout.write("F")
	  self.results[combination.name] = status

    def validate(self, tvalue):
	if (tvalue[1] == "ERROR") or\
	   (tvalue[1] == ""):
	   return False
	if (tvalue[0] == "numeric/double"):
	   if (self.sense == 0) and\
	      (math.abs(tvalue[1]-self.value) > self.tolerance):
	      return False
	   if (self.sense != 0) and\
	      ((self.sense*(tvalue[1]-self.value)) > self.tolerance):
	      return False
	if ((tvalue[0] == "numeric/boolean") or (tvalue[0] == "")) and\
	   (tvalue[1] != self.value):
	   return False
	return True



#
# This class illustrates how we can analyze an ExperimentalResults object to 
# compare a slide of results against another.  The final statistics
# are the relative ratio of performance of one slice against the other slices.
#
# The goal of this example is to exercise the interface to ExperimentalResults to
# ensure that results can be flexibly analyzed.
#
class RelativePerformanceAnalysis(object):
    def __init__(self):
	"""
	An object that can analyze an experiment to compare
	results with one or more sets of 'reference' experiments.  In this
	case, we assume that all computations are performance within
	the same experiment.
	Note: this comparison is only done w.r.t. the first trial for 
	each treatment combination.
	"""
	self.target_factor=None
	self.target_level=None
	self.measurement=None
	self.total={}
	self.errors={}
	self.ranks={}

    def analyze(self, results):
	"""
	Analyze an experiment.
	"""
	target_levels = results.levelNames(self.target_factor)
	self.total = {}
	self.errors = {}
	self.ranks = {}
	for level in target_levels:
	  self.total[level] = []
	  self.errors[level] = []
	  self.ranks[level] = []
	#
	# Get subsets of treatment combinations that are orthogonal
	# to the target_factor
	#
	subsets = results.combinationSubsets([self.target_factor])
	for group in subsets:
	  #print "GROUP " + `group`
	  #
	  # Find the index of the target factor
	  #
	  i = 0
	  while (group[i] != self.target_factor) and (i < len(group)):
	    i = i + 2
	  if (i >= len(group)):
	     print "Bad target_factor: " + self.target_factor
	     sys.exit(1)
	  i = i + 1
	  group[i] = self.target_level
          target_combination = results[group]
	  #
	  # Compare combination measurements
	  #
	  # Note: this loops through the data a few more times than is strictly
	  # necessary, but my goal was to keep this code simple.
	  #
	  measurements = []
	  status = []
	  ndx = {}
	  j = 0
	  for level in target_levels:
	    group[i] = level
	    measurements = measurements + [results[group].replication(0)[self.measurement][1]]
	    status = status + [results[group].replication(0)["exit_status"][1]]
	    ndx[level] = j
	    j = j + 1
	  #
	  # This is an O(n^2) algorithm, but we _could_ do this in O(nlogn)
	  #
	  rank = []
	  for level in target_levels:
	    ctr1 = 0
	    ctr2 = 0
	    for level_i in target_levels:
	      if (level == level_i):
		 continue
	      if (status[ndx[level]] == 0 and (status[ndx[level_i]] == 0 and measurements[ndx[level]] > measurements[ndx[level_i]])) or\
		 (status[ndx[level]] == 1 and status[ndx[level_i]] == 0):
		 ctr1 = ctr1 + 1
	      if (status[ndx[level]] == 0 and status[ndx[level_i]] == 0 and measurements[ndx[level]] == measurements[ndx[level_i]]) or\
		 (status[ndx[level]] == 1 and status[ndx[level_i]] == 1):
		 ctr2 = ctr2 + 1
	    rank = rank + [ctr1 + ctr2/2.0]
	  #
	  # Compute totals ... which are performance ratios relative to the target
	  #
	  for level in target_levels:
	    if level == self.target_level:
	       if status[ndx[level]] == 0:
                  self.total[level] = self.total[level] + [1.0]
	    elif status[ndx[level]] == 0 and status[ndx[self.target_level]] == 0:
               self.total[level] = self.total[level] + [measurements[ndx[level]]/measurements[ndx[self.target_level]]]
	    self.ranks[level] = self.ranks[level] + [rank[ndx[level]]]
	    if status[ndx[level]] == 1:
	       self.errors[level] = self.errors[level] + [1]
	    #print level + " " + `rank[ndx[level]]` + " " + `measurements[ndx[level]]` + " " + `status[ndx[level]]`
	print ""
	print "  SOLVER                     MEAN     MEDIAN"
	for level in target_levels:
	  if level == self.target_level:
	       print "  %-20s %10.3f %10.3f" % (level,1.0,1.0)
	  else:
	       print "  %-20s %10.3f %10.3f" % (level,sum(self.total[level])/len(subsets),median(self.total[level]))
	       #tmp = self.total[level]
	       #tmp.sort()
	       #print "  " + `tmp`
	print ""
	print "  SOLVER                  %Errors   Avg Rank"
	for level in target_levels:
	  print "  %-20s %10.3f %10.3f" % (level,100.0*sum(self.errors[level])/len(subsets),sum(self.ranks[level])/len(self.ranks[level]))
	  #tmp = self.ranks[level]
	  #tmp.sort()
	  #print "  " + `tmp`
	print ""




class XMLObjectBase(object):
    def reset():
	self.path = ""
	self.filename = ""


    def initialize(self, node, performReset=True):
	"""
	Initialize the object with either a filename or a node of an 
	XML parsed tree.
	"""
	if performReset:
	   self.reset()
	
	if isinstance(node,types.StringType) or isinstance(node,types.UnicodeType):
	   #
	   # Load in from a file
	   #
	   self.filename = node
	   if not os.path.exists(node):
              raise IOError, "ERROR: missing file " + node
	   self.path = os.path.dirname(node)
	   if self.path == "":
	      self.path = "./"
	   else:
	      self.path = self.path + "/"
	   #
	   # Parse XML
	   #
           doc = minidom.parse(node)
	   #
	   # Recursively call initialization, setting the current working
	   # directory to the directory where the config-info file is
	   # located.
	   #
	   try:
	      currdir = os.path.abspath('')
	      os.chdir(self.path)
	      self.parse_xml(doc.documentElement)
	   except:
	      os.chdir(currdir)
	      raise
	   os.chdir(currdir)
	else:
	   self.parse_xml(node)



class ExperimentalTrial(object):
    def __init__(self, node=None):
	"""
	An object that contains experimental results for a single 
	trial
	"""
	self.reset()
	if node:
	   self.initialize(node)


    def reset(self):
	self.id = -1
	self.seed = -1
	self.value = {}


    def initialize(self, node):
	self.reset()
	for (name,value) in node.attributes.items():
	  if name == "id":
	     self.id = value
	  elif name == "seed":
	     self.seed = value
	for child in node.childNodes:
	  if child.nodeType == Node.ELEMENT_NODE and child.nodeName == "Value":
	     vname = "unknown"
	     vtype = "unknown"
	     for (name,value) in child.attributes.items():
	       if name == "name":
	          vname = value
	       elif name == "type":
	          vtype = value
	     vtext = get_text(child)
	     if (vtype == "numeric/double" or vtype == "numeric/integer" or\
		 vtype == "numeric/boolean") and (vtext != "ERROR"):
	        self.value[vname] = (vtype, eval(vtext))
	     else:
	        self.value[vname] = (vtype, vtext)


    def pprint(self,prefix=""):
	print prefix + "id=" + self.id + " seed=" + self.seed,
	vkeys = self.value.keys()
	vkeys.sort()
	for vname in vkeys:
	  foo = vname + "="
	  tmp = self.value[vname][1]
          if isinstance(tmp,types.StringType) or isinstance(tmp,types.UnicodeType):
	     foo = foo + "\"" + self.value[vname][1].strip() + "\""
	  else:
	     foo = foo + `tmp`
	  print foo,
        print ""


    def __getitem__(self,index):
	try:
	   if index == "seed":
	      return self.seed
	   elif index == "id":
	      return self.id
	   return self.value[index]
	except KeyError, info:
	   print "Bad key for trial measurement"
	   print "Valid keys are " + `self.value.keys()`
	   sys.exit(0)




class TreatmentCombination(object):
    def __init__(self, node=None):
	"""
	An object that contains experimental results for a single 
	treatment combination (i.e. combination of factor levels)
	"""
	self.reset()
	if node:
	   self.initialize(node)


    def reset(self):
	self.name = "unknown"
	self.start_time = ""
	self.run_time = -1.0
	self.expparams = ""
	self.level_name = {}  # factor_name -> level_name
	self.level_text = {}  # factor_name -> level_text
	self.replicationlist = []
	self.key = None
	self.status = "Fail"


    def levelName(self, factor_name):
	return self.level_name[factor_name]


    def levelText(self, factor_name):
	return self.level_text[factor_name]


    def initialize(self, node):
	"""
	Initialize a treatment condition with a node of an XML parsed tree.
	"""
	self.reset()
	for child in node.childNodes:
	  if child.nodeType == Node.ELEMENT_NODE:
	     if child.nodeName == "StartTime":
		self.start_time = get_text(child)

	     elif child.nodeName == "Name":
		self.name = get_text(child)

	     elif child.nodeName == "RunTime":
		self.run_time = get_text(child)

	     elif child.nodeName == "ExecutionStatus":
		self.status = get_text(child)

	     elif child.nodeName == "Description":
		self.expparams = get_text(child)
		for fnode in child.childNodes:
		  if fnode.nodeType == Node.ELEMENT_NODE and\
		     fnode.nodeName == "Factor":
		     #
		     # Setup a diagnostic text string
		     #
		     ftext = get_text(fnode)
		     self.expparams = self.expparams + " " + ftext
		     #
		     # Get the factor levels
		     #
		     fname = ""
		     flevel = -1
		     for (name,value) in fnode.attributes.items():
		       if name == "name":
			  fname = value
		       elif name == "level":
			  flevel = value
		     self.level_name[fname] = flevel
		     self.level_text[fname] = ftext

	     elif child.nodeName == "Trial":
		self.replicationlist = self.replicationlist + [ExperimentalTrial(child)]


    def numReplications(self):
	return len(self.replicationlist)


    def replication(self,num):
	return self.replicationlist[num]


    def __iter__(self):
	"""
	Returns the iterator to the replication list.  This method allows
	a TreatmentCombinations() object to be treated as the generator for the 
	iterator of the list of ExperimentalTrial().
	"""
	return self.replicationlist.__iter__()


    def pprint(self,prefix=""):
	print prefix+"Treatment Combination: " + self.name
	print prefix+"  Start Time:          " + self.start_time
	print prefix+"  Run Time:            " + self.run_time
	print prefix+"  Status:              " + self.status
	print prefix+"  Levels:"
	for name in self.level_name.keys():
	  print prefix+"        " + name + "\t" + self.level_name[name] + " : " + self.level_text[name]
	print prefix+"  Exp Params:          \"" + self.expparams.strip() + "\""
	print prefix+"  Trials:"
	for trial in self.replicationlist:
	  trial.pprint(prefix+"  ")




class ExperimentalResults(XMLObjectBase):
    def __init__(self, node=None):
        """
        An object that describes experimental results.
        """
	self.reset()
	if node:
	   self.initialize(node)


    def reset(self):
	self.combinations = []
	self.comb_dict = {}    # flattened_comb -> index

    def parse_xml(self, node):
	"""
	Initialize the experimental results with a node of an XML parsed tree.
	"""
	self.reset()
	for child in node.childNodes:
	  if child.nodeType == Node.ELEMENT_NODE and child.nodeName == "Experiment":
	     tc = TreatmentCombination(child)
	     self.combinations = self.combinations + [tc]
	     self.comb_dict[self.flatten_dict(tc.level_name)] = len(self.combinations)-1
	     

    def __iter__(self):
	"""
	Returns the iterator to the combination list.  This method allows
	a ExperimentalResults() object to be treated as the generator for the 
	iterator of the list of ExperimentalCombination().
	"""
	return self.combinations.__iter__()


    def __getitem__(self,index):
	"""
	Given a list or tuple of factor_name/level_name pairs, return the
	corresponding combination
	"""
	if isinstance(index,list):
	   return self.combinations[self.comb_dict[tuple(index)]]
	return self.combinations[self.comb_dict[index]]


    def pprint(self,prefix=""):
	for combination in self.combinations:
	  combination.pprint(prefix)


    def levelNames(self, factor_name):
	tmp = {}
	for combination in self.combinations:
	  tmp[combination.levelName(factor_name)] = 1;
	#
	# Return keys in sorted order to make things pretty
	#
	keys = tmp.keys()
	keys.sort()
	return keys


    def combinationSubsets(self, factor_list):
	ans = []
	subsets=Set()
	factor_list.sort()
	i = 0
	for combination in self.combinations:
	  tmp = combination.level_name
	  for factor in factor_list:
	    tmp[factor] = '*'
	  subset_tuple = self.flatten_dict(tmp)
	  if subset_tuple not in subsets:
	     subsets = subsets.union(Set([subset_tuple]))
	     ans = ans + [list(subset_tuple)]
	  i = i + 1
	return ans
	  

    def flatten_dict(self,mydict):
	"""
	A function that takes a dictionary and flattens it into a tuple in
	a uniform manner: keys are sorted, and the tuple consists of 
	key-value pairs in order.
	Note: this method could be treated as a function, but at the
	moment it is only used within this class.
	"""
	keys = mydict.keys()
	keys.sort()
	ans = ()
	for key in keys:
	  ans += (key,)
	  ans += (mydict[key],)
	return ans




class FactorLevel(object):
    def __init__(self, node=None, number=None):
        """
        An object describes a factor level for an experiment.  This is
	simply a list of FactorLevel objects
        """
	self.reset()
	if node:
	   self.initialize(node,number)

    def reset(self):
	self.attr = {}
	self.name = "unknown"
	self.text = ""
	self.number = -1


    def initialize(self, node, number):
	"""
	Initialize the factor level with a node of an XML parsed tree.
	"""
	self.reset()
	self.number = number
	if isinstance(node,types.StringType) or isinstance(node,types.UnicodeType):
	   self.text = node
	   self.name = "level_" + `number`
	   #
	   # Look for the a token of the form "_level_name=" in the level definition.
	   # If it exists, use it to define the level name.
	   #
	   for token in re.split('[ \t]+',self.text):
	     if token[0:12] == "_level_name=":
	        self.name = token[12:len(token)]
	else:
	  for (name,value) in node.attributes.items():
	    self.attr[name] = value
	  if self.attr.has_key("name"):
	     self.name = self.attr["name"]
	  if self.name == "unknown":
	     self.name = "level_" + `number`
	  for cnode in node.childNodes:
            if cnode.nodeType == Node.TEXT_NODE:
               self.text = self.text + cnode.nodeValue
          self.text.strip()



class Factor(object):
    def __init__(self, node=None, number=None):
        """
        An object describes a factor for an experiment.  This is
	simply a list of FactorLevel objects
        """
	self.reset()
	if node and number:
	   self.initialize(node,number)


    def reset(self):
	self.levels = []
	self.attr = {}
	self.name = "unknown"
	self.number = -1


    def initialize(self, node, number):
	"""
	Initialize the factor with a node of an XML parsed tree.
	"""
	self.reset()
	self.number = number
	for (name,value) in node.attributes.items():
	  self.attr[name] = value
	if self.attr.has_key("name"):
	   self.name = self.attr["name"]
	if self.name == "unknown":
	   self.name = "factor_" + `number`
	if self.attr.has_key("filename"):
	   #
	   # Read factor levels from a file
	   #
	   #self.read_factor_levels(self.attr["filename"])
  	   INPUT = open(self.attr["filename"])
	   self.i = 1
  	   for line in INPUT.xreadlines():
    	     if line[0] == "#":
       	       continue
    	     self.levels = self.levels + [FactorLevel(line.strip(),self.i)] 
	     self.i = self.i + 1
  	   INPUT.close()
	else:
	   #
	   # Initialize factor levels from nodes of an XML parsed tree.
	   #
	   self.i = 1
  	   for cnode in node.childNodes:
    	     if cnode.nodeType == Node.ELEMENT_NODE:
       	        self.levels = self.levels + [ FactorLevel(cnode,self.i) ] 
	        self.i = self.i + 1


    def pprint(self,prefix=""):
	print prefix+"Factor " + self.name + " (" + `self.number` + ")"
	for level in self.levels:
	  print prefix+"  Level: ",
	  if level.name != "unknown":
	     print level.name
	  else:
	     print ""
	  print prefix+"    \"" + level.text + "\""



class Factors(object):
    def __init__(self, node=None):
        """
        An object that describes the factors for an experiment.  This is
	simply a list of Factor objects
        """
        self.debug = False
	self.reset()
	if node:
	   self.initialize(node)


    def reset(self):
	self.factorlist = []


    def initialize(self, node):
	"""
	Initialize the factor list with a node of an XML parsed tree.
	"""
	self.reset()
  	for cnode in node.childNodes:
    	  if cnode.nodeType == Node.ELEMENT_NODE:
       	     self.factorlist = self.factorlist + [ Factor(cnode,len(self.factorlist)+1) ] 


    def __iter__(self):
	"""
	Returns the iterator to the factor list.  This method allows
	a Factors() object to be treated as the generator for the 
	iterator of the list of Factors().
	"""
	return self.factorlist.__iter__()




class Controls(object):
    def __init__(self, node=None):
        """
        An object that describes the controls for an experiment.
        """
	self.reset()
	if node:
	   self.initialize(node)


    def reset(self):
	self.seeds = []


    def initialize(self, node):
	"""
	Initialize the experimental controls with a node of an XML parsed tree.
	"""
	self.reset()
	#
	# TODO: more stuff here later
	#



class Experiment(object):
    def __init__(self, node=None, expnum=None, expname=None):
	"""
	An experiment object, which contains the factors for an
	experiment as well as the experimental results.
	"""
	if node and expnum:
	   self.initialize(node,expnum)
	if expname:
	   try:
	      self.readResults(expname)
	   except IOError:
	      #
	      # TODO: resolve error handling semantics.
	      #
	      # The default behavior is to try to read in an
	      # experimental results file.  It shouldn't be an error
	      # if this file doesn't exist, but we should be more careful
	      # with our error management here.  The file could have existed,
	      # but we had an error parsing/processing it!
	      #
	      pass


    def reset(self):
	self.name = "unknown"
	self.factors = None
	self.controls = None
	self.results = None


    def initialize(self,node,expnum):
	"""
	Initialize the experiment with a node of an XML parsed tree.
	"""
	self.reset()
	#
	# Setup experiment name
	#
	for (name,value) in node.attributes.items():
	  if name == "name":
	     self.name = value
	if self.name == "unknown":
	   self.name = "exp" + `expnum`
	#
	# Process factors and controls
	#
  	for cnode in node.childNodes:
    	  if cnode.nodeType == Node.ELEMENT_NODE:
	     if cnode.nodeName == "factors":
       	        self.factors = Factors(cnode)
	     elif cnode.nodeName == "controls":
		self.controls = Controls(cnode)


    def readResults(self, studyname):
	fname = studyname + "." + self.name + ".results.xml"
	if not os.path.exists(fname):
	   raise IOError, "Unable to load file " + fname
	print "  ... loading results file " + fname
        doc = minidom.parse(fname)
	self.results = ExperimentalResults(doc.documentElement)
	if self.results == None:
	   print "FOO"


    def pprint(self,prefix=""):
	print prefix+"Factors: " + `len(self.factors.factorlist)`
	for factor in self.factors:
	  factor.pprint(prefix+"  ")
	print prefix+"Results:"
	if self.results:
	   self.results.pprint(prefix+"  ")
	else:
	   print prefix+"  None"
	
	


class Analysis(object):
    def __init__(self, node=None):
	"""
	An experimental analysis object.
	"""
	if node:
	   self.initialize(node)


    def reset(self):
	self.name = "unknown"


    def initialize(self, node):
	"""
	Initialize the analysis with a node of an XML parsed tree.
	"""
	self.reset()
	for (name,value) in node.attributes.items():
	  if name == "name":
	     self.name = value
	#
	# TODO: more here later
	#


    def pprint(self,prefix=""):
	print prefix+"TODO"
		


class ExperimentalStudy(XMLObjectBase):
    def __init__(self, node=None):
	"""
An experimental study object, which contains one or more 
experiments and experimental analyses.

Note: this object could be initialize with a node from an XML tree,
or with a filename, or with no argument at all

Note: when an experimental study is created, results are loaded 
if they are found.  Otherwise, an explicit 'readResults()' needs to
be called when results are generated after this object is constructed.
"""
	if node:
	   self.initialize(node)


    def reset(self):
	self.name = "study"
	self.experiments = {}
	self.analyses = {}
	self.path = "./"


    def parse_xml(self, node):
	"""
	Initialize the study with a node of an XML parsed tree.
	"""
	#
	# Load in from an XML node
	#
	for (name,value) in node.attributes.items():
	  if name == "name":
	     self.name = value
  	for cnode in node.childNodes:
    	  if cnode.nodeType == Node.ELEMENT_NODE:
	     if cnode.nodeName == "experiment":
 		exp = Experiment(cnode, len(self.experiments)+1, self.name)
       	        self.experiments[exp.name] = exp
	     elif cnode.nodeName == "analyze":
		analysis = Analysis(cnode)
		self.analyses[analysis.name] = analysis


    def readResults(self):
	"""
	Read results from an XML file.
	"""
	for exp in experiments:
	  try: 
	     exp.readResults(self.path,self.name)
	  except IOError:
	     dummy=1 


    def pprint(self,prefix=""):
	"""
	Print out the information in the experimental study.
	"""
	print prefix+"Experimental Study: " + self.name
	print prefix+"  Experiments: " + `len(self.experiments)`
	print prefix+"  Analyses:    " + `len(self.analyses)`
	print ""
	expkeys = self.experiments.keys()
	expkeys.sort()
	for name in expkeys:
	  print prefix+"Experiment: " + name
	  self.experiments[name].pprint(prefix+"  ")
	  print ""
	for (name,analysis) in self.analyses.iteritems():
	  print prefix+"Analysis:   " + name
	  analysis.pprint(prefix+"  ")


class ScenarioKey(object):
    def __init__(self, node=None):
	"""
	Host information
	"""
	self.xml_attr = []
	if node:
	   self.initialize(node)


    def initialize(self, node):
	for (name,value) in node.attributes.items():
	  self.__dict__["xml_" + name] = value
	  self.xml_attr = self.xml_attr + ["xml_" + name]


    def pprint(self,prefix=""):
	for attr in self.xml_attr:
	  print prefix+"  %-20s %s" % (attr[4:],"\"" + self.__dict__[attr] + "\"")



class SoftwareInfo(XMLObjectBase):
    def __init__(self, node=None):
	"""
	Read in information from an XML configuration summary.
	"""
	if node:
	   self.initialize(node)


    def reset(self):
	self.path = ""
	self.filename = ""
	self.key = None
	self.flags = ""
	self.log_file = ""
	self.start_time = ""
	self.run_time = ""
	self.execution_status = "Fail"
	self.integrity_status = "Fail"
	self.warnings = []


    def parse_xml(self, node):
	"""
	Initialize the config info with a node of an XML parsed tree.
	"""
	#
	# Load in from an XML node
	#
  	for cnode in node.childNodes:
    	  if cnode.nodeType == Node.ELEMENT_NODE:
	     if cnode.nodeName == "Key":
 		self.key = ScenarioKey(cnode)
	     elif cnode.nodeName == "Flags":
		self.config_flags = get_text(cnode)
	     elif cnode.nodeName == "LogFile":
		self.log_file = get_text(cnode)
	     elif cnode.nodeName == "StartTime":
		self.start_time = get_text(cnode)
	     elif cnode.nodeName == "RunTime":
		self.run_time = get_text(cnode)
	     elif cnode.nodeName == "ExecutionStatus":
		self.execution_status = get_text(cnode)
	     elif cnode.nodeName == "IntegrityStatus":
		self.integrity_status = get_text(cnode)
	     elif cnode.nodeName == "Warnings":
		for gchild in cnode.childNodes:
		  if gchild.nodeType == Node.ELEMENT_NODE and\
	             gchild.nodeName == "Explanation":
		     for (name,value) in gchild.attributes.items():
		       if name == "line":
			  self.warnings = self.warnings + [eval(value)]

	
    def pprint(self,prefix=""):
	print prefix+"Software Info: " + self.filename
	print prefix+"  Flags:           " + self.flags
	print prefix+"  LogFile:         " + self.log_file
	print prefix+"  StartTime:       " + self.start_time
	print prefix+"  RunTime:         " + self.run_time
	print prefix+"  ExecutionStatus: " + self.execution_status
	print prefix+"  IntegrityStatus: " + self.integrity_status
	print prefix+"  Warnings:        " + `len(self.warnings)`
	print prefix+"  Key:"
        if self.key:
	   self.key.pprint(prefix+"  ")



class Scenario(XMLObjectBase):
    def __init__(self, node=None):
	"""
	Read in information from an XML EXACT scenario summary.
	"""
	if node:
	   self.initialize(node)


    def reset(self):
	self.path = ""
	self.filename = None
	self.key = None
	self.description = ""
	self.start_time = ""
	self.end_time = ""
	self.run_time = ""
	self.config_info = None
	self.build_info = None
	self.results = {}


    def parse_xml(self, node):
	"""
	Initialize the scenario with a node of an XML parsed tree.
	"""
	#
	# Load in from an XML node
	#
  	for cnode in node.childNodes:
    	  if cnode.nodeType == Node.ELEMENT_NODE:
	     if cnode.nodeName == "Key":
 		self.key = ScenarioKey(cnode)
	     elif cnode.nodeName == "Description":
		self.description = get_text(cnode)
	     elif cnode.nodeName == "StartTime":
		self.start_time = get_text(cnode)
	     elif cnode.nodeName == "EndTime":
		self.end_time = get_text(cnode)
	     elif cnode.nodeName == "RunTime":
		self.run_time = get_text(cnode)
	     elif cnode.nodeName == "Files":
		for gchild in cnode.childNodes:
		  ctr=1
		  if gchild.nodeType == Node.ELEMENT_NODE and\
	        	gchild.nodeName == "Name":
		        filename = get_text(gchild)
			if filename[-9:] == "build.xml":
			   self.build_info = SoftwareInfo(filename)
			elif filename[-10:] == "config.xml":
			   self.config_info = SoftwareInfo(filename)
			elif filename[-11:] == "results.xml":
			   print "TODO " + filename
			   self.results["results_" + `ctr`] = ExperimentalResults(filename)

	
    def pprint(self,prefix=""):
        print "Scenario:"
	if self.config_info:
	   self.config_info.pprint(prefix+"  ")
	if self.build_info:
	   self.build_info.pprint(prefix+"  ")
	for key in self.results.keys():
	  print "  Experimental Results: " + key
	  self.results[key].pprint(prefix+"    ")



###
### Testing stuff
###
### Ignore IOError exceptions, but other exceptions cause fatal faults
###

try:
  #
  # Load the UTILIB experimental results and print out the information
  #
  study = ExperimentalStudy("../../../utilib/test/utilib.exp.xml")
  study.pprint()
  #
  # Perform an analysis on utilib
  #
  analysis = ValidationAnalysis()
  analysis.measurement = "diffs"
  analysis.value = 0
  if study.experiments["exp1"].results:
     analysis.analyze(study.experiments["exp1"].results)
     print ""
except IOError:
  pass

try:
  #
  # Load the PICO performance comparison results
  #
  study = ExperimentalStudy("../../../pico/test/milp-comparison.exp.xml")
  study.pprint()
  #
  # Perform a performance analysis of PICO
  #
  analysis = RelativePerformanceAnalysis()
  analysis.target_factor="solver"
  analysis.target_level="pico-nocuts"
  analysis.measurement="Seconds"
  if study.experiments["exp1"].results:
     analysis.analyze(study.experiments["exp1"].results)
except IOError, detail:
  print detail
  pass


try:
  #
  # Load the build and config information
  #
  configinfo = SoftwareInfo("../../../../../test/20061011#001504#wehart#tofu.sandia.gov#acro-gnlp-npsol-n#scenario.xml")
  configinfo.pprint()
  configinfo = SoftwareInfo("../../../../test/config.xml")
  configinfo.pprint()
  buildinfo = SoftwareInfo("../../../../test/build.xml")
  buildinfo.pprint()
except IOError, detail:
  print detail
  pass

try:
  #
  # Load a study
  #
  scenario = Scenario("../../../../../test/20061011#001504#wehart#tofu.sandia.gov#acro-gnlp-npsol-n#scenario.xml")
  scenario.pprint()
except IOError, detail:
  print detail
  pass

