/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblemAppWrapper.h
 *
 * Defines the colin::OptProblemAppWrapper class.
 */

#ifndef colin_OptProblemAppWrapper_h
#define colin_OptProblemAppWrapper_h

#include <acro_config.h>
#include <colin/OptProblemAppHandle.h>
#include <colin/OptProblemAppBody.h>

namespace colin {

using utilib::SmartPtr;
using utilib::CachedAllocator;

/**
 * The subclass of \c OptProblemAppHandle that Simply forwards function calls
 * to another \c OptProblemAppHandle class with a different problem 
 * representation.
 */
template <class DomainT, class ResponseT, class LDomainT, class LResponseT>
class OptProblemAppWrapper : public OptProblemAppHandle<DomainT,ResponseT>
{
public:

  /// Constructor
  OptProblemAppWrapper(
		SmartPtr< OptProblemAppHandle<LDomainT,LResponseT> >& handle_)
		: handle(handle_)
	{lpoint = &(handle->point);}

  ///
  void Eval(std::vector<int>& asv, int mode, ResponseT& response)
		{
		LResponseT* lresponse = new_lresponse(&response, false);
		handle->Eval(asv,mode,*lresponse);
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}

  ///
  void Eval(int mode, ResponseT& response)
		{
		state()->init_request_asv(mode);
		LResponseT* lresponse = new_lresponse(&response, false);
		handle->Eval(state()->REQUEST_ASV,mode,*lresponse);
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}

  ///
  void EvalF(real& value)
		{
		LResponseT* lresponse = LEval(state()->F_ASV,mode_f);
		value = lresponse->function_value();
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}

  ///
  void EvalF(typename ResponseT::realarray_t& values)
		{
		LResponseT* lresponse = LEval(state()->F_ASV,mode_f);
		lresponse->function_values(values);
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}

  ///
  void EvalCF(typename ResponseT::vector_t& value)
		{
		LResponseT* lresponse = LEval(state()->CF_ASV,mode_cf);
		value << lresponse->constraint_values();
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}

  ///
  void EvalG(typename ResponseT::vector_t& value)
		{
		LResponseT* lresponse = LEval(state()->G_ASV,mode_g);
		value << lresponse->function_gradient();
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}
 
  ///
  void EvalFG(real& ans, typename ResponseT::vector_t& value)
		{
		LResponseT* lresponse = LEval(state()->FG_ASV,mode_f|mode_g);
		ans =  lresponse->function_value();
		value << lresponse->function_gradient();
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}
 
  ///
  void EvalCG(typename ResponseT::vectorarray_t & value)
		{
		LResponseT* lresponse = LEval(state()->CG_ASV,mode_cg);
		map_jacobian(value,lresponse->constraint_gradients());
		CachedAllocator<LResponseT> :: deallocate(lresponse);
		}
 
  ///
  void AsyncEval(std::vector<int>& asv, int mode, int& priority, ResponseT* response, int tag=-1)
		{
		LResponseT* lresponse = new_lresponse(response, false);
		handle->AsyncEval(asv,mode,priority,lresponse,tag);
		}

  ///
  void AsyncEval(int mode, int& priority, ResponseT* response, int tag=-1)
		{
		state()->init_request_asv(mode);
		LResponseT* lresponse = new_lresponse(response, false);
		handle->AsyncEval(state()->REQUEST_ASV,mode,priority,lresponse,
						tag);
		}

  ///
  void AsyncEvalF(real* value, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferF(value);
		AsyncEval(State->F_ASV, mode_f, priority, response,tag);
		}

  ///
  void AsyncEvalF(typename ResponseT::realarray_t* values, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferF(&(values[0]));
		AsyncEval(State->F_ASV, mode_f, priority, response,tag);
		}

  ///
  void AsyncEvalCF(typename ResponseT::vector_t* value, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferCF(value);
		AsyncEval(State->CF_ASV, mode_cf, priority, response,tag);
		}

  ///
  void AsyncEvalG(typename ResponseT::vector_t *value, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferG(value);
		AsyncEval(State->G_ASV, mode_g, priority, response,tag);
		}

  ///
  void AsyncEvalFG(real* ans, typename ResponseT::vector_t *value, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferF(ans);
		response->response_bufferG(value);
		AsyncEval(State->FG_ASV, mode_f | mode_g,priority,response,tag);
		}
 
  ///
  void AsyncEvalCG(typename ResponseT::vectorarray_t * value, int& priority, int tag=-1)
		{
  		OptProblemStateBase* State = state();
		ResponseT* response = CachedAllocator<ResponseT> :: allocate();
		response->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		response->response_bufferCG(value);
		AsyncEval(State->CG_ASV, mode_cg, priority, response,tag);
		}

  ///
  void set_point(const DomainT& point_)
		{
		map_domain(*lpoint,point_);
		handle->set_point(*lpoint);
		}

  ///
  DomainT& get_point()
		{
		map_domain(point,handle->get_point());
		return point;
		}

  ///
  void set_linear_constraints(const typename ResponseT::matrix_t& mat, const typename ResponseT::vector_t& rhs, 
					const int num_equality)
		{
		typename LResponseT::vector_t tvec;
		typename LResponseT::matrix_t tmat;
		tvec << rhs;
		map_matrix(tmat,mat);
		handle->set_linear_constraints(tmat,tvec,num_equality);
		}

  ///
  const typename ResponseT::matrix_t& linear_constraint_mat() 
		{
		map_matrix(tmp_mat,handle->linear_constraint_mat());
		return tmp_mat;
		}

  ///
  const typename ResponseT::vector_t& linear_constraint_rhs() 
		{
		tmp_vec << handle->linear_constraint_rhs();
		return tmp_vec;
		}

  ///
  OptProblemStateBase* state()
		{return handle->state();}

  ///
  OptApplicationBase* app()
		{return handle->app();}

  ///
  void extract_state(OptProblemAppBody<DomainT,ResponseT>* app_)
  		{ app_->State = 0; }

protected:

  /// A handle to an OptProblem.
  SmartPtr< OptProblemAppHandle<LDomainT,LResponseT> > handle;
  //SmartPtr< OptProblemAppHandle<DomainT,ResponseT,LDomainT,LResponseT> > handle;

  /// The point being evaluated.
  DomainT point;

  /// A reference to the mapped version of the point
  LDomainT* lpoint;

  /// Temporary vector
  typename ResponseT::vector_t tmp_vec;

  /// Temporary vector
  typename ResponseT::matrix_t tmp_mat;

private:

  /// Evalaute the mapped version of the point
  LResponseT* LEval(std::vector<int>& asv, int mode)
		{
  		OptProblemStateBase* State = state();
		LResponseT* lresponse = 
				CachedAllocator<LResponseT> :: allocate();
		lresponse->resize(State->numFunctions,
				State->numConstraints, 
				State->num_real_params);
		handle->Eval(asv,mode,*lresponse);
		return lresponse;
		}

  /// Allocate a response for the mapped point.
  LResponseT* new_lresponse(ResponseT* response, bool ownership=true)
		{
		AppResponseLValues<typename LResponseT::types_t, typename ResponseT::types_t>* lval = 
			CachedAllocator<AppResponseLValues<typename LResponseT::types_t,typename ResponseT::types_t> > :: allocate();
		lval->values = response->values;
		lval->response = response;
		lval->response_owned = ownership;
		LResponseT* lresponse = 
				CachedAllocator<LResponseT> :: allocate();
		lresponse->reset(response->info,lval);
		return lresponse;
		}

};

} // namespace colin

#endif
