/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       NestedModel
//- Description: Implementation code for the NestedModel class
//- Owner:       Mike Eldred
//- Checked by:

#include "NestedModel.H"
#include "ProblemDescDB.H"
#include "system_defs.h"

static const char rcsId[]="@(#) $Id: NestedModel.C 5337 2008-10-09 18:12:05Z mseldre $";


namespace Dakota {

NestedModel::NestedModel(ProblemDescDB& problem_db):
  Model(BaseConstructor(), problem_db), nestedModelEvals(0),
  optInterfacePointer(problem_db.get_string("model.interface_pointer"))
{
  // NestedModel may set the DB list nodes, but is required to restore them
  // to their previous setting in order to remove the need to continuously
  // reset at the strategy level (which would be wasteful since the type
  // of derived model may not be known at the strategy level).
  size_t method_index = problem_db.get_db_method_node(); // for restoration
  size_t model_index  = problem_db.get_db_model_node();  // for restoration

  // interface for non-nested data is optional
  if (optInterfacePointer.empty())
    numOptInterfPrimary = numOptInterfIneqCon = numOptInterfEqCon = 0;
  else {
    const String& optional_interf_resp_pointer
      = problem_db.get_string("model.optional_interface_responses_pointer");
    bool valid_resp_pointer = !optional_interf_resp_pointer.empty();
    if (valid_resp_pointer)
      problem_db.set_db_responses_node(optional_interf_resp_pointer);

    numOptInterfIneqCon
      = problem_db.get_sizet("responses.num_nonlinear_inequality_constraints");
    numOptInterfEqCon
      = problem_db.get_sizet("responses.num_nonlinear_equality_constraints");
    optInterfaceResponse = problem_db.get_response(currentVariables);
    optionalInterface    = problem_db.get_interface();
    const size_t& num_fns = optInterfaceResponse.num_functions();
    numOptInterfPrimary = num_fns - numOptInterfIneqCon - numOptInterfEqCon;

    if (valid_resp_pointer) {
      // Echo a warning if there is a user specification of constraint bounds/
      // targets that will be superceded by top level constraint bounds/targets.
      const RealVector& interf_ineq_l_bnds
	= problem_db.get_drv("responses.nonlinear_inequality_lower_bounds");
      const RealVector& interf_ineq_u_bnds
	= problem_db.get_drv("responses.nonlinear_inequality_upper_bounds");
      const RealVector& interf_eq_targets
	= problem_db.get_drv("responses.nonlinear_equality_targets");
      bool warning_flag = false;
      size_t i;
      for (i=0; i<numOptInterfIneqCon; i++)
	if ( interf_ineq_l_bnds[i] > -DBL_MAX || interf_ineq_u_bnds[i] != 0.0 )
	  warning_flag = true;
      for (i=0; i<numOptInterfEqCon; i++)
	if ( interf_eq_targets[i] != 0.0 )
	  warning_flag = true;
      if (warning_flag)
	Cerr << "Warning: nonlinear constraint bounds and targets in nested "
	     << "model optional interfaces\n         are superceded by "
	     << "composite response constraint bounds and targets." << endl;
    }

    // db_responses restore not needed since set_db_list_nodes below will reset
  }

  const String& sub_method_pointer
    = problem_db.get_string("model.nested.sub_method_pointer");
  problem_db.set_db_list_nodes(sub_method_pointer); // even if empty

  subModel = problem_db.get_model();
  //check_submodel_compatibility(subModel); // all checks performed below
  // if outer level output is verbose/debug, request fine-grained evaluation 
  // reporting for purposes of the final output summary.  This allows verbose
  // final summaries without verbose output on every sub-iterator completion.
  if (outputLevel > NORMAL_OUTPUT)
    subModel.fine_grained_evaluation_counters();

  //if (iteratorCommRank == 0) // only master needs an iterator object
    subIterator = problem_db.get_iterator(subModel);
  subIterator.sub_iterator_flag(true);

  problem_db.set_db_method_node(method_index); // restore method only
  problem_db.set_db_model_nodes(model_index);  // restore all model nodes

  // Retrieve the variable mapping inputs, perform error checks, and
  // convert from strings to indices for efficiency at run time.
  const StringArray& primary_var_mapping
    = problem_db.get_dsa("model.nested.primary_variable_mapping");
  const StringArray& secondary_var_mapping
    = problem_db.get_dsa("model.nested.secondary_variable_mapping");
  size_t i, num_var_map_1 = primary_var_mapping.length(),
    num_var_map_2   = secondary_var_mapping.length(),
    num_curr_c_vars = currentVariables.cv(),
    num_curr_d_vars = currentVariables.dv(),
    num_curr_vars   = num_curr_c_vars + num_curr_d_vars;
  // Error checks: maps can be empty strings, but must be present to assure
  // correct association.
  if ( ( num_var_map_1 && num_var_map_1 != num_curr_vars ) ||
       ( num_var_map_2 && num_var_map_2 != num_var_map_1 ) ) {
    Cerr << "Error: length of variable mapping specification(s) does not "
	 << "match number of active variables." << endl;
    abort_handler(-1);
  }
  primaryACVarMapIndices.reshape(num_curr_c_vars);
  primaryADVarMapIndices.reshape(num_curr_d_vars);
  extraCVarsData.resize(num_curr_c_vars);
  extraDVarsData.resize(num_curr_d_vars);
  if (num_var_map_2) {
    secondaryACVarMapTargets.reshape(num_curr_c_vars);
    secondaryADVarMapTargets.reshape(num_curr_d_vars);
  }
  const StringArray& curr_c_types
    = currentVariables.continuous_variable_types();
  const StringArray& submodel_a_c_types
    = subModel.all_continuous_variable_types();
  //const StringArray& curr_c_labels
  //  = currentVariables.continuous_variable_labels();
  const StringArray& submodel_a_c_labels
    = subModel.all_continuous_variable_labels();
  size_t sm_acv_cntr = 0, sm_adv_cntr = 0, sm_acv_avail = 0, sm_adv_avail = 0;
  String prev_c_type, prev_d_type, empty_str;
  for (i=0; i<num_curr_c_vars; i++) {
    const String& map1 = (num_var_map_1) ? primary_var_mapping[i] : empty_str;
    if (map1.empty()) {
      extraCVarsData[i] = true;
      // Can't use label matching, since subModel labels may not be updated
      // until runtime.  index() returns the _first_ instance of the type.
      const String& curr_c_type = curr_c_types[i];
      size_t sm_acv_offset = submodel_a_c_types.index(curr_c_type);
      if (sm_acv_offset == _NPOS) {
	Cerr << "Error: active continuous variable type '" << curr_c_type
	     << "' could not be matched within all sub-model continuous "
	     << "variable types." << endl;
	abort_handler(-1);
      }
      // For multiple types, sm_acv_cntr must be reset to 0 at the type change
      if (curr_c_type != prev_c_type) {
	sm_acv_cntr  = 0;
	sm_acv_avail = submodel_a_c_types.count(curr_c_type);
      }
      primaryACVarMapIndices[i] = sm_acv_offset + sm_acv_cntr++;
      prev_c_type = curr_c_type;
      if (sm_acv_cntr > sm_acv_avail) {
	Cerr << "Error: default insertions of type '" << curr_c_type
	     << "' exceed sub-model allocation." << endl;
	abort_handler(-1);
      }
      if (num_var_map_2)
	secondaryACVarMapTargets[i] = NO_TARGET;
    }
    else {
      extraCVarsData[i] = false;
      size_t ac_index1 = submodel_a_c_labels.index(map1);
      if (ac_index1 == _NPOS) {
	Cerr << "Error: primary mapping " << map1 << " could not be matched "
	     << "within all sub-model continuous variable labels." << endl;
	abort_handler(-1);
      }
      primaryACVarMapIndices[i] = ac_index1;
      if (num_var_map_2) {
	const String& map2 = secondary_var_mapping[i];
	if (map2.empty())
	  secondaryACVarMapTargets[i] = NO_TARGET;
	else {
	  const String& type = submodel_a_c_types[ac_index1];
	  if (type == "continuous_design") {
	    if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = CDV_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = CDV_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "continuous design variables." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "normal_uncertain") {
	    if (map2 == "mean")
	      secondaryACVarMapTargets[i] = N_MEAN;
	    else if (map2 == "std_deviation")
	      secondaryACVarMapTargets[i] = N_STD_DEV;
	    else if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = N_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = N_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "normal distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "lognormal_uncertain") {
	    // TO DO: 3 parameter lognormal adds a location parameter
	    if (map2 == "mean")
	      secondaryACVarMapTargets[i] = LN_MEAN;
	    else if (map2 == "std_deviation") {
	      if (subModel.lognormal_std_deviations().empty()) {
		Cerr << "Error: cannot insert std_deviation for lognormal "
		     << "error factor specification." << endl;
		abort_handler(-1);
	      }
	      else
		secondaryACVarMapTargets[i] = LN_STD_DEV;
	    }
	    else if (map2 == "error_factor") {
	      if (subModel.lognormal_error_factors().empty()) {
		Cerr << "Error: cannot insert error_factor for lognormal "
		     << "std_deviation specification." << endl;
		abort_handler(-1);
	      }
	      else
		secondaryACVarMapTargets[i] = LN_ERR_FACT;
	    }
	    else if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = LN_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = LN_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "lognormal distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "uniform_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = U_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = U_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "uniform distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "loguniform_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = LU_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = LU_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "loguniform distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "triangular_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "mode")
	      secondaryACVarMapTargets[i] = T_MODE;
	    else if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = T_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = T_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "triangular distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "exponential_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "beta")
	      secondaryACVarMapTargets[i] = E_BETA;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "exponential distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "beta_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "alpha")
	      secondaryACVarMapTargets[i] = B_ALPHA;
	    else if (map2 == "beta")
	      secondaryACVarMapTargets[i] = B_BETA;
	    else if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = B_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = B_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "beta distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "gamma_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "alpha")
	      secondaryACVarMapTargets[i] = GA_ALPHA;
	    else if (map2 == "beta")
	      secondaryACVarMapTargets[i] = GA_BETA;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "gamma distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "gumbel_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "alpha")
	      secondaryACVarMapTargets[i] = GU_ALPHA;
	    else if (map2 == "beta")
	      secondaryACVarMapTargets[i] = GU_BETA;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "gumbel distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "frechet_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "alpha")
	      secondaryACVarMapTargets[i] = F_ALPHA;
	    else if (map2 == "beta")
	      secondaryACVarMapTargets[i] = F_BETA;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "frechet distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "weibull_uncertain") {
	    // TO DO: mean/std deviation or location/scale
	    if (map2 == "alpha")
	      secondaryACVarMapTargets[i] = W_ALPHA;
	    else if (map2 == "beta")
	      secondaryACVarMapTargets[i] = W_BETA;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "weibull distributions." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "continuous_state") {
	    if (map2 == "lower_bound")
	      secondaryACVarMapTargets[i] = CSV_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryACVarMapTargets[i] = CSV_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "continuous state variables." << endl;
	      abort_handler(-1);
	    }
	  }
	  else {
	    Cerr << "Error: " << type << " variable type not supported in "
		 << "secondary variable mappings." << endl;
	    abort_handler(-1);
	  }
	}
      }
    }
  }
  const StringArray& curr_d_types = currentVariables.discrete_variable_types();
  const StringArray& submodel_a_d_types
    = subModel.all_discrete_variable_types();
  //const StringArray& curr_d_labels
  //  = currentVariables.discrete_variable_labels();
  const StringArray& submodel_a_d_labels
    = subModel.all_discrete_variable_labels();
  for (i=0; i<num_curr_d_vars; i++) {
    const String& map1 = (num_var_map_1) ?
      primary_var_mapping[i+num_curr_c_vars] : empty_str;
    if (map1.empty()) {
      extraDVarsData[i] = true;
      // Can't use label matching, since subModel labels may not be updated
      // until runtime.  index() returns the _first_ instance of the type.
      const String& curr_d_type = curr_d_types[i];
      size_t sm_adv_offset = submodel_a_d_types.index(curr_d_type);
      if (sm_adv_offset == _NPOS) {
	Cerr << "Error: active discrete variable type '" << curr_d_type
	     << "' could not be matched within all sub-model discrete "
	     << "variable types." << endl;
	abort_handler(-1);
      }
      // For multiple types, sm_adv_cntr must be reset to 0 at the type change
      if (curr_d_type != prev_d_type) {
	sm_adv_cntr  = 0;
	sm_adv_avail = submodel_a_d_types.count(curr_d_type);
      }
      primaryADVarMapIndices[i] = sm_adv_offset + sm_adv_cntr++;
      prev_d_type = curr_d_type;
      if (sm_adv_cntr > sm_adv_avail) {
	Cerr << "Error: default insertions of type '" << curr_d_type
	     << "' exceed sub-model allocation." << endl;
	abort_handler(-1);
      }
      if (num_var_map_2)
	secondaryADVarMapTargets[i] = NO_TARGET;
    }
    else {
      extraDVarsData[i] = false;
      size_t ad_index1 = submodel_a_d_labels.index(map1);
      if (ad_index1 == _NPOS) {
	Cerr << "Error: primary mapping " << map1 << " could not be matched "
	     << "to active sub-model discrete variables." << endl;
	abort_handler(-1);
      }
      primaryADVarMapIndices[i] = ad_index1;
      if (num_var_map_2) {
	const String& map2 = secondary_var_mapping[i+num_curr_c_vars];
	if (map2.empty())
	  secondaryADVarMapTargets[i] = NO_TARGET;
	else {
	  const String& type = submodel_a_d_types[ad_index1];
	  if (type == "discrete_design") {
	    if (map2 == "lower_bound")
	      secondaryADVarMapTargets[i] = DDV_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryADVarMapTargets[i] = DDV_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "discrete design variables." << endl;
	      abort_handler(-1);
	    }
	  }
	  else if (type == "discrete_state") {
	    if (map2 == "lower_bound")
	      secondaryADVarMapTargets[i] = DSV_LWR_BND;
	    else if (map2 == "upper_bound")
	      secondaryADVarMapTargets[i] = DSV_UPR_BND;
	    else {
	      Cerr << "Error: " << map2 << " mapping not supported for "
		   << "discrete state variables." << endl;
	      abort_handler(-1);
	    }
	  }
	  else {
	    Cerr << "Error: " << type << " variable type not supported in "
		 << "secondary variable mappings." << endl;
	    abort_handler(-1);
	  }
	}
      }
    }
  }
  subIterator.variable_mappings(primaryACVarMapIndices, primaryADVarMapIndices,
    secondaryACVarMapTargets, secondaryADVarMapTargets);

  const RealVector& primary_resp_coeffs_drv
    = problem_db.get_drv("model.nested.primary_response_mapping");
  const RealVector& secondary_resp_coeffs_drv
    = problem_db.get_drv("model.nested.secondary_response_mapping");
  if (primary_resp_coeffs_drv.empty() && secondary_resp_coeffs_drv.empty()) {
    Cerr << "Error: no mappings provided for sub-iterator functions." << endl;
    abort_handler(-1);
  }
  // Convert vectors to matrices using the number of subIterator response
  // results (e.g., the number of UQ statistics) as the number of columns
  // (rows are open ended since the number of subModel primary fns may
  // be different from the total number of primary fns and subModel's
  // constraints are a subset of the total constraints).
  numSubIterFns = subIterator.response_results().num_functions();
  if (!primary_resp_coeffs_drv.empty()) {
    // this error would also be caught within copy_data(), but by checking here,
    // we can provide a more helpful error message.
    if (primary_resp_coeffs_drv.length() % numSubIterFns) {
      Cerr << "Error: number of entries in primary_response_mapping ("
	   << primary_resp_coeffs_drv.length() << ") not evenly divisible"
	   << "\n       by number of sub-iterator response functions ("
	   << numSubIterFns << ")." << endl;
      abort_handler(-1);
    }
    copy_data(primary_resp_coeffs_drv, primaryRespCoeffs, 0, numSubIterFns);
  }
  if (!secondary_resp_coeffs_drv.empty()) {
    // this error would also be caught within copy_data(), but by checking here,
    // we can provide a more helpful error message.
    if (secondary_resp_coeffs_drv.length() % numSubIterFns) {
      Cerr << "Error: number of entries in secondary_response_mapping ("
	   << secondary_resp_coeffs_drv.length() << ") not evenly divisible"
	   << "\n       by number of sub-iterator response functions ("
	   << numSubIterFns << ")." << endl;
      abort_handler(-1);
    }
    copy_data(secondary_resp_coeffs_drv, secondaryRespCoeffs, 0, numSubIterFns);
  }

  // Back out the number of eq/ineq constraints within secondaryRespCoeffs
  // (subIterator constraints) from the total number of equality/inequality
  // constraints and the number of interface equality/inequality constraints.
  size_t num_mapped_ineq_con
    = problem_db.get_sizet("responses.num_nonlinear_inequality_constraints"),
  num_mapped_eq_con
    = problem_db.get_sizet("responses.num_nonlinear_equality_constraints");
  numSubIterMappedIneqCon = num_mapped_ineq_con - numOptInterfIneqCon;
  numSubIterMappedEqCon   = num_mapped_eq_con   - numOptInterfEqCon;
}


/** Update subModel's inactive variables with active variables from
    currentVariables, compute the optional interface and sub-iterator
    responses, and map these to the total model response. */
void NestedModel::derived_compute_response(const ActiveSet& set)
{
  // Set currentResponse asv and extract opt_interface_asv/sub_iterator_asv
  currentResponse.active_set(set);
  ActiveSet opt_interface_set, sub_iterator_set;
  bool      opt_interface_map, sub_iterator_map;
  set_mapping(set, opt_interface_set, opt_interface_map,
	           sub_iterator_set,  sub_iterator_map);

  // Perform optionalInterface map (opt_interface_asv is updated within
  // optInterfaceResponse by map):
  if (opt_interface_map) {
    Cout << "\n----------------------------------------------------------------"
	 << "--\nNestedModel Evaluation " << setw(4) << nestedModelEvals+1 
	 << ": performing optional interface mapping\n-------------------------"
	 << "-----------------------------------------\n";
    component_parallel_mode(OPTIONAL_INTERFACE);
    optionalInterface.map(currentVariables, opt_interface_set,
			  optInterfaceResponse);
  }

  if (sub_iterator_map) {
    // need comm set up and master break off (see Strategy::run_iterator())
    Cout << "\n-------------------------------------------------\n"
	 << "NestedModel Evaluation " << setw(4) << nestedModelEvals+1 
	 << ": running sub_iterator"
	 << "\n-------------------------------------------------\n";
    component_parallel_mode(SUB_MODEL);
    update_sub_model();
    subIterator.response_results_active_set(sub_iterator_set);
    if (subIterator.output_level() > NORMAL_OUTPUT)
      subIterator.run_iterator(Cout); // include sub-iterator diagnostics
    else
      subIterator.run_iterator();     // quiet mode: no annotation or summary
    Cout << "\nActive response data from sub_iterator:\n"
	 << subIterator.response_results() << '\n';
  }
  // Perform mapping of optInterface & subIterator results to currentResponse
  response_mapping(optInterfaceResponse, subIterator.response_results(),
		   currentResponse);

  nestedModelEvals++;
}


/** Not currently supported by NestedModels (need to add concurrent
    iterator support).  As a result, derived_synchronize() and
    derived_synchronize_nowait() are inactive as well). */
void NestedModel::derived_asynch_compute_response(const ActiveSet& set)
{
  Cerr << "Error: derived_asynch_compute_response not yet available in "
       << "NestedModel." << endl;
  abort_handler(-1);

  // Set currentResponse asv and extract opt_interface_asv/sub_iterator_asv
  currentResponse.active_set(set);
  ActiveSet opt_interface_set, sub_iterator_set;
  bool      opt_interface_map, sub_iterator_map;
  set_mapping(set, opt_interface_set, opt_interface_map,
	           sub_iterator_set,  sub_iterator_map);

  // Perform optionalInterface map (opt_interface_asv is updated within
  // optInterfaceResponse by map):
  if (opt_interface_map) {
    Cout << "\n----------------------------------------------------------------"
	 << "--\nNestedModel Evaluation " << setw(4) << nestedModelEvals+1 
	 << ": performing optional interface mapping\n-------------------------"
	 << "-----------------------------------------\n";
    // don't need to set component parallel mode since this only queues the job
    optionalInterface.map(currentVariables, opt_interface_set,
			  optInterfaceResponse, true);
  }

  if (sub_iterator_map) {
    // need comm set up and master break off (see Strategy::run_iterator())
    Cout << "\n-------------------------------------------------\n"
	 << "NestedModel Evaluation " << setw(4) << nestedModelEvals+1 
	 << ": running sub_iterator"
	 << "\n-------------------------------------------------\n";

    // *** Need concurrent iterator support ***

    // Only makes sense for message passing (can't launch an asynch local 
    // iterator): load up queue of iterator jobs to be scheduled in derived 
    // synchronize fns.
    /*
    subIteratorEvalId++;
    beforeSynchIdList.insert(subIteratorEvalId);
    ParamResponsePair current_pair(vars, nestedModelId, response,
                                   subIteratorEvalId);
    beforeSynchPRPList.insert(current_pair);
    // jobs are not queued until call to synch() to allow self-scheduling.
    Cout << "(Nested model asynchronous job " << subIteratorEvalId 
         << " added to queue)\n";
    */
  }

  nestedModelEvals++;
  //varsMap[nestedModelEvals] = currentVariables.copy();
}


/* Asynchronous response computations are not currently supported by
   NestedModels.  Return a dummy to satisfy the compiler. */
//const ResponseArray& NestedModel::derived_synchronize()
//{
//  Cerr << "Error: derived_synchronize not yet available in NestedModel."
//       << endl;
//  abort_handler(-1);
//
//  varsMap.clear();
//  return responseArray;
//}


/* Asynchronous response computations are not currently supported by
   NestedModels.  Return a dummy to satisfy the compiler. */
//const IntResponseMap& NestedModel::derived_synchronize_nowait()
//{
//  Cerr <<"Error: derived_synchronize_nowait not yet available in NestedModel."
//       <<endl;
//  abort_handler(-1);
//
//  varsMap.erase(eval_id);
//  return responseMap;
//}


void NestedModel::
set_mapping(const ActiveSet& mapped_set, ActiveSet& opt_interface_set,
	    bool& opt_interface_map,     ActiveSet& sub_iterator_set,
	    bool& sub_iterator_map)
{
  size_t i, j, num_sub_iter_mapped_primary = primaryRespCoeffs.num_rows(), 
    num_sub_iter_mapped_con = secondaryRespCoeffs.num_rows(),
    num_opt_interf_con = numOptInterfIneqCon + numOptInterfEqCon,
    num_mapped_primary = max(numOptInterfPrimary, num_sub_iter_mapped_primary);

  const ShortArray& mapped_asv = mapped_set.request_vector();
  const UIntArray&  mapped_dvv = mapped_set.derivative_vector();

  if (mapped_asv.length() != num_mapped_primary + num_opt_interf_con +
                             num_sub_iter_mapped_con) {
    Cerr << "Error: mapped_asv length mismatch in NestedModel::asv_mapping()"
	 << endl;
    abort_handler(-1);
  }

  // opt_interface_asv:

  // num_mapped_primary >= numOptInterfPrimary with 1-to-1 correspondence
  // up to the cut off
  opt_interface_map = false;
  size_t num_opt_interf_fns = numOptInterfPrimary+num_opt_interf_con;
  ShortArray opt_interface_asv(num_opt_interf_fns);
  for (i=0; i<numOptInterfPrimary; i++)
    opt_interface_asv[i] = mapped_asv[i];
  // num_opt_interf_con has 1-to-1 correspondence with different offsets
  for (i=0; i<num_opt_interf_con; i++)
    opt_interface_asv[i+numOptInterfPrimary] = mapped_asv[i+num_mapped_primary];
  for (i=0; i<num_opt_interf_fns; i++)
    if (opt_interface_asv[i])
      { opt_interface_map = true; break; }
  if (opt_interface_map)
    opt_interface_set.request_vector(opt_interface_asv);

  // sub_iterator_asv:

  sub_iterator_map = false;
  ShortArray sub_iterator_asv(numSubIterFns, 0);
  // augment sub_iterator_asv based on mapped primary asv and primaryRespCoeffs
  for (i=0; i<num_sub_iter_mapped_primary; i++) {
    short mapped_asv_val = mapped_asv[i];
    if (mapped_asv_val) {
      for (j=0; j<numSubIterFns; j++)
	if (fabs(primaryRespCoeffs[i][j]) > DBL_MIN)
	  sub_iterator_asv[j] |= mapped_asv_val;
      sub_iterator_map = true;
    }
  }
  // augment sub_iterator_asv based on mapped constr asv and secondaryRespCoeffs
  for (i=0; i<num_sub_iter_mapped_con; i++) {
    short mapped_asv_val = mapped_asv[i+num_mapped_primary+num_opt_interf_con];
    if (mapped_asv_val) {
      for (j=0; j<numSubIterFns; j++)
	if (fabs(secondaryRespCoeffs[i][j]) > DBL_MIN)
	  sub_iterator_asv[j] |= mapped_asv_val;
      sub_iterator_map = true;
    }
  }
  if (sub_iterator_map)
    sub_iterator_set.request_vector(sub_iterator_asv);

  // opt_interface_dvv:

  if (opt_interface_map)
    opt_interface_set.derivative_vector(mapped_dvv);

  // sub_iterator_dvv:

  if (sub_iterator_map) {
    const UIntArray&     cv_ids = currentVariables.continuous_variable_ids();
    const UIntArray& sm_acv_ids = subModel.all_continuous_variable_ids();

    /* Old Approach: subIterator must decipher/replace top-level DVV as reqd.
    // Note: the ordering of top-level active variables may differ from the
    // ordering of the inactive sub-model variables.  However, the subModel's
    // inactive_continuous_variable_ids() omit inserted variables, which would
    // result in erroneous subIterator final response array sizing.  Therefore,
    // use the top-level active variable ids and rely on the subIterator to
    // perform the mappings to the subModel variables.
    sub_iterator_set.derivative_vector(cv_ids);
    */

    size_t num_mapped_dvv = mapped_dvv.length();
    UIntArray sub_iterator_dvv;
    bool var_map_2 = !secondaryACVarMapTargets.empty();
    if (!var_map_2)
      sub_iterator_dvv.reshape(num_mapped_dvv);
    for (i=0; i<num_mapped_dvv; i++) {
      size_t cv_index = cv_ids.index(mapped_dvv[i]);
      if (cv_index == _NPOS) {
	Cerr << "Error: NestedModel DVV component not contained within "
	     << "active continuous variable ids." << endl;
	abort_handler(-1);
      }
      size_t sm_acv_index = primaryACVarMapIndices[cv_index];
      unsigned int sm_acv_id = sm_acv_ids[sm_acv_index];
      if (var_map_2) { // enforce uniqueness in insertion targets
	if (!sub_iterator_dvv.contains(sm_acv_id))
	  sub_iterator_dvv.push_back(sm_acv_id);
      }
      else
	sub_iterator_dvv[i] = sm_acv_id;
    }
    sub_iterator_set.derivative_vector(sub_iterator_dvv);
    //Cout << "\nmapped_dvv:\n" << mapped_dvv << "\nsub_iterator_dvv:\n"
    //     << sub_iterator_dvv << '\n';
  }
}


/** In the OUU case,
\verbatim
optionalInterface fns = {f}, {g} (deterministic primary functions, constraints)
subIterator fns       = {S}      (UQ response statistics)

Problem formulation for mapped functions:
                  minimize    {f} + [W]{S}
                  subject to  {g_l} <= {g}    <= {g_u}
                              {a_l} <= [A]{S} <= {a_u}
                              {g}    == {g_t}
                              [A]{S} == {a_t}
\endverbatim

where [W] is the primary_mapping_matrix user input (primaryRespCoeffs
class attribute), [A] is the secondary_mapping_matrix user input
(secondaryRespCoeffs class attribute), {{g_l},{a_l}} are the top level
inequality constraint lower bounds, {{g_u},{a_u}} are the top level
inequality constraint upper bounds, and {{g_t},{a_t}} are the top
level equality constraint targets.

NOTE: optionalInterface/subIterator primary fns (obj/lsq/generic fns)
overlap but optionalInterface/subIterator secondary fns (ineq/eq 
constraints) do not.  The [W] matrix can be specified so as to allow

\li some purely deterministic primary functions and some combined:
    [W] filled and [W].num_rows() < {f}.length() [combined first] 
    \e or [W].num_rows() == {f}.length() and [W] contains rows of 
    zeros [combined last]
\li some combined and some purely stochastic primary functions:
    [W] filled and [W].num_rows() > {f}.length()
\li separate deterministic and stochastic primary functions:
    [W].num_rows() > {f}.length() and [W] contains {f}.length()
    rows of zeros.

If the need arises, could change constraint definition to allow overlap
as well: {g_l} <= {g} + [A]{S} <= {g_u} with [A] usage the same as for
[W] above.

In the UOO case, things are simpler, just compute statistics of each 
optimization response function: [W] = [I], {f}/{g}/[A] are empty. */
void NestedModel::response_mapping(const Response& opt_interface_response,
				   const Response& sub_iterator_response,
				   Response& mapped_response)
{
  // mapped data initialization
  const ShortArray& mapped_asv = mapped_response.active_set_request_vector();
  const UIntArray&  mapped_dvv = mapped_response.active_set_derivative_vector();
  size_t i, j, k, l, index, num_mapped_fns = mapped_asv.length(),
    num_mapped_deriv_vars = mapped_dvv.length();
  RealVector      mapped_fn_vals(num_mapped_fns, 0.);
  RealMatrix      mapped_fn_grads;
  RealMatrixArray mapped_fn_hessians;
  bool grad_flag = false, hess_flag = false;
  for (i=0; i<num_mapped_fns; i++) {
    if (mapped_asv[i] & 2)
      grad_flag = true;
    if (mapped_asv[i] & 4)
      hess_flag = true;
  }
  // Sanity check on derivative compatibility: (1) the optional interface
  // response must have the same DVV, (2) the derivatives in the sub-iterator
  // response must be with respect to the same variables; but since the
  // numbering may be different following insertion/augmentation, only the
  // DVV length is verified.
  if ( (grad_flag || hess_flag) &&
       (sub_iterator_response.active_set_derivative_vector().length() !=
	num_mapped_deriv_vars || (!optInterfacePointer.empty() &&
	opt_interface_response.active_set_derivative_vector() != mapped_dvv))) {
    Cerr << "Error: derivative variables vector mismatch in NestedModel::"
         << "response_mapping()." << endl;
    abort_handler(-1);
  }
  if (grad_flag) {
    mapped_fn_grads.reshape_2d(num_mapped_fns, num_mapped_deriv_vars);
    mapped_fn_grads = 0.;
  }
  if (hess_flag) {
    mapped_fn_hessians.reshape(num_mapped_fns);
    for (i=0; i<num_mapped_fns; i++) {
      mapped_fn_hessians[i].reshape_2d(num_mapped_deriv_vars,
				       num_mapped_deriv_vars);
      mapped_fn_hessians[i]=0.;
    }
  }

  RealVector      empty_rv;
  RealMatrix      empty_rm;
  RealMatrixArray empty_rma;

  // interface data extraction
  //opt_interface_asv = opt_interface_response.active_set_request_vector();
  const RealVector& opt_interface_fn_vals  = (!optInterfacePointer.empty())
    ? opt_interface_response.function_values() : empty_rv;
  const RealMatrix& opt_interface_fn_grads = (!optInterfacePointer.empty() &&
    grad_flag) ? opt_interface_response.function_gradients() : empty_rm;
  const RealMatrixArray& opt_interface_fn_hessians
    = (!optInterfacePointer.empty() && hess_flag) ?
    opt_interface_response.function_hessians() : empty_rma;

  // subIterator data extraction
  //const ShortArray& sub_iterator_asv
  //  = sub_iterator_response.active_set_request_vector();
  const RealVector& sub_iterator_fn_vals
    = sub_iterator_response.function_values();
  const RealMatrix& sub_iterator_fn_grads
    = (grad_flag) ? sub_iterator_response.function_gradients() : empty_rm;
  const RealMatrixArray& sub_iterator_fn_hessians
    = (hess_flag) ? sub_iterator_response.function_hessians()  : empty_rma;

  // counter initialization & sanity checking
  size_t num_opt_interf_fns = opt_interface_fn_vals.length(),
    num_opt_interf_con      = numOptInterfIneqCon + numOptInterfEqCon,
    num_sub_iter_mapped_primary = primaryRespCoeffs.num_rows(),
    num_sub_iter_mapped_con     = secondaryRespCoeffs.num_rows(),
    num_mapped_primary = max(numOptInterfPrimary, num_sub_iter_mapped_primary);
  // NOTE: numSubIterFns != num_sub_iter_mapped_primary+num_sub_iter_mapped_con
  // since subIterator response is converted to sub_iter_mapped_primary/con
  // through the action of [W] and [A].
  if (num_opt_interf_fns      != numOptInterfPrimary + num_opt_interf_con
   || num_sub_iter_mapped_con != numSubIterMappedIneqCon + numSubIterMappedEqCon
   || num_mapped_fns          != num_mapped_primary + num_opt_interf_con +
                                 num_sub_iter_mapped_con) {
    Cerr << "Error: bad function counts in NestedModel::response_mapping()."
         << endl;
    abort_handler(-1);
  }

  // build mapped response data:

  // {f} + [W]{S}:
  for (i=0; i<num_mapped_primary; i++) {
    if (mapped_asv[i] & 1) { // mapped_fn_vals
      if (i<numOptInterfPrimary)
        mapped_fn_vals[i] = opt_interface_fn_vals[i]; // {f}
      if (i<num_sub_iter_mapped_primary) {
        Real inner_prod = 0.;
        for (j=0; j<numSubIterFns; j++)
          inner_prod += primaryRespCoeffs[i][j]*sub_iterator_fn_vals[j];
        mapped_fn_vals[i] += inner_prod;          // [W]{S}
      }
    }
    if (mapped_asv[i] & 2) { // mapped_fn_grads
      if (i<numOptInterfPrimary)
        mapped_fn_grads[i] = opt_interface_fn_grads[i];
      if (i<num_sub_iter_mapped_primary) {
        for (j=0; j<num_mapped_deriv_vars; j++) {
          Real inner_prod = 0.;
          for (k=0; k<numSubIterFns; k++)
            inner_prod += primaryRespCoeffs[i][k]*sub_iterator_fn_grads[k][j];
          mapped_fn_grads[i][j] += inner_prod;
	}
      }
    }
    if (mapped_asv[i] & 4) { // mapped_fn_hessians
      if (i<numOptInterfPrimary)
        mapped_fn_hessians[i] = opt_interface_fn_hessians[i];
      if (i<num_sub_iter_mapped_primary) {
        for (j=0; j<num_mapped_deriv_vars; j++) {
          for (k=0; k<num_mapped_deriv_vars; k++) {
            Real inner_prod = 0.;
            for (l=0; l<numSubIterFns; l++)
              inner_prod += primaryRespCoeffs[i][l]
                         *  sub_iterator_fn_hessians[l][j][k];
            mapped_fn_hessians[i][j][k] += inner_prod;
	  }
	}
      }
    }
  }

  // {g}:
  for (i=0; i<num_opt_interf_con; i++) {
    if (i<numOptInterfIneqCon) // {g_l} <= {g} <= {g_u}
      index = i + num_mapped_primary;
    else                       // {g} == {g_t}
      index = i + num_mapped_primary + numOptInterfIneqCon
            + numSubIterMappedIneqCon;
    if (mapped_asv[index] & 1) // mapped_fn_vals
      mapped_fn_vals[index] = opt_interface_fn_vals[i+numOptInterfPrimary];
    if (mapped_asv[index] & 2) // mapped_fn_grads
      mapped_fn_grads[index] = opt_interface_fn_grads[i+numOptInterfPrimary];
    if (mapped_asv[index] & 4) // mapped_fn_hessians
      mapped_fn_hessians[index]
        = opt_interface_fn_hessians[i+numOptInterfPrimary];
  }

  // [A]{S}:
  for (i=0; i<num_sub_iter_mapped_con; i++) {
    if (i<numSubIterMappedIneqCon) // {a_l} <= [A]{S} <= {a_u}
      index = i + num_mapped_primary + numOptInterfIneqCon;
    else                            // [A]{S} == {a_t}
      index = i + num_mapped_primary + num_opt_interf_con
            + numSubIterMappedIneqCon;
    if (mapped_asv[index] & 1) { // mapped_fn_vals
      Real inner_prod = 0.;
      for (j=0; j<numSubIterFns; j++)
        inner_prod += secondaryRespCoeffs[i][j]*sub_iterator_fn_vals[j];
      mapped_fn_vals[index] = inner_prod;
    }
    if (mapped_asv[index] & 2) { // mapped_fn_grads
      for (j=0; j<num_mapped_deriv_vars; j++) {
        Real inner_prod = 0.;
        for (k=0; k<numSubIterFns; k++)
          inner_prod += secondaryRespCoeffs[i][k]*sub_iterator_fn_grads[k][j];
        mapped_fn_grads[index][j] = inner_prod;
      }
    }
    if (mapped_asv[index] & 4) { // mapped_fn_hessians
      for (j=0; j<num_mapped_deriv_vars; j++) {
        for (k=0; k<num_mapped_deriv_vars; k++) {
          Real inner_prod = 0.;
          for (l=0; l<numSubIterFns; l++)
            inner_prod += secondaryRespCoeffs[i][l]
	               *  sub_iterator_fn_hessians[l][j][k];
          mapped_fn_hessians[index][j][k] = inner_prod;
	}
      }
    }
  }

  mapped_response.function_values(mapped_fn_vals);
  if (grad_flag)
    mapped_response.function_gradients(mapped_fn_grads);
  if (hess_flag)
    mapped_response.function_hessians(mapped_fn_hessians);
  Cout << "\n---------------------------\nNestedModel total response:"
       << "\n---------------------------\n\nActive response data from nested "
       << "mapping:\n" << mapped_response << '\n';
}


void NestedModel::component_parallel_mode(short mode)
{
  // mode may be correct, but can't guarantee active parallel configuration is
  // in synch
  //if (componentParallelMode == mode)
  //  return; // already in correct parallel mode

  // terminate previous serve mode (if active)
  if (componentParallelMode != mode) {
    if (componentParallelMode == OPTIONAL_INTERFACE) {
      parallelLib.parallel_configuration_iterator(modelPCIter);
      const ParallelConfiguration& pc = parallelLib.parallel_configuration();
      if (parallelLib.si_parallel_level_defined() && 
	  pc.si_parallel_level().server_communicator_size() > 1)
	optionalInterface.stop_evaluation_servers();
    }
    else if (componentParallelMode == SUB_MODEL) {
      parallelLib.parallel_configuration_iterator(
        subModel.parallel_configuration_iterator());
      const ParallelConfiguration& pc = parallelLib.parallel_configuration();
      if (parallelLib.si_parallel_level_defined() && 
	  pc.si_parallel_level().server_communicator_size() > 1)
	subModel.stop_servers();
    }
  }

  // set ParallelConfiguration for new mode
  if (mode == SUB_MODEL)
    parallelLib.parallel_configuration_iterator(
      subModel.parallel_configuration_iterator());
  else if (mode == OPTIONAL_INTERFACE)
    parallelLib.parallel_configuration_iterator(modelPCIter);

  // retrieve new ParallelConfiguration data
  int new_iter_comm_size = 1;
  if (parallelLib.si_parallel_level_defined()) {
    const ParallelConfiguration& pc = parallelLib.parallel_configuration();
    new_iter_comm_size = pc.si_parallel_level().server_communicator_size();
  }

  // activate the new serve mode
  if (new_iter_comm_size > 1 && componentParallelMode != mode)
    parallelLib.bcast_i(mode);
  componentParallelMode = mode;
}


void NestedModel::update_sub_model()
{
  // Update subModel variables using active currentVariables through a
  // combination of variable insertions and augmentations.  Insertions
  // and augmentations are not mutually exclusive, so subModel updates must
  // account for both, potentially on a variable by variable basis.

  // Top-level active variables/bounds/labels are mapped to the subModel, but
  // top-level linear/nonlinear constraint coefficients/bounds/targets are not
  // propagated (as they are in the surrogate model classes) since they are not
  // relevant at the subModel level.

  // For defined variable mappings, insert active currentVariables into subModel
  // variables.  When secondary mappings are defined, this insertion involves
  // the updating of sub-parameters for the subModel variables.  When only
  // primary mappings are defined, the insertions update the subModel variable
  // values.  For null primary mappings (empty strings), augmentations are
  // performed by updating the subModel variables of the same variable type as
  // the active currentVariables.

  // Bounds and labels are only updated for the augmented variables.  Labels
  // should not be updated for inserted variables, and bounds should not be
  // updated for inserted variables with secondary mappings (since this would
  // be redundant with bound parameter insertion).  It may be desirable in the
  // future to update bounds for inserted variables with only primary mappings,
  // but this is omitted for now.

  // Distribution parameter insertion is used for second-order probability
  // (UQ sample inserted into distr. parameters for UQ reliability/sampling)
  // and for OUU with design/uncertain overlap (optimization design variables
  // inserted into distr. parameters for UQ reliability/sampling).  For now,
  // only the existing distribution parameters are supported for the secondary
  // variable mappings.  In the future, it would be desirable to allow
  // additional parameter mappings, such as setting the mean of a weibull.

  size_t i, num_var_map_1c = primaryACVarMapIndices.length(),
    num_var_map_1d = primaryADVarMapIndices.length(),
    num_var_map_2c = secondaryACVarMapTargets.length(),
    num_var_map_2d = secondaryADVarMapTargets.length(), num_cdv, num_nuv,
    num_lnuv, num_uuv, num_luuv, num_tuv, num_euv, num_buv, num_gauv, num_guuv,
    num_fuv, num_wuv;
  bool need_sm_c = false, need_sm_acv = false,
       need_sm_d = false, need_sm_adv = false;
  for (i=0; i<num_var_map_1c; i++) {
    if (!num_var_map_2c || secondaryACVarMapTargets[i] == NO_TARGET) {
      need_sm_acv = true;
      if (extraCVarsData[i])
	{ need_sm_c = true; break; }
    }
  }
  for (i=0; i<num_var_map_1d; i++) {
    if (!num_var_map_2d || secondaryADVarMapTargets[i] == NO_TARGET) {
      need_sm_adv = true;
      if (extraDVarsData[i])
	{ need_sm_d = true; break; }
    }
  }

  RealVector  submodel_a_c_vars,   submodel_a_c_l_bnds, submodel_a_c_u_bnds;
  IntVector   submodel_a_d_vars,   submodel_a_d_l_bnds, submodel_a_d_u_bnds;
  StringArray submodel_a_c_labels, submodel_a_d_labels;
  if (need_sm_c || need_sm_acv)
    submodel_a_c_vars = subModel.all_continuous_variables();
  if (need_sm_c) {
    submodel_a_c_l_bnds = subModel.all_continuous_lower_bounds();
    submodel_a_c_u_bnds = subModel.all_continuous_upper_bounds();
    if (!nestedModelEvals)
      submodel_a_c_labels = subModel.all_continuous_variable_labels();
  }
  if (need_sm_d || need_sm_adv)
    submodel_a_d_vars = subModel.all_discrete_variables();
  if (need_sm_d) {
    submodel_a_d_l_bnds = subModel.all_discrete_lower_bounds();
    submodel_a_d_u_bnds = subModel.all_discrete_upper_bounds();
    if (!nestedModelEvals)
      submodel_a_d_labels = subModel.all_discrete_variable_labels();
  }

  RealVector acv_lower_bnds, acv_upper_bnds;
  RealDenseVector n_means, n_std_devs, n_lower_bnds, n_upper_bnds,
    ln_means, ln_std_devs, ln_err_facts, ln_lower_bnds, ln_upper_bnds,
    u_lower_bnds, u_upper_bnds, lu_lower_bnds, lu_upper_bnds,
    t_modes, t_lower_bnds, t_upper_bnds, e_betas,
    b_alphas, b_betas, b_lower_bnds, b_upper_bnds, ga_alphas, ga_betas,
    gu_alphas, gu_betas, f_alphas, f_betas, w_alphas, w_betas;
  if (num_var_map_2c) {
    const StringArray& acv_types = subModel.all_continuous_variable_types();
    num_cdv  = acv_types.count("continuous_design");
    num_nuv  = subModel.normal_means().length();
    num_lnuv = subModel.lognormal_means().length();
    num_uuv  = subModel.uniform_lower_bounds().length();
    num_luuv = subModel.loguniform_lower_bounds().length();
    num_tuv  = subModel.triangular_modes().length();
    num_euv  = subModel.exponential_betas().length();
    num_buv  = subModel.beta_alphas().length();
    num_gauv = subModel.gamma_alphas().length();
    num_guuv = subModel.gumbel_alphas().length();
    num_fuv  = subModel.frechet_alphas().length();
    num_wuv  = subModel.weibull_alphas().length();

    if (secondaryACVarMapTargets.contains(CDV_LWR_BND) ||
	secondaryACVarMapTargets.contains(CSV_LWR_BND))
      acv_lower_bnds = subModel.all_continuous_lower_bounds();
    if (secondaryACVarMapTargets.contains(CDV_UPR_BND) ||
	secondaryACVarMapTargets.contains(CSV_UPR_BND))
      acv_upper_bnds = subModel.all_continuous_upper_bounds();
    if (secondaryACVarMapTargets.contains(N_MEAN))
      n_means = subModel.normal_means();
    if (secondaryACVarMapTargets.contains(N_STD_DEV))
      n_std_devs = subModel.normal_std_deviations();
    if (secondaryACVarMapTargets.contains(N_LWR_BND))
      n_lower_bnds = subModel.normal_lower_bounds();
    if (secondaryACVarMapTargets.contains(N_UPR_BND))
      n_upper_bnds = subModel.normal_upper_bounds();
    if (secondaryACVarMapTargets.contains(LN_MEAN))
      ln_means = subModel.lognormal_means();
    if (secondaryACVarMapTargets.contains(LN_STD_DEV))
      ln_std_devs = subModel.lognormal_std_deviations();
    if (secondaryACVarMapTargets.contains(LN_ERR_FACT))
      ln_err_facts = subModel.lognormal_error_factors();
    if (secondaryACVarMapTargets.contains(LN_LWR_BND))
      ln_lower_bnds = subModel.lognormal_lower_bounds();
    if (secondaryACVarMapTargets.contains(LN_UPR_BND))
      ln_upper_bnds = subModel.lognormal_upper_bounds();
    if (secondaryACVarMapTargets.contains(U_LWR_BND))
      u_lower_bnds = subModel.uniform_lower_bounds();
    if (secondaryACVarMapTargets.contains(U_UPR_BND))
      u_upper_bnds = subModel.uniform_upper_bounds();
    if (secondaryACVarMapTargets.contains(LU_LWR_BND))
      lu_lower_bnds = subModel.loguniform_lower_bounds();
    if (secondaryACVarMapTargets.contains(LU_UPR_BND))
      lu_upper_bnds = subModel.loguniform_upper_bounds();
    if (secondaryACVarMapTargets.contains(T_MODE))
      t_modes = subModel.triangular_modes();
    if (secondaryACVarMapTargets.contains(T_LWR_BND))
      t_lower_bnds = subModel.triangular_lower_bounds();
    if (secondaryACVarMapTargets.contains(T_UPR_BND))
      t_upper_bnds = subModel.triangular_upper_bounds();
    if (secondaryACVarMapTargets.contains(E_BETA))
      e_betas = subModel.exponential_betas();
    if (secondaryACVarMapTargets.contains(B_ALPHA))
      b_alphas = subModel.beta_alphas();
    if (secondaryACVarMapTargets.contains(B_BETA))
      b_betas = subModel.beta_betas();
    if (secondaryACVarMapTargets.contains(B_LWR_BND))
      b_lower_bnds = subModel.beta_lower_bounds();
    if (secondaryACVarMapTargets.contains(B_UPR_BND))
      b_upper_bnds = subModel.beta_upper_bounds();
    if (secondaryACVarMapTargets.contains(GA_ALPHA))
      ga_alphas = subModel.gamma_alphas();
    if (secondaryACVarMapTargets.contains(GA_BETA))
      ga_betas = subModel.gamma_betas();
    if (secondaryACVarMapTargets.contains(GU_ALPHA))
      gu_alphas = subModel.gumbel_alphas();
    if (secondaryACVarMapTargets.contains(GU_BETA))
      gu_betas = subModel.gumbel_betas();
    if (secondaryACVarMapTargets.contains(F_ALPHA))
      f_alphas = subModel.frechet_alphas();
    if (secondaryACVarMapTargets.contains(F_BETA))
      f_betas = subModel.frechet_betas();
    if (secondaryACVarMapTargets.contains(W_ALPHA))
      w_alphas = subModel.weibull_alphas();
    if (secondaryACVarMapTargets.contains(W_BETA))
      w_betas = subModel.weibull_betas();
  }
  IntVector adv_lower_bnds, adv_upper_bnds;
  if (num_var_map_2d) {
    if (secondaryADVarMapTargets.contains(DDV_LWR_BND) ||
	secondaryADVarMapTargets.contains(DSV_LWR_BND))
      adv_lower_bnds = subModel.all_discrete_lower_bounds();
    if (secondaryADVarMapTargets.contains(DDV_UPR_BND) ||
	secondaryADVarMapTargets.contains(DSV_UPR_BND))
      adv_upper_bnds = subModel.all_discrete_upper_bounds();
  }

  const RealVector& curr_c_vars = currentVariables.continuous_variables();
  const RealVector& curr_c_l_bnds
    = userDefinedConstraints.continuous_lower_bounds();
  const RealVector& curr_c_u_bnds
    = userDefinedConstraints.continuous_upper_bounds();
  const StringArray& curr_c_labels
    = currentVariables.continuous_variable_labels();
  for (i=0; i<num_var_map_1c; i++) {
    size_t pcvm_index = primaryACVarMapIndices[i];
    short scvm_target
      = (num_var_map_2c) ? secondaryACVarMapTargets[i] : NO_TARGET;
    if (scvm_target == NO_TARGET) {
      submodel_a_c_vars[pcvm_index] = curr_c_vars[i];
      if (extraCVarsData[i]) {
	submodel_a_c_l_bnds[pcvm_index] = curr_c_l_bnds[i];
	submodel_a_c_u_bnds[pcvm_index] = curr_c_u_bnds[i];
	if (!nestedModelEvals)
	  submodel_a_c_labels[pcvm_index] = curr_c_labels[i];
      }
    }
    else {
      size_t dist_index;
      switch (scvm_target) {
      case CDV_LWR_BND: case CSV_LWR_BND:
	acv_lower_bnds[pcvm_index] = curr_c_vars[i]; break;
      case CDV_UPR_BND: case CSV_UPR_BND:
	acv_upper_bnds[pcvm_index] = curr_c_vars[i]; break;
      case N_MEAN:
	dist_index = pcvm_index - num_cdv;
	n_means[dist_index] = curr_c_vars[i]; break;
      case N_STD_DEV:
	dist_index = pcvm_index - num_cdv;
	n_std_devs[dist_index] = curr_c_vars[i]; break;
      case N_LWR_BND:
	dist_index = pcvm_index - num_cdv;
	n_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case N_UPR_BND:
	dist_index = pcvm_index - num_cdv;
	n_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case LN_MEAN:
	dist_index = pcvm_index - num_cdv - num_nuv;
	ln_means[dist_index] = curr_c_vars[i]; break;
      case LN_STD_DEV:
	dist_index = pcvm_index - num_cdv - num_nuv;
	ln_std_devs[dist_index] = curr_c_vars[i]; break;
      case LN_ERR_FACT:
	dist_index = pcvm_index - num_cdv - num_nuv;
	ln_err_facts[dist_index] = curr_c_vars[i]; break;
      case LN_LWR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv;
	ln_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case LN_UPR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv;
	ln_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case U_LWR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv;
	u_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case U_UPR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv;
	u_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case LU_LWR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv;
	lu_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case LU_UPR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv;
	lu_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case T_MODE:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv;
	t_modes[dist_index] = curr_c_vars[i]; break;
      case T_LWR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv;
	t_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case T_UPR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv;
	t_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case E_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv;
	e_betas[dist_index] = curr_c_vars[i]; break;
      case B_ALPHA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv;
	b_alphas[dist_index] = curr_c_vars[i]; break;
      case B_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv;
	b_betas[dist_index] = curr_c_vars[i]; break;
      case B_LWR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv;
	b_lower_bnds[dist_index] = curr_c_vars[i]; break;
      case B_UPR_BND:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv;
	b_upper_bnds[dist_index] = curr_c_vars[i]; break;
      case GA_ALPHA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv;
	ga_alphas[dist_index] = curr_c_vars[i]; break;
      case GA_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv;
	ga_betas[dist_index] = curr_c_vars[i]; break;
      case GU_ALPHA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv;
	gu_alphas[dist_index] = curr_c_vars[i]; break;
      case GU_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv;
	gu_betas[dist_index] = curr_c_vars[i]; break;
      case F_ALPHA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv - num_guuv;
	f_alphas[dist_index] = curr_c_vars[i]; break;
      case F_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv - num_guuv;
	f_betas[dist_index] = curr_c_vars[i]; break;
      case W_ALPHA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv - num_guuv
	  - num_fuv;
	w_alphas[dist_index] = curr_c_vars[i]; break;
      case W_BETA:
	dist_index = pcvm_index - num_cdv - num_nuv - num_lnuv - num_uuv
	  - num_luuv - num_tuv - num_euv - num_buv - num_gauv - num_guuv
	  - num_fuv;
	w_betas[dist_index] = curr_c_vars[i]; break;
      }
    }
  }

  const IntVector& curr_d_vars = currentVariables.discrete_variables();
  const IntVector& curr_d_l_bnds
    = userDefinedConstraints.discrete_lower_bounds();
  const IntVector& curr_d_u_bnds
    = userDefinedConstraints.discrete_upper_bounds();
  const StringArray& curr_d_labels
    = currentVariables.discrete_variable_labels();
  for (i=0; i<num_var_map_1d; i++) {
    size_t pdvm_index = primaryADVarMapIndices[i];
    short sdvm_target
      = (num_var_map_2d) ? secondaryADVarMapTargets[i] : NO_TARGET;
    if (sdvm_target == NO_TARGET) {
      submodel_a_d_vars[pdvm_index] = curr_d_vars[i];
      if (extraDVarsData[i]) {
	submodel_a_d_l_bnds[pdvm_index] = curr_d_l_bnds[i];
	submodel_a_d_u_bnds[pdvm_index] = curr_d_u_bnds[i];
	if (!nestedModelEvals)
	  submodel_a_d_labels[pdvm_index] = curr_d_labels[i];
      }
    }
    else {
      switch (sdvm_target) {
      case DDV_LWR_BND: case DSV_LWR_BND:
	adv_lower_bnds[pdvm_index] = curr_d_vars[i]; break;
      case DDV_UPR_BND: case DSV_UPR_BND:
	adv_upper_bnds[pdvm_index] = curr_d_vars[i]; break;
      }
    }
  }

  if (need_sm_c || need_sm_acv)
    subModel.all_continuous_variables(submodel_a_c_vars);
  if (need_sm_c) {
    subModel.all_continuous_lower_bounds(submodel_a_c_l_bnds);
    subModel.all_continuous_upper_bounds(submodel_a_c_u_bnds);
    if (!nestedModelEvals)
      subModel.all_continuous_variable_labels(submodel_a_c_labels);
  }
  if (need_sm_d || need_sm_adv)
    subModel.all_discrete_variables(submodel_a_d_vars);
  if (need_sm_d) {
    subModel.all_discrete_lower_bounds(submodel_a_d_l_bnds);
    subModel.all_discrete_upper_bounds(submodel_a_d_u_bnds);
    if (!nestedModelEvals)
      subModel.all_discrete_variable_labels(submodel_a_d_labels);
  }
  if (num_var_map_2c) {
    if (!acv_lower_bnds.empty())
      subModel.all_continuous_lower_bounds(acv_lower_bnds);
    if (!acv_upper_bnds.empty())
      subModel.all_continuous_upper_bounds(acv_upper_bnds);
    if (!n_means.empty())
      subModel.normal_means(n_means);
    if (!n_std_devs.empty())
      subModel.normal_std_deviations(n_std_devs);
    if (!n_lower_bnds.empty())
      subModel.normal_lower_bounds(n_lower_bnds);
    if (!n_upper_bnds.empty())
      subModel.normal_upper_bounds(n_upper_bnds);
    if (!ln_means.empty())
      subModel.lognormal_means(ln_means);
    if (!ln_std_devs.empty())
      subModel.lognormal_std_deviations(ln_std_devs);
    if (!ln_err_facts.empty())
      subModel.lognormal_error_factors(ln_err_facts);
    if (!ln_lower_bnds.empty())
      subModel.lognormal_lower_bounds(ln_lower_bnds);
    if (!ln_upper_bnds.empty())
      subModel.lognormal_upper_bounds(ln_upper_bnds);
    if (!u_lower_bnds.empty())
      subModel.uniform_lower_bounds(u_lower_bnds);
    if (!u_upper_bnds.empty())
      subModel.uniform_upper_bounds(u_upper_bnds);
    if (!lu_lower_bnds.empty())
      subModel.loguniform_lower_bounds(lu_lower_bnds);
    if (!lu_upper_bnds.empty())
      subModel.loguniform_upper_bounds(lu_upper_bnds);
    if (!t_modes.empty())
      subModel.triangular_modes(t_modes);
    if (!t_lower_bnds.empty())
      subModel.triangular_lower_bounds(t_lower_bnds);
    if (!t_upper_bnds.empty())
      subModel.triangular_upper_bounds(t_upper_bnds);
    if (!e_betas.empty())
      subModel.exponential_betas(e_betas);
    if (!b_alphas.empty())
      subModel.beta_alphas(b_alphas);
    if (!b_betas.empty())
      subModel.beta_betas(b_betas);
    if (!b_lower_bnds.empty())
      subModel.beta_lower_bounds(b_lower_bnds);
    if (!b_upper_bnds.empty())
      subModel.beta_upper_bounds(b_upper_bnds);
    if (!ga_alphas.empty())
      subModel.gamma_alphas(ga_alphas);
    if (!ga_betas.empty())
      subModel.gamma_betas(ga_betas);
    if (!gu_alphas.empty())
      subModel.gumbel_alphas(gu_alphas);
    if (!gu_betas.empty())
      subModel.gumbel_betas(gu_betas);
    if (!f_alphas.empty())
      subModel.frechet_alphas(f_alphas);
    if (!f_betas.empty())
      subModel.frechet_betas(f_betas);
    if (!w_alphas.empty())
      subModel.weibull_alphas(w_alphas);
    if (!w_betas.empty())
      subModel.weibull_betas(w_betas);
  }
  if (num_var_map_2d) {
    if (!adv_lower_bnds.empty())
      subModel.all_discrete_lower_bounds(adv_lower_bnds);
    if (!adv_upper_bnds.empty())
      subModel.all_discrete_upper_bounds(adv_upper_bnds);
  }
}

} // namespace Dakota
