/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        Approximation
//- Description:  Class implementation of base class for approximations
//-               
//- Owner:        Mike Eldred, Sandia National Laboratories

#include "DakotaApproximation.H"
#include "ProblemDescDB.H"
#include "DakotaResponse.H"
#include "TaylorApproximation.H"
#include "TANA3Approximation.H"
#include "OrthogPolyApproximation.H"
#include "InterpPolyApproximation.H"
#include "GaussProcApproximation.H"
#ifdef DAKOTA_SURFPACK
#include "SurfpackApproximation.H"
#endif // DAKOTA_SURFPACK
#include "DakotaGraphics.H"


namespace Dakota {

/** This constructor is the one which must build the base class data
    for all derived classes.  get_approx() instantiates a derived
    class letter and the derived constructor selects this base class
    constructor in its initialization list (to avoid recursion in the
    base class constructor calling get_approx() again).  Since the
    letter IS the representation, its rep pointer is set to NULL (an
    uninitialized pointer causes problems in ~Approximation). */
Approximation::Approximation(BaseConstructor, const ProblemDescDB& problem_db,
			     const size_t& num_vars):
  // See base constructor in DakotaIterator.C for full discussion of output
  // verbosity.  For approximations, verbose adds quad poly coeff reporting.
  outputLevel(problem_db.get_short("method.output")),
  numVars(num_vars), approxType(problem_db.get_string("model.surrogate.type")),
  approxOrder(-1), approxRep(NULL), referenceCount(1)
{
  if (approxType.begins("global_")) {
    useGradsFlag = problem_db.get_bool("model.surrogate.gradient_usage");
    if (useGradsFlag)
      Cerr << "\nWarning: use_gradients is not yet supported by current global "
	   << "approximation methods.\n\n";
  }
  else // local, multipoint
    useGradsFlag = true;

#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::Approximation(BaseConstructor) called to build base "
       << "class for letter." << endl;
#endif
}


/** The default constructor is used in Array<Approximation> instantiations
    and by the alternate envelope constructor.  approxRep is NULL in this
    case (problem_db is needed to build a meaningful Approximation object).
    This makes it necessary to check for NULL in the copy constructor,
    assignment operator, and destructor. */
Approximation::Approximation(): useGradsFlag(false), outputLevel(NORMAL_OUTPUT),
  approxOrder(-1), approxRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::Approximation() called to build empty approximation "
       << "object." << endl;
#endif
}


/** Envelope constructor only needs to extract enough data to properly
    execute get_approx, since Approximation(BaseConstructor, problem_db)
    builds the actual base class data for the derived approximations. */
Approximation::
Approximation(ProblemDescDB& problem_db, const size_t& num_vars):
  referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::Approximation(ProblemDescDB&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived type
  approxRep = get_approx(problem_db, num_vars);
  if ( !approxRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Used only by the envelope constructor to initialize approxRep to the 
    appropriate derived type. */
Approximation* Approximation::
get_approx(ProblemDescDB& problem_db, const size_t& num_vars)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_approx(ProblemDescDB&)." <<endl;
#endif

  const String& approx_type = problem_db.get_string("model.surrogate.type");
  if (approx_type == "local_taylor")
    return new TaylorApproximation(problem_db, num_vars);
  else if (approx_type == "multipoint_tana")
    return new TANA3Approximation(problem_db, num_vars);
  else if (approx_type == "global_orthogonal_polynomial")
    return new OrthogPolyApproximation(problem_db, num_vars);
  else if (approx_type == "global_interpolation_polynomial")
    return new InterpPolyApproximation(problem_db, num_vars);
  else if (approx_type == "global_gaussian")
    return new GaussProcApproximation(problem_db, num_vars);
#ifdef DAKOTA_SURFPACK
  else if (approx_type == "global_polynomial"     ||
	   approx_type == "global_kriging"        ||
	   approx_type == "global_neural_network" || // TO DO: Two ANN's ?
	   approx_type == "global_radial_basis"   ||
	   approx_type == "global_mars"           ||
	   approx_type == "global_moving_least_squares")
    return new SurfpackApproximation(problem_db, num_vars);
#endif // DAKOTA_SURFPACK
  else {
    Cerr << "Error: Approximation type " << approx_type << " not available."
	 << endl;
    return NULL;
  }
}


/** This is the alternate envelope constructor for instantiations on
    the fly.  Since it does not have access to problem_db, the letter
    class is not fully populated.  This constructor executes
    get_approx(type), which invokes the default constructor of the
    derived letter class, which in turn invokes the default
    constructor of the base class. */
Approximation::Approximation(const String& approx_type, short approx_order,
			     const size_t& num_vars):
  referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::Approximation(String&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate derived type
  approxRep = get_approx(approx_type, approx_order, num_vars);
  if ( !approxRep ) // bad type or insufficient memory
    abort_handler(-1);
}


/** Used only by the envelope constructor to initialize approxRep to the 
    appropriate derived type. */
Approximation* Approximation::
get_approx(const String& approx_type, short approx_order, 
	   const size_t& num_vars)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter in get_approx(String&)." << endl;
#endif

  Approximation* approx;
  if (approx_type == "local_taylor") {
    approx = new TaylorApproximation();
    approx->useGradsFlag = true;
  }
  else if (approx_type == "multipoint_tana") {
    approx = new TANA3Approximation();
    approx->useGradsFlag = true;
  }
  else if (approx_type == "global_orthogonal_polynomial")
    approx = new OrthogPolyApproximation();
  else if (approx_type == "global_interpolation_polynomial")
    approx = new InterpPolyApproximation();
  else if (approx_type == "global_gaussian")
    approx = new GaussProcApproximation();
#ifdef DAKOTA_SURFPACK
  else if (approx_type == "global_polynomial"     ||
	   approx_type == "global_kriging"        ||
	   approx_type == "global_neural_network" || // TO DO: Two ANN's ?
	   approx_type == "global_radial_basis"   ||
	   approx_type == "global_mars"           ||
	   approx_type == "global_moving_least_squares")
    approx = new SurfpackApproximation();
#endif // DAKOTA_SURFPACK
  else {
    Cerr << "Error: Approximation type " << approx_type << " not available."
	 << endl;
    approx = NULL;
  }
  // Note: this makes numVars and approxOrder available at run time,
  // but they were not available at construct time.
  if (approx) {
    approx->numVars     = num_vars;
    approx->approxOrder = approx_order;
  }
  return approx;
}


/** Copy constructor manages sharing of approxRep and incrementing
    of referenceCount. */
Approximation::Approximation(const Approximation& approx)
{
  // Increment new (no old to decrement)
  approxRep = approx.approxRep;
  if (approxRep) // Check for an assignment of NULL
    approxRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::Approximation(Approximation&)" << endl;
  if (approxRep)
    Cout << "approxRep referenceCount = " << approxRep->referenceCount << endl;
#endif
}


/** Assignment operator decrements referenceCount for old approxRep, assigns
    new approxRep, and increments referenceCount for new approxRep. */
Approximation Approximation::operator=(const Approximation& approx)
{
  if (approxRep != approx.approxRep) { // normal case: old != new
    // Decrement old
    if (approxRep) // Check for NULL
      if ( --approxRep->referenceCount == 0 ) 
	delete approxRep;
    // Assign and increment new
    approxRep = approx.approxRep;
    if (approxRep) // Check for NULL
      approxRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "Approximation::operator=(Approximation&)" << endl;
  if (approxRep)
    Cout << "approxRep referenceCount = " << approxRep->referenceCount << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes approxRep
    when referenceCount reaches zero. */
Approximation::~Approximation()
{ 
  // Check for NULL pointer 
  if (approxRep) {
    --approxRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "approxRep referenceCount decremented to " 
	 << approxRep->referenceCount << endl;
#endif
    if (approxRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting approxRep" << endl;
#endif
      delete approxRep;
    }
  }
}


const Real& Approximation::get_value(const RealVector& x)
{
  if (!approxRep) {
    Cerr << "Error: get_value() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }

  return approxRep->get_value(x);
}


const RealBaseVector& Approximation::get_gradient(const RealVector& x)
{
  if (!approxRep) {
    Cerr << "Error: get_gradient() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }

  return approxRep->get_gradient(x);
}


const RealMatrix& Approximation::get_hessian(const RealVector& x)
{
  if (!approxRep) {
    Cerr << "Error: get_hessian() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }
    
  return approxRep->get_hessian(x);
}

const Real& Approximation::get_variance(const RealVector& x)
{
  if (!approxRep) {
    Cerr << "Error: get_variance() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }

  return approxRep->get_variance(x);
}

const bool Approximation::diagnostics_available()
{
  if (approxRep) // envelope fwd to letter
    return approxRep->diagnostics_available();
  else // default for letter lacking virtual fn redefinition
    return false;
}

const Real& Approximation::get_diagnostic(const String& metric_type)
{
  //if (!approxRep) {
  //  Cerr << "Error: get_diagnostic() not available for this" 
  //	 << " approximation type."
  //	 << endl;
  // return 0.;
  //}

  return approxRep->get_diagnostic(metric_type);
}


const RealVector& Approximation::approximation_coefficients() const
{
  if (!approxRep) {
    Cerr << "Error: approximation_coefficients() not available for this "
	 << "approximation type." << endl;
    abort_handler(-1);
  }
   
  return approxRep->approximation_coefficients();
}


void Approximation::approximation_coefficients(const RealVector& approx_coeffs)
{
  if (approxRep)
    approxRep->approximation_coefficients(approx_coeffs);
  else {
    Cerr << "Error: approximation_coefficients() not available for this "
	 << "approximation type." << endl;
    abort_handler(-1);
  }
}


void Approximation::print_coefficients(ostream& s) const
{
  if (approxRep)
    approxRep->print_coefficients(s);
  else {
    Cerr << "Error: print_coefficients() not available for this approximation "
	 << "type." << endl;
    abort_handler(-1);
  }
}


int Approximation::min_coefficients() const
{
  if (!approxRep) { // no default implementation
    Cerr << "Error: min_coefficients() not defined for this approximation type."
         << endl;
    abort_handler(-1);
  }
   
  return approxRep->min_coefficients(); // fwd to letter
}


int Approximation::recommended_coefficients() const
{
  // derived class may override, else return minimum coefficients
  if (approxRep)
    return approxRep->recommended_coefficients(); // fwd to letter
  else
    return min_coefficients();
}


int Approximation::num_constraints() const
{
  if (approxRep) // fwd to letter
    return approxRep->num_constraints(); 
  else { // default implementation
    if (anchorPoint.is_null())
      return 0;
    else {
      int ng = anchorPoint.response_gradient().length(),
          nh = anchorPoint.response_hessian().num_rows();
      return 1 + ng + nh*(nh + 1)/2;
    }
  }
}


int Approximation::min_samples(bool constraint_flag) const
{
  if (approxRep)
    return approxRep->min_samples(constraint_flag); 
  else { // not virtual: implementation for all derived classes
    int coeffs = min_coefficients();
    if (constraint_flag)
      coeffs -= num_constraints();
    if (useGradsFlag)
      return (int)ceil((Real)coeffs/(Real)(numVars+1));
    else
      return coeffs;
  }
}


int Approximation::recommended_samples(bool constraint_flag) const
{
  if (approxRep)
    return approxRep->recommended_samples(constraint_flag); 
  else { // not virtual: implementation for all derived classes
    int coeffs = recommended_coefficients();
    if (constraint_flag)
      coeffs -= num_constraints();
    if (useGradsFlag)
      return (int)ceil((Real)coeffs/(Real)(numVars+1));
    else
      return coeffs;
  }
}


void Approximation::find_coefficients()
{
  if (approxRep)
    approxRep->find_coefficients(); 
  else {
    Cerr << "Error: find_coefficients() not available for this approximation "
	 << "type." << endl;
    abort_handler(-1);
  }
}


void Approximation::
update(const VariablesArray& vars_array, const ResponseArray& resp_array,
       const int& fn_index)
{
  if (approxRep)
    approxRep->update(vars_array, resp_array, fn_index);
  else { // not virtual: all derived classes use following definition:
    currentPoints.clear(); // replace currentPoints with incoming samples
    append(vars_array, resp_array, fn_index);
  }
}


void Approximation::
append(const VariablesArray& vars_array, const ResponseArray& resp_array,
       const int& fn_index)
{
  if (approxRep)
    approxRep->append(vars_array, resp_array, fn_index);
  else { // not virtual: all derived classes use following definition:

    size_t i, num_points = vars_array.length();
    if ( resp_array.length() != num_points ) {
      Cerr << "Error: mismatch in sample lengths in Approximation::append()."
	   << endl;
      abort_handler(-1);
    }

    // append to currentPoints with incoming samples
    for (i=0; i<num_points; i++)
      add(vars_array[i], resp_array[i], fn_index, false);
  }
}


void Approximation::build()
{
  if (approxRep)
    approxRep->build();
  else { // not virtual: all derived classes use following definition:

    // all update()'s have been posted, now compute the data fit

    size_t num_curr_pts = currentPoints.entries();
    int ms = min_samples(true); // account for anchorPoint & use_gradients
    if (num_curr_pts < ms) {
      Cerr << "\nError: not enough samples to build approximation.  "
	   << "Construction of this approximation\n       requires at least "
	   << ms << " samples for " << numVars << " variables.  Only "
	   << num_curr_pts << " samples were provided." << endl;
      abort_handler(-1);
    }

    // compute coefficients
    find_coefficients();
  }
}


void Approximation::
add(const Variables& vars, const Response& response, const int& fn_index,
    bool anchor_flag)
{
  // private fn used only by build() and update() functions -> approxRep
  // forward not needed.  Recomputing coefficients, if needed, is managed
  // by the calling functions.

  const ShortArray& asv = response.active_set_request_vector();
  short asv_val = asv[fn_index];
  if (asv_val) {
    const RealVector&      fn_vals     = response.function_values();
    const RealMatrix&      fn_grads    = response.function_gradients();
    const RealMatrixArray& fn_hessians = response.function_hessians();
    size_t num_fns = asv.length();
    if ( vars.cv() != numVars ||
	 ( (asv_val & 1) && fn_vals.length()            != num_fns ) ||
	 ( (asv_val & 2) && fn_grads.num_rows()         != num_fns ) || 
			  //fn_grads[fn_index].length() != numVars)) ||
	 ( (asv_val & 4) && fn_hessians.length()        != num_fns ) ) {
			  //fn_hessians[fn_index].num_rows()    != numVars ||
			  //fn_hessians[fn_index].num_columns() != numVars))) {
      Cerr << "Error: array size mismatch in Approximation::add()." << endl;
      abort_handler(-1);
    }

    // Since SurrogateDataPoint does not contain an asv, assure the
    // use of empty vectors/matrices if data is not active.
    Real           empty_r = 0.;
    RealBaseVector empty_grad;
    RealMatrix     empty_hess;
    const Real&           fn_val  = (asv_val & 1) ? fn_vals[fn_index] : empty_r;
    const RealBaseVector& fn_grad = (asv_val & 2) ? fn_grads[fn_index]
                                                  : empty_grad;
    const RealMatrix&     fn_hess = (asv_val & 4) ? fn_hessians[fn_index]
                                                  : empty_hess;
    if (anchor_flag)
      add_anchor(vars.continuous_variables(), fn_val, fn_grad, fn_hess);
    else
      add_point(vars.continuous_variables(),  fn_val, fn_grad, fn_hess);
  }
  // else no data point added for this index
}


void Approximation::draw_surface()
{
  if (approxRep)
    approxRep->draw_surface();
  else { // draw surface of 2 independent variables

    if (approxLowerBounds.length() != 2 ||
	approxUpperBounds.length() != 2 || numVars != 2)
      return;

    // define number of equally spaced points in each direction
    int i, j, num_intervals = 50, num_axis_pts = num_intervals + 1;
    Real dx = (approxUpperBounds[0] - approxLowerBounds[0])/num_intervals,
         dy = (approxUpperBounds[1] - approxLowerBounds[1])/num_intervals;
    RealMatrix F(num_axis_pts, num_axis_pts);
    RealVector X(num_axis_pts), Y(num_axis_pts);
  
    // fill X & Y with grid of values to be displayed in graphics
    X[0] = approxLowerBounds[0];
    Y[0] = approxLowerBounds[1];
    for (i=1; i<num_axis_pts; i++) {
      X[i] = X[i-1] + dx;
      Y[i] = Y[i-1] + dy;
    }

    // evaluate approximate response values at all grid intersections to
    // populate F matrix
    RealVector xtemp(2);
    for (i=0; i<num_axis_pts; i++) {
      for (j=0; j<num_axis_pts; j++) {  
        xtemp[0] = X[i];
        xtemp[1] = Y[j];
        F[i][j]  = get_value(xtemp);
      }
    }
    extern Graphics dakota_graphics; // defined in ParallelLibrary.C
    dakota_graphics.show_data_3d(X, Y, F);
  }
}

} // namespace Dakota
