/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

/// This file registers Coliny solvers defined in header files.  It also
/// references data for other Coliny solvers that are registered.

#include <algorithm>
#include <coliny/Factory.h>
#include <coliny/MultiStart.h>
#include <coliny/PatternSearch.h>
#include <coliny/SolisWets.h>
#include <coliny/Cobyla2.h>
#include <coliny/EAminlp.h>

/* REGISTERING SOLVERS IN COLINY HEADERS */

typedef colin::OptSolver<utilib::BasicArray<double>,colin::AppResponse_Utilib> base_t;
typedef colin::OptSolver<utilib::MixedIntVars,colin::AppResponse_Utilib> mi_base_t;
typedef colin::OptSolver<std::vector<double> > std_base_t;

FACTORY_REGISTER(ms, mi_base_t*, return new coliny::MultiStart<utilib::MixedIntVars>();, "An alias to MultiStart")
FACTORY_REGISTER(MultiStart, mi_base_t*, return new coliny::MultiStart<utilib::MixedIntVars>();, "A simple random sampler")
FACTORY_REGISTER(multistart, mi_base_t*, return new coliny::MultiStart<utilib::MixedIntVars>();, "An alias to MultiStart")

FACTORY_NAMED_REGISTER(ms_sw, "ms-sw", mi_base_t*, coliny::MultiStart<utilib::MixedIntVars>* opt = new coliny::MultiStart<utilib::MixedIntVars>(); coliny::SolisWets* local = new coliny::SolisWets; opt->set_solver(local,true); return opt;, "Multi-start Solis-Wets")
FACTORY_NAMED_REGISTER(ms_ps, "ms-ps", mi_base_t*, coliny::MultiStart<utilib::MixedIntVars>* opt = new coliny::MultiStart<utilib::MixedIntVars>(); coliny::PatternSearch* local = new coliny::PatternSearch; opt->set_solver(local,true); return opt;, "Multi-start pattern search")
#ifdef ACRO_USING_COBYLA
FACTORY_NAMED_REGISTER(ms_cobyla, "ms-cobyla", mi_base_t*, coliny::MultiStart<utilib::MixedIntVars>* opt = new coliny::MultiStart<utilib::MixedIntVars>(); coliny::Cobyla* local = new coliny::Cobyla; opt->set_solver(local,true); return opt;, "Multi-start Cobyla")
#endif
#ifdef ACRO_USING_APPS
FACTORY_NAMED_REGISTER(ms_apps, "ms-apps", mi_base_t*, coliny::MultiStart<utilib::MixedIntVars>* opt = new coliny::MultiStart<utilib::MixedIntVars>(); coliny::APPS* local = new coliny::APPS; opt->set_solver(local,true); return opt;, "Multi-start APPSPACK")
#endif

FACTORY_REGISTER(EAminlp, mi_base_t*, return new coliny::EAminlp;, "An evolutionary algorithm")
FACTORY_REGISTER(ea, mi_base_t*, return new coliny::EAminlp;, "An alias to EAminlp")

FACTORY_NAMED_REGISTER(ea_sw, "ea-sw", mi_base_t*, coliny::EAminlp* opt = new coliny::EAminlp; coliny::SolisWets* local = new coliny::SolisWets; opt->set_solver(local,true); return opt;, "Hybrid EA with Solis-Wets local search")
FACTORY_NAMED_REGISTER(ea_ps, "ea-ps", mi_base_t*, coliny::EAminlp* opt = new coliny::EAminlp; coliny::PatternSearch* local = new coliny::PatternSearch; opt->set_solver(local,true); return opt;, "Hybrid EA with PatternSearch local search")
#ifdef ACRO_USING_COBYLA
FACTORY_NAMED_REGISTER(ea_cobyla, "ea-cobyla", mi_base_t*, coliny::EAminlp* opt = new coliny::EAminlp; coliny::Cobyla* local = new coliny::Cobyla; opt->set_solver(local,true); return opt;, "Hybrid EA with Cobyla local search")
#endif
#ifdef ACRO_USING_APPS
FACTORY_NAMED_REGISTER(ea_apps, "ea-apps", mi_base_t*, coliny::EAminlp* opt = new coliny::EAminlp; coliny::APPS* local = new coliny::APPS; opt->set_solver(local,true); return opt;, "Hybrid EA with APPSPACK local search")
#endif


namespace coliny {

colin::OptSolverBase* factory_create_base(const char* solver)
{
{
base_t* tmp = utilib::factory_create<base_t*>(solver);
if (tmp != 0) return tmp;
}
{
mi_base_t* tmp = utilib::factory_create<mi_base_t*>(solver);
if (tmp != 0) return tmp;
}
{
std_base_t* tmp = utilib::factory_create<std_base_t*>(solver);
if (tmp != 0) return tmp;
}
return 0;
}


void get_solver_info(std::vector<std::string>& names, std::vector<std::string>& descriptions)
{
std::vector<const char*> tmp1, tmp2;
int ctr=0;

tmp1 = utilib::global_factory<std_base_t*>().get_names();
tmp2 = utilib::global_factory<std_base_t*>().get_descriptions();
names.resize(names.size()+tmp1.size());
descriptions.resize(descriptions.size()+tmp1.size());
for (unsigned int i=0; i<tmp1.size(); i++) {
  names[ctr] = tmp1[i];
  descriptions[ctr] = tmp2[i];
  ctr++;
  }

tmp1 = utilib::global_factory<mi_base_t*>().get_names();
tmp2 = utilib::global_factory<mi_base_t*>().get_descriptions();
names.resize(names.size()+tmp1.size());
descriptions.resize(descriptions.size()+tmp1.size());
for (unsigned int i=0; i<tmp1.size(); i++) {
  names[ctr] = tmp1[i];
  descriptions[ctr] = tmp2[i];
  ctr++;
  }

tmp1 = utilib::global_factory<base_t*>().get_names();
tmp2 = utilib::global_factory<base_t*>().get_descriptions();
names.resize(names.size()+tmp1.size());
descriptions.resize(descriptions.size()+tmp1.size());
for (unsigned int i=0; i<tmp1.size(); i++) {
  names[ctr] = tmp1[i];
  descriptions[ctr] = tmp2[i];
  ctr++;
  }

}

void print_factory_solvers(std::ostream& os)
{
   std::vector<std::string> names;
   std::vector<std::string> descriptions;
   coliny::get_solver_info(names,descriptions);
   std::vector<int> ndx;
   utilib::order(ndx,names);
   if (ndx.size() == 0)
      os << "  solvers: None" << std::endl;
   else
      os << "  solvers: " << std::endl;
   for (unsigned i=0; i<ndx.size(); i++) {
     os << "    " << names[ndx[i]] << std::endl;
     os << "         " << descriptions[ndx[i]] << std::endl;
     }
}

}
