/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        LeastSq
//- Description:  Abstract base class to logically represent a variety
//-               of DAKOTA least squares objects in a generic fashion.
//- Owner:        Mike Eldred
//- Version: $Id: DakotaLeastSq.H 5143 2008-06-27 01:17:56Z mseldre $

#ifndef DAKOTA_LEAST_SQ_H
#define DAKOTA_LEAST_SQ_H

#include "DakotaMinimizer.H"


namespace Dakota {

/// Base class for the nonlinear least squares branch of the iterator hierarchy.

/** The LeastSq class provides common data and functionality for
    least squares solvers (including NL2OL, NLSSOLLeastSq, and SNLLLeastSq. */

class LeastSq: public Minimizer
{
public:

protected:

  //
  //- Heading: Constructors and destructor
  //

  LeastSq();                                   ///< default constructor
  LeastSq(Model& model);                       ///< standard constructor
  LeastSq(NoDBBaseConstructor, Model& model);  ///< alternate constructor
  ~LeastSq();                                  ///< destructor

  //
  //- Heading: Virtual member function redefinitions
  //

  void derived_pre_run();
  void run();
  void derived_post_run();
  void print_results(ostream& s);

  //
  //- Heading: New virtual member functions
  //

  /// Used within the least squares branch for minimizing the sum of
  /// squares residuals.  Redefines the run virtual function
  /// for the least squares branch.
  virtual void minimize_residuals() = 0;

  //
  //- Heading: Convenience/Helper functions
  //

  /// primary response conversion map for RecastModel used in scaling:
  /// transform least squares terms (fns, grads, Hessians) from native
  /// (user) to iterator space
  static void primary_resp_recast(const Variables& native_vars,
				  const Variables& scaled_vars,
				  const Response& native_response,
				  Response& scaled_response);

  /// read user data file to load observed data points
  void read_observed_data();

  /// Calculate confidence intervals on estimated parameters
  void get_confidence_intervals();

  //
  //- Heading: Data
  //

  int numLeastSqTerms; ///< number of least squares terms

  /// pointer to LeastSq instance used in static member functions
  static LeastSq* leastSqInstance;
  /// pointer containing previous value of leastSqInstance
  LeastSq* prevLSqInstance;

  // user-specified weights for weighted least squares
  //RealVector leastSqWeights;
  /// flag indicating whether weighted least squares is active
  bool weightFlag;
  /// filename from which to read observed data
  String obsDataFilename;
  /// flag indicating whether user-supplied data is active
  bool obsDataFlag;
  /// storage for user-supplied data for computing residuals
  RealVector obsData;
  /// lower bounds for confidence intervals on calibration parameters
  RealVector confBoundsLower;
  /// upper bounds for confidence intervals on calibration parameters
  RealVector confBoundsUpper;

private:

  //
  //- Heading: Data
  //
};


inline void LeastSq::run()
{ minimize_residuals(); }


inline LeastSq::LeastSq(): weightFlag(false), obsDataFlag(false)
{ }


inline LeastSq::~LeastSq()
{ 
  if (weightFlag || scaleFlag || obsDataFlag) 
    iteratedModel.free_communicators(maxConcurrency);
}

} // namespace Dakota

#endif
