/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class RandomBitMutator.

    NOTES:

        See notes of RandomBitMutator.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu Jun 12 10:12:12 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the RandomBitMutator class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <BitManipulator.hpp>
#include <GeneticAlgorithm.hpp>
#include <utilities/include/Math.hpp>
#include <Mutators/RandomBitMutator.hpp>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <utilities/include/RandomNumberGenerator.hpp>




/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Utilities;
using namespace JEGA::Logging;
using namespace eddy::utilities;







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {








/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/

const string&
RandomBitMutator::Name(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    static const string* ret = new string("bit_random");
    return *ret;
}

const string&
RandomBitMutator::Description(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    static const string ret(
        "This mutator performs mutation by randomly selecting a "
        "design variable and converting it to a binary "
        "representation, then randomly choosing a bit from that "
        "string and toggling it.\n\nThe binary representation is "
        "created using a BitManipulator.  The number of mutations is "
        "the rate times the size of the group passed in rounded "
        "to the nearest whole number."
        );
    return ret;
}

GeneticAlgorithmOperator*
RandomBitMutator::Create(
    GeneticAlgorithm& algorithm
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return new RandomBitMutator(algorithm);
}











/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/

string
RandomBitMutator::GetName(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return RandomBitMutator::Name();
}

string
RandomBitMutator::GetDescription(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return RandomBitMutator::Description();
}

GeneticAlgorithmOperator*
RandomBitMutator::Clone(
    GeneticAlgorithm& algorithm
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return new RandomBitMutator(*this, algorithm);
}

void
RandomBitMutator::Mutate(
    DesignGroup& group
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGALOG_II(GetLogger(), ldebug(), this,
        text_entry(ldebug(), GetName() + ": Performing mutation.")
        )

    // store the design target for repeated use
    DesignTarget& target = group.GetDesignTarget();
    EDDY_ASSERT(&target == &group.GetDesignTarget());

    // We need a BitManipulator to do the bit operations
    BitManipulator maniper(target);

    // Get the mutation rate.
    const double rate = GetRate();

    // get the number of design variables
    const size_t ndv = target.GetNDV();

    // determine the number of mutation operations;
    const size_t nmutate =
        static_cast<size_t>(Math::Round(rate * ndv * group.GetSize()));

    // if there will be no mutations, don't continue.
    if(nmutate < 1)
    {
        JEGALOG_II(GetLogger(), ldebug(), this,
            text_entry(ldebug(), GetName() + ": Rate and group size are such "
                       "that no mutation will occur.")
            )
        return;
    }

    // store the size of the group for re-use.
    size_t gsize = group.GetSize();

    // prepare some variables for use in the following loop.
    size_t dv;
    eddy::utilities::uint16_t bit;

    // The DesignGroup sort containers do not support random access so we will
    // copy the contents into a temporary container that does for speed.
    Design** temp = new Design*[gsize];
    std::copy(group.BeginDV(), group.EndDV(), temp);

    // Carry out the mutations
    for(size_t i=0; i<nmutate; ++i)
    {
        // pick a random design variable;
        dv = RandomNumberGenerator::UniformInt<size_t>(0, ndv-1);

        // pick a random bit to mutate
        bit = RandomNumberGenerator::UniformInt<eddy::utilities::uint16_t>(
                    0, maniper.GetNumberOfBits(dv)-1
                    );

        // store the design being changed.
        Design* chosen =
            temp[RandomNumberGenerator::UniformInt<size_t>(0, gsize-1)];

        // remove the design being changed
        group.Erase(chosen);

        // perform the mutation.

        // get an encoded version of the chosen Design variable
        eddy::utilities::int64_t dvval =
            maniper.ConvertToShiftedInt(chosen->GetVariableRep(dv), dv);

        // flip the chosen bit.
        dvval = maniper.ToggleBit(bit, dvval);

        // record the new variable value (after decoding it).
        chosen->SetVariableRep(dv, maniper.ConvertFromShiftedInt(dvval, dv));

        // now re-insert the design being changed
        group.Insert(chosen);
    }

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": Performed ") << nmutate
            << " total mutations."
        )

    // delete the temporary vector of Design*.
    delete [] temp;
}





/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/




RandomBitMutator::RandomBitMutator(
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmMutator(algorithm)
{
    EDDY_FUNC_DEBUGSCOPE
}

RandomBitMutator::RandomBitMutator(
    const RandomBitMutator& copy
    ) :
        GeneticAlgorithmMutator(copy)
{
    EDDY_FUNC_DEBUGSCOPE
}

RandomBitMutator::RandomBitMutator(
    const RandomBitMutator& copy,
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmMutator(copy, algorithm)
{
    EDDY_FUNC_DEBUGSCOPE
}









/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
