#!/bin/csh -f
#
# A csh script to make Dakota. It will:
#   1.  configure or (reconfigure / make clean Dakota/src) Dakota
#   2.  make Dakota
#   OPTION: reconfigure around any failed package builds (default: yes)
#
# --- external variables used ---
# $*          optional reconfig flags, then arguments for DAKOTA configure
# $DAKOTA     path to Dakota directory in which to build
#             (run_base_test.csh: $TEST_ROOT/test_$TEST_PLATFORM/Dakota)
# $TEST_SQA   script directory (passed to top-level scripts)
# $UNAME      output of uname (environment.csh)
# $BUILD_JOBS number of parallel jobs (environment.csh)
# $DAKOTA_CONFIG_OPT (optional variable): flags to pass to configure
# $DAKOTA_MAKE_OPT   (optional variable): flags to pass to make
# -------------------------------
#
#  Usage: make_script.csh [DAKOTA_RECONFIG] [NOPKG_RECONFIG] configure_options
#  !!! Only pass either of the first two arguments if overriding defaults
# ---------------------------------------------------------------------------

echo "Running $0 on `uname -n` at `date`"

# ------------------------------
# process command-line arguments
# ------------------------------

# default: no (this is not a reconfigure of entire DAKOTA)
if ( $#argv > 0 && $1 == 'DAKOTA_RECONFIG' ) then
  set dakota_reconfig = yes
  shift
else
  set dakota_reconfig = no
else

endif
# default: yes (reconfigure around any failed packages)
if ( $#argv > 0 && $1 == 'NOPKG_RECONFIG' ) then
  set pkg_reconfig = no
  shift
else
  set pkg_reconfig = yes
endif
# all trailing arguments are passed to configure
if ( ${?DAKOTA_CONFIG_OPT} == 0 ) then
  set config_flag = "$*"
else
  set config_flag = "$* $DAKOTA_CONFIG_OPT"
endif

# if no custom make options for this platform, set to empty
if ( ${?DAKOTA_MAKE_OPT} == 0 ) then
  set DAKOTA_MAKE_OPT = ""
endif

# -------
# cleanup
# -------
cd $DAKOTA
rm -f *_dakota.out
rm -f *_dakota.err


# ----------------
# configure Dakota
# ----------------
# configure with arguments passed in and make clean
configure $config_flag >&! config_dakota.out
$TEST_SQA/grep_error.perl config_dakota.out
if ( -e config_dakota.err ) then
  echo "Error configuring DAKOTA; exiting."
  exit
endif
# if reconfiguring, only make clean in Dakota/src
if ( $dakota_reconfig == 'yes') then
  cd src
  gmake clean
  cd ..
else
  gmake clean
endif


# -------------------------------
# reconfig around failed packages
# -------------------------------
# if requested, make lower-level packages and configure around any failures
@ reconfig = 0
if ( $pkg_reconfig == 'yes' ) then

  # iterate over the externals in the DAKOTA top-level directory
  # changed this since we are no longer using a checkout for nightly builds
  # may want to revisit
  foreach p ( `cat $TEST_SQA/package-list-externals` )
    # since p is a relative path, get the package after the last /
    set pp = `echo $p | awk -F\/ '{print $NF}'`
    set plower = `echo $pp | tr 'A-Z' 'a-z'`
    if ( -e $p ) then
      cd $p
      # TODO: confirm this has desired behavior
      gmake -j $BUILD_JOBS $DAKOTA_MAKE_OPT >&! make_"$pp".out
      $TEST_SQA/grep_error.perl make_"$pp".out
      if ( -e make_"$pp".err ) then
        set config_flag = " --without-$plower $config_flag"
        @ reconfig = 1
        echo "$pp Failure" >> $DAKOTA/reconfig_dakota.err
      endif
    else
      set config_flag = " --without-$plower $config_flag"
      @ reconfig = 1
      echo "$pp Failure" >> $DAKOTA/reconfig_dakota.err
    endif
    cd $DAKOTA
  end

endif

# reconfigure iff package build failure
cd $DAKOTA
if ( $reconfig == 1 ) then
  echo $config_flag
  configure $config_flag >&! config_dakota.out
  $TEST_SQA/grep_error.perl config_dakota.out
  if ( -e config_dakota.err ) then
    echo "Error re-configuring DAKOTA; exiting."
    exit
  endif
  rm -rf docs/.dvips
endif

# --------------------------
# final make and error check
# --------------------------

# HACK notice: These issues with the MPI C++ wrappers need resolution.
if ( $UNAME == 'AIX' ) then # POE executables can't fork child POE executables!
  if ( ! -z $CXX ) then
    sed "s/^CXX *= *.*/CXX = $CXX/" test/Makefile > t;mv t test/Makefile
  endif
endif
if ( $UNAME == 'SunOS' ) then # Solaris mpCC doesn't like our #define VERSION!
  foreach f ( dakota_config.h packages/teuchos/src/Teuchos_config.h )
    sed 's/^#define *VERSION.*$/\/\*&\*\//' $f > t;mv t $f
  end
endif

echo "${0}: starting dakota make"
cd $DAKOTA
echo "make command: gmake -j $BUILD_JOBS $DAKOTA_MAKE_OPT "
gmake -j $BUILD_JOBS $DAKOTA_MAKE_OPT >&! make_dakota.out
echo "${0}: finished dakota make"
# the caller is responsible for acting on any resulting errors
$TEST_SQA/grep_error.perl make_dakota.out
