/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class MetricTrackerConvergerBase.

    NOTES:

        See notes of MetricTrackerConvergerBase.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Tue Jul 22 15:43:27 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the MetricTrackerConvergerBase class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <utilities/include/Math.hpp>
#include <../Utilities/include/Logging.hpp>
#include <utilities/include/numeric_limits.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <Convergers/MetricTrackerConvergerBase.hpp>
#include <../Utilities/include/ParameterDatabase.hpp>
#include <../Utilities/include/ParameterExtractor.hpp>


/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;
using namespace JEGA::Utilities;
using namespace JEGA::Logging;
using namespace eddy::utilities;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {









/*
================================================================================
Static Member Data Definitions
================================================================================
*/
const DoubleStack::size_type MetricTrackerConvergerBase::DEFAULT_NUM_GENS(10);

const double MetricTrackerConvergerBase::DEFAULT_PCT_CHNG(0.1);








/*
================================================================================
Mutators
================================================================================
*/
void
MetricTrackerConvergerBase::SetPercentChange(
    double pctChange
    )
{
    EDDY_FUNC_DEBUGSCOPE
    _pctChange = pctChange;

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(),
            GetName() + ": Max allowable % change now = ") << _pctChange
        )
}

void
MetricTrackerConvergerBase::SetNumGenerations(
    std::size_t numGen
    )
{
    EDDY_FUNC_DEBUGSCOPE
    SetMetricStackMaxDepth(numGen);

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(),
            GetName() + ": Number of generations tracked now = ")
                << _metricTracker.GetStackMaxDepth()
        )
}








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/








/*
================================================================================
Subclass Visible Methods
================================================================================
*/


int
MetricTrackerConvergerBase::GetNumDP(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // If the percent change is 0, then we use 2.
    if(this->_pctChange == 0.0) return 2;

    return static_cast<int>(
        Math::Max(Math::Ceil(Math::Abs(
            Math::Log10(this->_pctChange*100.0)
            ))+1, 2.0)
        );
}






/*
================================================================================
Subclass Overridable Methods
================================================================================
*/

bool
MetricTrackerConvergerBase::CheckConvergence(
    const DesignGroup& group,
    const FitnessRecord& fitnesses
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(this->MaxGenEvalConverger::CheckConvergence(group, fitnesses))
        return true;

    // now get the metric to push onto the stack.
    double metric = this->GetMetricValue(group, fitnesses);
    this->_metricTracker.Push(metric);
    JEGALOG_II(this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), this->GetName() + ": Pushed metric value of ")
            << metric << " onto the convergence stack."
        )

    // If we have not yet stored the required number of metric
    // values, we certainly cannot be done.
    if(!this->_metricTracker.IsFull()) return false;

    // Otherwise, we must compare the new value to all previous
    // values to determine if we converged.  The new value is at
    // the top of the stack and is "metric"

    // Iterate the number of entries minus the last one and if we can determine
    // that we don't converge, break.
    size_t e = this->_metricTracker.GetStackDepth() - 1;
    bool converged = true;
    double maxPct = eddy::utilities::numeric_limits<double>::smallest();
    for(size_t i=0; i<e; ++i)
    {
        double currPct = this->GetProgressPercentage(i, e);
        maxPct = Math::Max(currPct, maxPct);
        converged &= (maxPct <= this->_pctChange);
    }

    // report to 1 more decimal place than necessary to cover the _pctChange
    // using at least 2.
    JEGALOG_II(this->GetLogger(), lverbose(), this,
        ostream_entry(lverbose(),
            this->GetName() + ": Current max percentage progress is ")
            << Math::Round(maxPct * 100.0, this->GetNumDP()) << "%."
        )

    // Otherwise we are done if converged is true
    this->SetConverged(converged);

    return this->GetConverged();
}

bool
MetricTrackerConvergerBase::PollForParameters(
    const JEGA::Utilities::ParameterDatabase& db
    )
{
    EDDY_FUNC_DEBUGSCOPE

    size_t ngens = DEFAULT_NUM_GENS;
    bool success = ParameterExtractor::GetSizeTypeFromDB(
        db, "method.jega.num_generations", ngens
		);

    // If we did not find the num gens, warn about it and use the default
    // value.  Note that if !success, then ngens has not been altered.
    JEGAIFLOG_II(!success, GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": The number of generations "
            "was not found in the parameter database.  Using the current "
            "value of ") << ngens
        )

    SetNumGenerations(ngens);

    success = ParameterExtractor::GetDoubleFromDB(
        db, "method.jega.percent_change", _pctChange
        );

    JEGAIFLOG_II(!success, GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": The percent change "
            "was not found in the parameter database.  Using the current "
            "value of ") << _pctChange
        )

    // Otherwise, finish recording and output at the verbose level.
    SetPercentChange(_pctChange);

    return MaxGenEvalConverger::PollForParameters(db);
}





/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/




MetricTrackerConvergerBase::MetricTrackerConvergerBase(
    GeneticAlgorithm& algorithm
    ) :
        MaxGenEvalConverger(algorithm),
        _metricTracker(DEFAULT_NUM_GENS),
        _pctChange(DEFAULT_PCT_CHNG)
{
    EDDY_FUNC_DEBUGSCOPE
}

MetricTrackerConvergerBase::MetricTrackerConvergerBase(
    const MetricTrackerConvergerBase& copy
    ) :
        MaxGenEvalConverger(copy),
        _metricTracker(copy._metricTracker),
        _pctChange(copy._pctChange)
{
    EDDY_FUNC_DEBUGSCOPE
}

MetricTrackerConvergerBase::MetricTrackerConvergerBase(
    const MetricTrackerConvergerBase& copy,
    GeneticAlgorithm& algorithm
    ) :
        MaxGenEvalConverger(copy, algorithm),
        _metricTracker(copy._metricTracker),
        _pctChange(copy._pctChange)
{
    EDDY_FUNC_DEBUGSCOPE
}




/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
