/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       EffGlobalMinimizer
//- Description: Implementation code for the EffGlobalMinimizer class
//- Owner:       Barron J Bichon, Vanderbilt University

#include "system_defs.h"
#include "NonDLHSSampling.H"
#include "RecastModel.H"
#include "DataFitSurrModel.H"
#include "DakotaApproximation.H"
#include "ProblemDescDB.H"
#include "DakotaGraphics.H"
#ifdef DAKOTA_NCSU
#include "NCSUOptimizer.H"
#endif
#include "DakotaModel.H"
#include "DakotaResponse.H"
#include "EffGlobalMinimizer.H"

//#define DEBUG
//#define DEBUG_PLOTS

namespace Dakota {

EffGlobalMinimizer* EffGlobalMinimizer::effGlobalInstance(NULL);

// define special values for componentParallelMode
//#define SURROGATE_MODEL 1
#define TRUTH_MODEL 2


// This constructor accepts a Model
EffGlobalMinimizer::EffGlobalMinimizer(Model& model): 
  SurrBasedMinimizer(model), setUpType("model")
{
  bestVariables = iteratedModel.current_variables().copy();

  // initialize augmented Lagrange multipliers
  size_t num_multipliers = numNonlinearEqConstraints;
  for (size_t i=0; i<numNonlinearIneqConstraints; i++) {
    if (origNonlinIneqLowerBnds[i] > -bigRealBoundSize) // g has a lower bound
      num_multipliers++;
    if (origNonlinIneqUpperBnds[i] <  bigRealBoundSize) // g has an upper bound
      num_multipliers++;
  }
  augLagrangeMult.reshape(num_multipliers);
  augLagrangeMult = 0.;

  truthFnStar.reshape(numFunctions);

  // Always build a global Gaussian process model.  No correction is needed.
  String approx_type = "global_gaussian", sample_reuse = "none", corr_type;
  short approx_order = -1, corr_order = -1;
  // Use a hardwired minimal initial samples.
  int samples = (numContinuousVars+1)*(numContinuousVars+2)/2,
    lhs_seed  = probDescDB.get_int("method.random_seed");

  //int symbols = samples; // symbols needed for DDACE
  Iterator dace_iterator;
  // The following uses on the fly derived ctor:
  dace_iterator.assign_rep(new NonDLHSSampling(iteratedModel, samples,
    lhs_seed, ACTIVE_UNIFORM), false);
    
  // Construct f-hat using a GP approximation for each response function over
  // the active/design vars (same view as iteratedModel: not the typical All
  // view for DACE).
  fHatModel.assign_rep(new DataFitSurrModel(dace_iterator,
    iteratedModel, iteratedModel.current_variables().view(),
    iteratedModel.current_response().active_set(), approx_type,
    approx_order, corr_type, corr_order, sample_reuse), false);

  // The following is not performed in the Optimizer constructor since
  // maxConcurrency may be updated by EGO. The matching free_communicators()
  // appears in the Optimizer destructor.
  //if (scaleFlag || multiObjFlag)
  //  iteratedModel.init_communicators(maxConcurrency);

  // Strategy::init_communicators() initializes the parallel configuration for
  // EffGlobalMinimizer + iteratedModel using maxConcurrency.  It is
  // called after Iterator construction, so performing the initialization here
  // at the f_hat_model level will chain down through the recursions and
  // supercede the subsequent initialization in Strategy::init_comms().
  // maxConcurrency is the local derivative concurrency, which is distinct from
  // the DIRECT concurrency.
  fHatModel.init_communicators(maxConcurrency);

  // Configure a RecastModel with one objective and no constraints using the
  // alternate minimalist constructor: the recast fn pointers are reset for
  // each level within the run fn.
  eifModel.assign_rep(new RecastModel(fHatModel, 1, 0, 0), false);

  // must use alternate NoDB ctor chain
  int max_iterations = 10000, max_fn_evals = 50000;
#ifdef DAKOTA_NCSU
  approxSubProbMinimizer.assign_rep(new
    NCSUOptimizer(eifModel, max_iterations, max_fn_evals), false);
  eifModel.init_communicators(approxSubProbMinimizer.maximum_concurrency());
#else
  Cerr << "NCSU DIRECT is not available to optimize the GP subproblems. " 
       << "Aborting process. " << endl;
        abort_handler(-1);
#endif //DAKOTA_NCSU
}


/** This is an alternate constructor for instantiations on the fly
    using a Model but no ProblemDescDB. */
//EffGlobalMinimizer::
//EffGlobalMinimizer(Model& model, const int& max_iter, const int& max_eval) :
//  SurrBasedMinimizer(NoDBBaseConstructor(), model), setUpType("model")
//{ maxIterations = max_iter; maxFunctionEvals = max_eval; }


EffGlobalMinimizer::~EffGlobalMinimizer() 
{
  // deallocate communicators for DIRECT on eifModel
  eifModel.free_communicators(approxSubProbMinimizer.maximum_concurrency());

  // deallocating communicators here at the fHatModel level will chain down
  // through the recursions and supercede the subsequent deallocation in
  // Strategy::free_communicators().  maxConcurrency is the local derivative
  // concurrency, which is distinct from the DIRECT concurrency.
  fHatModel.free_communicators(maxConcurrency);
}


void EffGlobalMinimizer::minimize_surrogates()
{
  if (setUpType=="model")
    minimize_surrogates_on_model();
  else if (setUpType=="user_functions") {
    //minimize_surrogates_on_user_functions();
    Cerr << "Error: bad setUpType in EffGlobalMinimizer::minimize_surrogates()."
	 << endl;
    abort_handler(-1);
  }
  else {
    Cerr << "Error: bad setUpType in EffGlobalMinimizer::minimize_surrogates()."
	 << endl;
    abort_handler(-1);
  }
}


void EffGlobalMinimizer::minimize_surrogates_on_model()
{
  //------------------------------------------------------------------
  //     Solve the problem.
  //------------------------------------------------------------------

  // set the object instance pointers for use within the static member fns
  EffGlobalMinimizer* prev_instance = effGlobalInstance;
  effGlobalInstance = this;

  // now that variables/labels/bounds/targets have flowed down at run-time from
  // any higher level recursions, propagate them up the instantiate-on-the-fly
  // Model recursion so that they are correct when they propagate back down.
  eifModel.update_from_subordinate_model(); // recurse_flag = true

  // Build initial GP once for all response functions
  fHatModel.build_approximation();

  // Iterate until EGO converges
  size_t convergence_cntr = 0;
  sbIterNum = 0;
  bool approxConverged = false;
  while (!approxConverged) {

    sbIterNum++;

    // initialize EIF recast model
    Sizet2DArray vars_map, primary_resp_map(1), secondary_resp_map;
    BoolDequeArray nonlinear_resp_map(1, BoolDeque(numFunctions, true));
    primary_resp_map[0].reshape(numFunctions);
    for (size_t i=0; i<numFunctions; i++)
      primary_resp_map[0][i] = i;
    RecastModel* eif_model_rep = (RecastModel*)eifModel.model_rep();
    eif_model_rep->initialize(vars_map, false, NULL, NULL,
      primary_resp_map, secondary_resp_map, nonlinear_resp_map,
      EIF_objective_eval, NULL);

    // determine fnStar from among sample data
    get_best_sample();

    // Execute GLOBAL search and retrieve results
    Cout << "\n>>>>> Initiating global optimization\n";
    approxSubProbMinimizer.run_iterator(); // quiet mode
    const Variables&  vars_star = approxSubProbMinimizer.variables_results();
    const RealVector& c_vars    = vars_star.continuous_variables();
    const Response&   resp_star = approxSubProbMinimizer.response_results();
    const Real&       eif_star  = resp_star.function_values()[0];

    // Get expected value for output
    fHatModel.continuous_variables(c_vars);
    fHatModel.compute_response();
    const RealVector& mean = fHatModel.current_response().function_values();
    Real aug_lag = augmented_lagrangian_merit(mean, origNonlinIneqLowerBnds,
      origNonlinIneqUpperBnds, origNonlinEqTargets);

    Cout << "\nResults of EGO iteration:\nFinal point =\n" << c_vars 
	 << "Expected Improvement    =\n                     "
	 << setw(write_precision+7) << -eif_star << "\n                     "
	 << setw(write_precision+7) << aug_lag << " [merit]\n";

#ifdef DEBUG
    RealVector variance = fHatModel.approximation_variances(c_vars);
    RealVector ev = expected_violation(mean,variance);
    RealVector stdv(numFunctions);
    for (size_t i=0; i<numFunctions; i++)
      stdv[i] = sqrt(variance[i]);
    cout << "\nexpected values    =\n" << mean
	 << "\nstandard deviation =\n" << stdv
    	 << "\nexpected violation =\n" << ev << endl;
#endif //DEBUG

    // Check for convergence based on max EIF
    convergenceTol = 1.e-12;
    maxIterations  = 25*numContinuousVars;
    if ( -eif_star < convergenceTol )
      convergence_cntr++;
    // If DIRECT failed to find a point with EIF>0, it returns the
    //   center point as the optimal solution. EGO may have converged,
    //   but DIRECT may have just failed to find a point with a good
    //   EIF value. Adding this midpoint can alter the GPs enough to
    //   to allow DIRECT to find something useful, so we force 
    //   max(EIF)<tol twice to make sure. Note that we cannot make
    //   this check more than 2 because it would cause EGO to add
    //   the center point more than once, which will damage the GPs.
    //   Unfortunately, when it happens the second time, it may still
    //   be that DIRECT failed and not that EGO converged.
    if ( convergence_cntr==2 || sbIterNum >= maxIterations )
      approxConverged = true;
    else {
      // Evaluate response_star_truth
      fHatModel.component_parallel_mode(TRUTH_MODEL);
      iteratedModel.continuous_variables(c_vars);
      ActiveSet set = iteratedModel.current_response().active_set();
      set.request_values(0);
      for (size_t i=0; i<numFunctions; i++)
	set.request_value(i, 1);
      iteratedModel.compute_response(set);
      const Response& resp_star_truth = iteratedModel.current_response();
    
      if (numNonlinearConstraints) {
	// Update the merit function parameters
	// Logic follows Conn, Gould, and Toint, section 14.4:
	const RealVector& fns_star_truth = resp_star_truth.function_values();

	Real norm_cv_star = sqrt(constraint_violation(fns_star_truth, 0.));
	if (norm_cv_star < etaSequence)
	  update_augmented_lagrange_multipliers(fns_star_truth);
	else
	  update_penalty();
      }

      // Update the GP approximation
      fHatModel.append_approximation(vars_star, resp_star_truth, true);
    }

  } // end approx convergence while loop

  // pull optimal result from sample data
  get_best_sample();

  // Set bestVariables and bestResponse for use by strategy level.
  // c_vars, fmin contain the optimal design 
  bestVariables.continuous_variables(varStar);
  bestResponse.function_values(truthFnStar);

  // restore in case of recursion
  effGlobalInstance = prev_instance;

#ifdef DEBUG_PLOTS
  // DEBUG - output set of samples used to build the GP
  // If problem is 2d, output a grid of points on the GP 
  //   and truth (if requested)
  for (size_t i=0; i<numFunctions; i++) {
    char samsfile[] = "ego_sams", out[] = ".out", tag[] = "_", num[3];
    snprintf(num, 3, "%d", i+1);
    strcat(tag, num);
    strcat(tag, out);
    strcat(samsfile, tag);
    ofstream samsOut(samsfile,ios::out);
    samsOut << scientific;
    const List<SurrogateDataPoint>& gp_data
      = fHatModel.approximation_data(i);
    size_t num_data_pts = gp_data.entries();
    size_t num_vars = fHatModel.continuous_variables().length();
    List<SurrogateDataPoint>::const_iterator cit = gp_data.begin();
    RealVector sams(num_vars);
    for (size_t j=0; j<num_data_pts; j++, cit++) {
      sams = cit->continuous_variables();
      Real true_fn = cit->response_function();

      samsOut << '\n';
      for (size_t k=0; k<num_vars; k++)
	samsOut << setw(13) << sams[k] << ' ';
      samsOut<< setw(13) << true_fn;
    }
    samsOut << endl;

    // Plotting the GP over a grid is intended for visualization and
    //   is therefore only available for 2D problems
    if (num_vars==2) {
      char gpfile[] = "ego_gp", varfile[] = "ego_var";
      strcat(gpfile,  tag);
      strcat(varfile, tag);
      ofstream  gpOut(gpfile,        ios::out);
      ofstream varOut(varfile,       ios::out);
      ofstream eifOut("ego_eif.out", ios::out);
      gpOut  << scientific;
      varOut << scientific;
      eifOut << scientific;
      RealVector test_pt(2);
      Real lbnd0 = fHatModel.continuous_lower_bounds()[0], 
	   ubnd0 = fHatModel.continuous_upper_bounds()[0],
	   lbnd1 = fHatModel.continuous_lower_bounds()[1], 
	   ubnd1 = fHatModel.continuous_upper_bounds()[1];
      Real interval0 = (ubnd0-lbnd0)/100., interval1 = (ubnd1-lbnd1)/100.;
      for (size_t j=0; j<101; j++){
	test_pt[0] = lbnd0 + float(j)*interval0;
	for (size_t k=0; k<101; k++){
	  test_pt[1] = lbnd1 + float(k)*interval1;
      
	  fHatModel.continuous_variables(test_pt);
	  fHatModel.compute_response();
	  const Response& gp_resp = fHatModel.current_response();
	  const RealVector& gp_fn = gp_resp.function_values();
	  
	  gpOut << '\n' << setw(13) << test_pt[0] << ' ' 
		<< setw(13) << test_pt[1] << ' ' << setw(13) << gp_fn[i];

	  RealVector variances = fHatModel.approximation_variances(test_pt);

	  varOut << '\n' << setw(13) << test_pt[0] << ' '
		 << setw(13) << test_pt[1] << ' ' << setw(13) << variances[i];

	  if (i==numFunctions-1) {
	    Real m = augmented_lagrangian_merit(gp_fn, origNonlinIneqLowerBnds, 
	      origNonlinIneqUpperBnds, origNonlinEqTargets);
	    RealVector merit(1);
	    merit[0] = m;

	    Real ei = expected_improvement(merit, test_pt);

	    eifOut << '\n' << setw(13) << test_pt[0] << ' ' 
		   << setw(13) << test_pt[1] << ' ' << setw(13) << ei;
	  }
	}
	gpOut  << endl;
	varOut << endl;
	if (i==numFunctions-1)
	  eifOut << endl;
      }
    }
  }
#endif //DEBUG_PLOTS
}


void EffGlobalMinimizer::
EIF_objective_eval(const Variables& sub_model_vars,
		   const Variables& recast_vars,
		   const Response& sub_model_response,
		   Response& recast_response)
{
  // Means are passed in, but must retrieve variance from the GP
  const RealVector& x     = recast_vars.continuous_variables();
  const RealVector& means = sub_model_response.function_values();
  const RealVector& variances
    = effGlobalInstance->fHatModel.approximation_variances(x);

  const ShortArray& recast_asv = recast_response.active_set_request_vector();
  if (recast_asv[0] & 1) { // return -EI since we are maximizing
    Real neg_ei = -effGlobalInstance->expected_improvement(means, variances);
    recast_response.function_value(neg_ei, 0);
  }
}


Real EffGlobalMinimizer::
expected_improvement(const RealVector& means, const RealVector& variances)
{
  Real mean = objective(means), stdv;
  if ( numNonlinearConstraints ) {
    // mean_M = mean_f + lambda*EV + r_p*EV*EV
    // stdv_M = stdv_f
    RealVector ev = expected_violation(means, variances);
    for (size_t i=0; i<numNonlinearConstraints; i++)
      mean += augLagrangeMult[i]*ev[i] + penaltyParameter*ev[i]*ev[i];
    stdv = sqrt(variances[0]); // ***
  }
  else { // extend for NLS/MOO ***
    // mean_M = M(mu_f)
    // stdv_M = sqrt(var_f)
    stdv = sqrt(variances[0]); // *** sqrt(sum(variances(1:numUserPrimaryFns));
  }

  // Calculate the expected improvement
  Real snv = (meritFnStar - mean)/stdv, // standard normal variate
       cdf = Phi(snv), pdf = phi(snv), ei = (meritFnStar - mean)*cdf + stdv*pdf;

  return ei;
}


RealVector EffGlobalMinimizer::
expected_violation(const RealVector& means, const RealVector& variances)
{
  RealVector ev(numNonlinearConstraints);

  size_t i, cntr=0;
  // inequality constraints
  for (i=0; i<numNonlinearIneqConstraints; i++) {
    const Real& mean = means[numUserPrimaryFns+i];
    const Real& stdv = sqrt(variances[numUserPrimaryFns+i]);
    const Real& lbnd = origNonlinIneqLowerBnds[i];
    const Real& ubnd = origNonlinIneqUpperBnds[i];
    if (lbnd > -bigRealBoundSize) {
      Real snv = (lbnd-mean)/stdv, cdf = Phi(snv), pdf = phi(snv);
      ev[cntr++] = (lbnd-mean)*cdf + stdv*pdf;
    }
    if (ubnd < bigRealBoundSize) {
      Real snv = (ubnd-mean)/stdv, cdf = Phi(snv), pdf = phi(snv);
      ev[cntr++] = (mean-ubnd)*(1.-cdf) + stdv*pdf;
    }
  }
  // equality constraints
  for (i=0; i<numNonlinearEqConstraints; i++) {
    const Real& mean = means[numUserPrimaryFns+numNonlinearIneqConstraints+i];
    const Real& stdv
      = sqrt(variances[numUserPrimaryFns+numNonlinearIneqConstraints+i]);
    const Real& zbar = origNonlinEqTargets[i];
    Real snv = (zbar-mean)/stdv, cdf = Phi(snv), pdf = phi(snv);
    ev[cntr++] = (zbar-mean)*(2.*cdf-1.) + 2.*stdv*pdf;
  }

  return ev;
}


void EffGlobalMinimizer::get_best_sample()
{
  // Pull the samples and responses from data used to build latest GP
  // to determine fnStar for use in the expected improvement function

  const List<SurrogateDataPoint>& gp_data_0 = fHatModel.approximation_data(0);

  size_t i, sam_star_idx = 0;
  Real fn, fn_star = DBL_MAX;
  List<SurrogateDataPoint>::const_iterator cit;
  for (i=0, cit = gp_data_0.begin(); cit != gp_data_0.end(); i++, cit++) {
    const RealVector& sams = cit->continuous_variables();

    fHatModel.continuous_variables(sams);
    fHatModel.compute_response();
    const RealVector& f_hat = fHatModel.current_response().function_values();
    fn = augmented_lagrangian_merit(f_hat, origNonlinIneqLowerBnds,
      origNonlinIneqUpperBnds, origNonlinEqTargets);

    if (fn < fn_star) {
      sam_star_idx   = i;
      fn_star        = fn;
      varStar        = sams;
      meritFnStar    = fn;
      truthFnStar[0] = cit->response_function();
    }
  }

  // update truthFnStar with all additional primary/secondary fns corresponding
  // to lowest merit function value
  for (i=1; i<numFunctions; i++) {
    const List<SurrogateDataPoint>& gp_data_i = fHatModel.approximation_data(i);
    cit = gp_data_i.begin();
    advance(cit, sam_star_idx);
    truthFnStar[i] = cit->response_function();
  }
}


void EffGlobalMinimizer::update_penalty()
{
  // Logic follows Conn, Gould, and Toint, section 14.4, step 3
  //   CGT use mu *= tau with tau = 0.01 ->   r_p *= 50
  //   Rodriguez, Renaud, Watson:             r_p *= 10
  //   Robinson, Willcox, Eldred, and Haimes: r_p *= 5
  penaltyParameter *= 10.;
  //penaltyParameter = min(penaltyParameter, 1.e+20); // cap the max penalty?
  Real mu = 1./2./penaltyParameter; // conversion between r_p and mu penalties
  etaSequence = eta*pow(mu, alphaEta);

#ifdef DEBUG
  Cout << "Penalty updated: " << penaltyParameter << '\n'
       << "eta updated:     " << etaSequence      << '\n'
       << "Augmented Lagrange multipliers:\n" << augLagrangeMult;
#endif
}

} // namespace Dakota
