/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class OffsetMutatorBase.

    NOTES:

        See notes of OffsetMutatorBase.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu Jul 10 07:56:01 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the OffsetMutatorBase class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <Mutators/OffsetMutatorBase.hpp>
#include <../Utilities/include/Logging.hpp>
#include <../Utilities/include/DesignGroup.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/ParameterExtractor.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>
#include <utilities/include/RandomNumberGenerator.hpp>





/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA;
using namespace JEGA::Logging;
using namespace JEGA::Utilities;
using namespace eddy::utilities;








/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Algorithms {








/*
================================================================================
Static Member Data Definitions
================================================================================
*/
const double OffsetMutatorBase::DEFAULT_OFFSET_RANGE(0.1);







/*
================================================================================
Mutators
================================================================================
*/
void
OffsetMutatorBase::SetOffsetRange(
    double val
    )
{
    EDDY_FUNC_DEBUGSCOPE

    _offsetRange = val;

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(),
            GetName() + ": Offset range now = ") << _offsetRange
        )

    JEGAIFLOG_II((_offsetRange < 0.0) || (_offsetRange > 1.0), GetLogger(),
        lquiet(), this, text_entry(lquiet(),
            "Offset Mutator Base: Offset range is not in [0, 1].  This may be "
            "a problem for some offset mutators.")
        )
}








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/








/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/




void
OffsetMutatorBase::Mutate(
    DesignGroup& group
    )
{
    EDDY_FUNC_DEBUGSCOPE

    JEGALOG_II(GetLogger(), ldebug(), this,
        text_entry(ldebug(), GetName() + ": Performing mutation.")
        )

    // store the design target for repeated use
    DesignTarget& target = group.GetDesignTarget();

    // Get the number of design variables per Design
    const size_t ndv = target.GetNDV();

    // extract the design variable information
    const DesignVariableInfoVector& dvis =
      target.GetDesignVariableInfos();

    // Get the mutation rate.
    const double rate = GetRate();

    // determine the number of mutation operations;
    const size_t nmutate =
        static_cast<size_t>(Math::Round(rate * group.GetSize()));

    // if there will be no mutations, don't continue.
    if(nmutate < 1)
    {
        JEGALOG_II(GetLogger(), ldebug(), this,
            text_entry(ldebug(), GetName() + ": Rate and group size are such "
                       "that no mutation will occur.")
            )
        return;
    }

    // prepare some variables for use in the following loop.
    DesignDVSortSet::size_type csize = group.GetSize();

    // The DesignGroup sort containers do not support random access so we will
    // copy the contents into a temporary container that does for speed.
    Design** temp = new Design*[csize];
    std::copy(group.BeginDV(), group.EndDV(), temp);

    // Carry out the mutations
    for(size_t i=0; i<nmutate; ++i)
    {
        // pick a random design variable;
        size_t dv =
            RandomNumberGenerator::UniformInt<size_t>(0, ndv-1);

        // store the design being changed.
        Design* chosen = temp[
            RandomNumberGenerator::UniformInt<DesignDVSortSet::size_type>(
                0, csize-1
                )
            ];

        // remove the design being changed
        group.Erase(chosen);

        // perform the mutation.

        // get the old representation to modify it.
        double varrep = chosen->GetVariableRep(dv);

        // add the offset amount
        varrep += GetOffsetAmount(*dvis.at(dv));

        // Note that there is a strong likelyhood that
        // this will produce invalid variable representations
        // especially for discrete variables.
        // Just a warning.
        JEGAIFLOG_II(!dvis.at(dv)->IsValidDoubleRep(varrep), GetLogger(),
            lquiet(), this,
            ostream_entry(lquiet(), GetName() + ": Produced invalid "
                " variable representation ") << varrep << ". Mutation"
                " performed without corrective action."
            )

        // record the new value
        chosen->SetVariableRep(dv, varrep);

        // now re-insert the design being changed
        group.Insert(chosen);
    }

    JEGALOG_II(GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": Performed ") << nmutate
            << " total mutations."
        )

    // delete the temporary vector of Design*.
    delete [] temp;
}

bool
OffsetMutatorBase::PollForParameters(
    const JEGA::Utilities::ParameterDatabase& db
    )
{
    EDDY_FUNC_DEBUGSCOPE

    bool success = ParameterExtractor::GetDoubleFromDB(
        db, "method.mutation_scale", _offsetRange
        );

    // If we did not find the offset range, warn about it and use the default
    // value.  Note that if !success, then _offsetRange has not been altered.
    JEGAIFLOG_II(!success, GetLogger(), lverbose(), this,
        ostream_entry(lverbose(), GetName() + ": The offset range was not "
            "found in the parameter database.  Using the current value of ")
            << _offsetRange
        )

    SetOffsetRange(_offsetRange);

    return GeneticAlgorithmMutator::PollForParameters(db);
}




/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/




OffsetMutatorBase::OffsetMutatorBase(
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmMutator(algorithm),
        _offsetRange(DEFAULT_OFFSET_RANGE)
{
    EDDY_FUNC_DEBUGSCOPE
}

OffsetMutatorBase::OffsetMutatorBase(
    const OffsetMutatorBase& copy
    ) :
        GeneticAlgorithmMutator(copy),
        _offsetRange(copy._offsetRange)
{
    EDDY_FUNC_DEBUGSCOPE
}

OffsetMutatorBase::OffsetMutatorBase(
    const OffsetMutatorBase& copy,
    GeneticAlgorithm& algorithm
    ) :
        GeneticAlgorithmMutator(copy, algorithm),
        _offsetRange(copy._offsetRange)
{
    EDDY_FUNC_DEBUGSCOPE
}








/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Algorithms
} // namespace JEGA
