/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA) Managed Front End

    CONTENTS:

        Implementation of class MGeneticAlgorithm.

    NOTES:

        See notes of MGeneticAlgorithm.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        This library is free software; you can redistribute it and/or
        modify it under the terms of the GNU Lesser General Public
        License as published by the Free Software Foundation; either
        version 2.1 of the License, or (at your option) any later version.
        
        This library is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
        Lesser General Public License for more details.
        
        You should have received a copy of the GNU Lesser General Public
        License along with this library; if not, write to the Free Software
        Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
        USA

    VERSION:

        1.0.0

    CHANGES:

        Wed Feb 14 16:01:13 2007 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the MGeneticAlgorithm class.
 */




/*
================================================================================
Includes
================================================================================
*/
#include <stdafx.h>
#include <MSolution.hpp>
#include <MGeneticAlgorithm.hpp>

#pragma unmanaged
#include <../include/GeneticAlgorithm.hpp>
#pragma managed






/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace JEGA::Utilities;
using namespace JEGA::Algorithms;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace FrontEnd {
        namespace Managed {





/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/
GeneticAlgorithm&
MGeneticAlgorithm::Manifest(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_theAlg != 0x0)
    return *this->_theAlg;
}

void
MGeneticAlgorithm::DisassociateGA(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->_theAlg = 0x0;
}

MSolution MOH
MGeneticAlgorithm::GetBestDesign(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_theAlg != 0x0)
    const Design* best = this->_theAlg->GetBestDesign();
    if(best == 0x0) return MANAGED_NULL_HANDLE;
    return MANAGED_GCNEW MSolution(*best);
}

SolutionVector MOH
MGeneticAlgorithm::GetCurrentSolution(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_theAlg != 0x0)

    DesignOFSortSet solutions(this->_theAlg->GetCurrentSolution());

    SolutionVector MOH ret =
        MANAGED_GCNEW SolutionVector(solutions.size());

    for(DesignOFSortSet::const_iterator it(solutions.begin());
        it!=solutions.end(); ++it) ret->Add(MANAGED_GCNEW MSolution(**it));

    return ret;
}

SolutionVector MOH
MGeneticAlgorithm::GetCurrentPopulation(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_theAlg != 0x0)

    DesignOFSortSet solutions(
        this->_theAlg->GetPopulation().GetOFSortContainer()
        );

    SolutionVector MOH ret =
        MANAGED_GCNEW SolutionVector(solutions.size());

    for(DesignOFSortSet::const_iterator it(solutions.begin());
        it!=solutions.end(); ++it) ret->Add(MANAGED_GCNEW MSolution(**it));

    return ret;
}

size_t
MGeneticAlgorithm::GetGenerationNumber(
    )
{
    return static_cast<size_t>(this->_theAlg->GetGenerationNumber());
}

std::size_t
MGeneticAlgorithm::GetNDV(
    )
{
    return this->_theAlg->GetDesignTarget().GetNDV();
}

std::size_t
MGeneticAlgorithm::GetNOF(
    )
{
    return this->_theAlg->GetDesignTarget().GetNOF();
}

std::size_t
MGeneticAlgorithm::GetNCN(
    )
{
    return this->_theAlg->GetDesignTarget().GetNCN();
}

void
MGeneticAlgorithm::FlushLogStreams(
    )
{
    this->_theAlg->GetLogger().FlushStreams();
}





/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/
void
MGeneticAlgorithm::MANAGED_DISPOSE(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    // Do not destroy the GA here.  That will be done externally by a driver.
    this->DisassociateGA();
}








/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/

MGeneticAlgorithm::MGeneticAlgorithm(
    GeneticAlgorithm* wrapped
    ) :
        _theAlg(wrapped)
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(this->_theAlg != 0x0)
}

MGeneticAlgorithm::~MGeneticAlgorithm(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    this->MANAGED_DISPOSE();
}









/*
================================================================================
End Namespace
================================================================================
*/
        } // namespace Managed
    } // namespace FrontEnd
} // namespace JEGA

