/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        BasisPolyApproximation
//- Description:  Base Class for Orthogonal/Interpolation Polynomial
//-               Approximations
//-               
//- Owner:        Mike Eldred

#ifndef BASIS_POLY_APPROXIMATION_H
#define BASIS_POLY_APPROXIMATION_H

#include "DakotaApproximation.H"
#include "BasisPolynomial.H"
#include "NonDIntegration.H"
#include "DataModel.H"


namespace Dakota {

class ProblemDescDB;


/// Derived approximation class for global basis polynomials.

/** The BasisPolyApproximation class provides a global approximation
    based on basis polynomials.  This includes orthogonal polynomials
    used for polynomial chaos expansions and interpolation polynomials
    used for stochastic collocation. */

class BasisPolyApproximation: public Approximation
{
public:

  //
  //- Heading: Constructor and destructor
  //

  /// default constructor
  BasisPolyApproximation();
  /// standard constructor
  BasisPolyApproximation(const ProblemDescDB& problem_db,
			 const size_t& num_acv);
  /// destructor
  ~BasisPolyApproximation();

  //
  //- Heading: Virtual member functions
  //

  /// return the mean of the expansion, treating all variables as random
  virtual const Real& get_mean() = 0;
  /// return the mean of the expansion for a given parameter vector,
  /// treating a subset of the variables as random
  virtual const Real& get_mean(const RealVector& x) = 0;
  /// return the gradient of the expansion mean for a given parameter
  /// vector, treating all variables as random
  virtual const RealBaseVector& get_mean_gradient() = 0;
  /// return the gradient of the expansion mean for a given parameter vector
  /// and given DVV, treating a subset of the variables as random
  virtual const RealBaseVector& get_mean_gradient(const RealVector& x,
						  const UIntArray& dvv) = 0;

  /// return the variance of the expansion, treating all variables as random
  virtual const Real& get_variance() = 0;
  /// return the variance of the expansion for a given parameter vector,
  /// treating a subset of the variables as random
  virtual const Real& get_variance(const RealVector& x) = 0;
  /// return the gradient of the expansion variance for a given parameter
  /// vector, treating all variables as random
  virtual const RealBaseVector& get_variance_gradient() = 0;
  /// return the gradient of the expansion variance for a given parameter
  /// vector and given DVV, treating a subset of the variables as random
  virtual const RealBaseVector& get_variance_gradient(const RealVector& x,
						      const UIntArray& dvv) = 0;

  //
  //- Heading: Member functions
  //

  /// set expCoeffsSolnApproach
  void solution_approach(short soln_approach);
  /// get expCoeffsSolnApproach
  short solution_approach() const;

  /// set expansionCoeffFlag
  void expansion_coefficient_flag(bool coeff_flag);
  /// get expansionCoeffFlag
  bool expansion_coefficient_flag() const;

  /// set expansionGradFlag
  void expansion_gradient_flag(bool grad_flag);
  /// get expansionGradFlag
  bool expansion_gradient_flag() const;

  /// set integrationRep
  void integration_iterator(const Iterator& iterator);
  /// set randomVarsKey
  void random_variables_key(const BoolDeque& random_vars_key);

protected:

  //
  //- Heading: Virtual function redefinitions
  //

  int num_constraints()  const;

  const RealVector& approximation_coefficients() const;
  void approximation_coefficients(const RealVector& approx_coeffs);

  //
  //- Heading: Data
  //

  /// identifies the approach taken in find_coefficients():
  /// QUADRATURE, SPARSE_GRID, REGRESSION, or SAMPLING
  short expCoeffsSolnApproach;

  /// flag for calculation of expansionCoeffs from response values
  bool expansionCoeffFlag;
  /// flag for calculation of expansionCoeffGrads from response gradients
  bool expansionGradFlag;

  /// NonDIntegration instance used for access to quadrature order and
  /// weight products
  NonDIntegration* integrationRep;

  /// array of booleans identifying the random variable subset within
  /// the active variables (used in all_variables mode)
  BoolDeque randomVarsKey;
  /// list of indices identifying the random variable subset within the active
  /// variables (used in all_variables mode; defined from randomVarsKey)
  SizetList randomIndices;
  /// list of indices identifying the non-random variable subset within the
  /// active variables (used in all_variables mode; defined from randomVarsKey)
  SizetList nonRandomIndices;

  /// expected value of the expansion
  Real expansionMean;
  /// gradient of the expected value of the expansion
  RealBaseVector expansionMeanGrad;
  /// variance of the expansion
  Real expansionVariance;
  /// gradient of the variance of the expansion
  RealBaseVector expansionVarianceGrad;

  /// the coefficients of the expansion
  RealVector expansionCoeffs;
  /// the gradients of the expansion coefficients
  /** may be interpreted as either the gradients of the expansion
      coefficients or the coefficients of expansions for the response
      gradients.  This array is used when sensitivities of moments are
      needed with respect to variables that do not appear in the
      expansion (e.g., with respect to design variables for an
      expansion only over the random variables). */
  RealMatrix expansionCoeffGrads;

private:

  //
  //- Heading: Data
  //
};


inline BasisPolyApproximation::BasisPolyApproximation():
  expCoeffsSolnApproach(SAMPLING), expansionCoeffFlag(true),
  expansionGradFlag(false), integrationRep(NULL)
{ }


inline BasisPolyApproximation::~BasisPolyApproximation()
{ }


inline void BasisPolyApproximation::solution_approach(short soln_approach)
{ expCoeffsSolnApproach = soln_approach; }


inline short BasisPolyApproximation::solution_approach() const
{ return expCoeffsSolnApproach; }


inline void BasisPolyApproximation::expansion_coefficient_flag(bool coeff_flag)
{ expansionCoeffFlag = coeff_flag; }


inline bool BasisPolyApproximation::expansion_coefficient_flag() const
{ return expansionCoeffFlag; }


inline void BasisPolyApproximation::expansion_gradient_flag(bool grad_flag)
{ expansionGradFlag = grad_flag; }


inline bool BasisPolyApproximation::expansion_gradient_flag() const
{ return expansionGradFlag; }


inline void BasisPolyApproximation::
integration_iterator(const Iterator& iterator)
{ integrationRep = (NonDIntegration*)iterator.iterator_rep(); }


inline void BasisPolyApproximation::
random_variables_key(const BoolDeque& random_vars_key)
{
  randomVarsKey = random_vars_key;
  randomIndices.clear();
  nonRandomIndices.clear();
  for (size_t i=0; i<numVars; i++)
    if (random_vars_key[i])
      randomIndices.insert(i);
    else
      nonRandomIndices.insert(i);
}


inline int BasisPolyApproximation::num_constraints() const
{ return (anchorPoint.is_null()) ? 0 : 1; }


inline const RealVector& BasisPolyApproximation::
approximation_coefficients() const
{ return expansionCoeffs; }


inline void BasisPolyApproximation::
approximation_coefficients(const RealVector& approx_coeffs)
{ expansionCoeffs = approx_coeffs; }

} // namespace Dakota

#endif
