# include <cstdlib>
# include <iostream>
# include <iomanip>
# include <cmath>
# include <string>
# include "sandia_rules.H"

using namespace std;
using namespace webbur2;

int main ( void );
void test01 ( void );
void test02 ( void );
void test03 ( void );
void test04 ( void );
void test05 ( void );
void test06 ( void );
void test07 ( void );
void test08 ( void );
void test09 ( void );
void test10 ( void );
void test11 ( void );
void test12 ( void );
void test13 ( void );
void test14 ( void );
void test15 ( void );
void test16 ( void );
void test17 ( void );
void test18 ( void );

//****************************************************************************80

int main ( void )

//****************************************************************************80
//
//  Purpose:
//
//    MAIN is the main program for SANDIA_RULES_PRB.
//
//  Discussion:
//
//    SANDIA_RULES_PRB calls a set of tests for the SANDIA_RULES library.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  timestamp ( );

  cout << "\n";
  cout << "SANDIA_RULES_PRB\n";
  cout << "  C++ version\n";
  cout << "\n";
  cout << "  Test the routines in the SANDIA_RULES library.\n";

  test01 ( );
  test02 ( );
  test03 ( );
  test04 ( );
  test05 ( );
  test06 ( );
  test07 ( );
  test08 ( );
  test09 ( );
  test10 ( );
  test11 ( );
  test12 ( );
  test13 ( );
  test14 ( );
  test15 ( );
  test16 ( );
  test17 ( );
  test18 ( );

  cout << "\n";
  cout << "SANDIA_RULES_PRB\n";
  cout << "  Normal end of execution.\n";

  cout << "\n";
  timestamp ( );
 
  return 0;
}
//****************************************************************************80

void test01 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST01 tests CHEBYSHEV1_COMPUTE against CHEBYSHEV1_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST01\n";
  cout << "  CHEBYSHEV1_COMPUTE computes a Gauss-Chebyshev type 1 rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) / sqrt ( 1 - x^2 ) dx.\n";
  cout << "\n";
  cout << "  CHEBYSHEV1_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Estimate       Exact            Error\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";

    f = new double[order];
    w = new double[order];
    x = new double[order];

    chebyshev1_compute ( order, x, w );

    for ( n = 0; n <= 2 * order + 2; n = n + 1 )
    {
      exact = chebyshev1_integral ( n );

      if ( n == 0 )
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = 1.0;
        }
      }
      else
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = pow ( x[i], n );
        }
      }
      estimate = 0.0;
      for ( i = 0; i < order; i++ )
      {
        estimate = estimate + w[i] * f[i];
      }

      error = r8_abs ( exact - estimate );
  
      cout << "  " << setw(8)  << order
           << "  " << setw(8)  << n
           << "  " << setprecision(6) << setw(14) << estimate
           << "  " << setprecision(6) << setw(14) << exact
           << "  " << setprecision(6) << setw(14) << error << "\n";
    }
    delete [] f;
    delete [] w;
    delete [] x;
  }
  return;
}
//****************************************************************************80

void test02 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST02 tests CHEBYSHEV1_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST02\n";
  cout << "  CHEBYSHEV1_COMPUTE computes a Gauss-Chebyshev type 1 rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) / sqrt(1-x^2) dx.\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";
    cout << "  Order = " << order << "\n";

    w = new double[order];
    x = new double[order];

    chebyshev1_compute ( order, x, w );
 
    for ( i = 0; i < order; i =i + 1 )
    {
      cout << "  " << setw(8) << i
           << "  " << setprecision(16) << setw(24) << x[i]
           << "  " << setprecision(16) << setw(24) << w[i] << "\n";
    }
    delete [] w;
    delete [] x;
  }
 
  return;
}
//****************************************************************************80

void test03 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST03 tests CHEBYSHEV2_COMPUTE against CHEBYSHEV2_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST03\n";
  cout << "  CHEBYSHEV2_COMPUTE computes a Gauss-Chebyshev type 2 rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) * sqrt ( 1 - x^2 ) dx.\n";
  cout << "\n";
  cout << "  CHEBYSHEV2_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Estimate       Exact            Error\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";

    f = new double[order];
    w = new double[order];
    x = new double[order];

    chebyshev2_compute ( order, x, w );

    for ( n = 0; n <= 2 * order + 2; n = n + 1 )
    {
      exact = chebyshev2_integral ( n );

      if ( n == 0 )
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = 1.0;
        }
      }
      else
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = pow ( x[i], n );
        }
      }
      estimate = 0.0;
      for ( i = 0; i < order; i++ )
      {
        estimate = estimate + w[i] * f[i];
      }

      error = r8_abs ( exact - estimate );
  
      cout << "  " << setw(8)  << order
           << "  " << setw(8)  << n
           << "  " << setprecision(6) << setw(14) << estimate
           << "  " << setprecision(6) << setw(14) << exact
           << "  " << setprecision(6) << setw(14) << error << "\n";
    }
    delete [] f;
    delete [] w;
    delete [] x;
  }
  return;
}
//****************************************************************************80

void test04 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST04 tests CHEBYSHEV2_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST04\n";
  cout << "  CHEBYSHEV2_COMPUTE computes a Gauss-Chebyshev type 2 rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) * sqrt(1-x^2) dx.\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";
    cout << "  Order = " << order << "\n";

    w = new double[order];
    x = new double[order];

    chebyshev2_compute ( order, x, w );
 
    for ( i = 0; i < order; i =i + 1 )
    {
      cout << "  " << setw(8) << i
           << "  " << setprecision(16) << setw(24) << x[i]
           << "  " << setprecision(16) << setw(24) << w[i] << "\n";
    }
    delete [] w;
    delete [] x;
  }
 
  return;
}
//****************************************************************************80

void test05 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST05 tests GEGENBAUER_COMPUTE against GEGENBAUER_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST05\n";
  cout << "  GEGENBAUER_COMPUTE computes a generalized Gauss-Gegenbauer rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) (1-x^2)^alpha dx.\n";
  cout << "\n";
  cout << "  GEGENBAUER_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Alpha           Estimate       Exact            Error\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";

      f = new double[order];
      w = new double[order];
      x = new double[order];

      gegenbauer_compute ( order, alpha, x, w );
 
      for ( n = 0; n <= 2 * order + 2; n = n + 1 )
      {
        exact = gegenbauer_integral ( n, alpha );

        if ( n == 0 )
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = 1.0;
          }
        }
        else
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = pow ( x[i], n );
          }
        }
        estimate = 0.0;
        for ( i = 0; i < order; i++ )
        {
          estimate = estimate + w[i] * f[i];
        }

        error = r8_abs ( exact - estimate );
  
        cout << "  " << setw(8)  << order
             << "  " << setw(8)  << n
             << "  " << setw(14) << alpha
             << "  " << setprecision(6) << setw(14) << estimate
             << "  " << setprecision(6) << setw(14) << exact
             << "  " << setprecision(6) << setw(14) << error << "\n";
      }
      delete [] f;
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test06 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST06 tests GEGENBAUER_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST06\n";
  cout << "  GEGENBAUER_COMPUTE computes a generalized Gauss-Gegenbauer rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) (1-x^2)^alpha dx.\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";
      cout << "  Order = " << order << "\n";
      cout << "  ALPHA = " << alpha << "\n";

      w = new double[order];
      x = new double[order];

      gegenbauer_compute ( order, alpha, x, w );
 
      for ( i = 0; i < order; i =i + 1 )
      {
        cout << "  " << setw(8) << i
             << "  " << setprecision(16) << setw(24) << x[i]
             << "  " << setprecision(16) << setw(24) << w[i] << "\n";
      }
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test07 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST07 tests GEN_HERMITE_COMPUTE against GEN_HERMITE_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST07\n";
  cout << "  GEN_HERMITE_COMPUTE computes a generalized Gauss-Hermite rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -oo < x < +oo ) f(x) x^alpha exp(-x*x) dx.\n";
  cout << "\n";
  cout << "  GEN_HERMITE_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Alpha           Estimate       Exact            Error\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";

      f = new double[order];
      w = new double[order];
      x = new double[order];

      gen_hermite_compute ( order, alpha, x, w );
 
      for ( n = 0; n <= 2 * order + 2; n = n + 1 )
      {
        exact = gen_hermite_integral ( n, alpha );

        if ( n == 0 )
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = 1.0;
          }
        }
        else
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = pow ( x[i], n );
          }
        }
        estimate = 0.0;
        for ( i = 0; i < order; i++ )
        {
          estimate = estimate + w[i] * f[i];
        }

        error = r8_abs ( exact - estimate );
  
        cout << "  " << setw(8)  << order
             << "  " << setw(8)  << n
             << "  " << setprecision(6) << setw(14) << alpha
             << "  " << setprecision(6) << setw(14) << estimate
             << "  " << setprecision(6) << setw(14) << exact
             << "  " << setprecision(6) << setw(14) << error << "\n";
      }
      delete [] f;
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test08 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST08 tests GEN_HERMITE_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST08\n";
  cout << "  GEN_HERMITE_COMPUTE computes a generalized Gauss-Hermite rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -oo < x < +oo ) f(x) x^alpha exp(-x*x) dx.\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";
      cout << "  Order = " << order << "\n";
      cout << "  ALPHA = " << alpha << "\n";

      w = new double[order];
      x = new double[order];

      gen_hermite_compute ( order, alpha, x, w );
 
      for ( i = 0; i < order; i =i + 1 )
      {
        cout << "  " << setw(8) << i
             << "  " << setprecision(16) << setw(24) << x[i]
             << "  " << setprecision(16) << setw(24) << w[i] << "\n";
      }
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test09 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST09 tests GEN_LAGUERRE_COMPUTE against GEN_LAGUERRE_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST09\n";
  cout << "  GEN_LAGUERRE_COMPUTE computes a generalized Gauss-Laguerre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) x^alpha exp(-x) dx.\n";
  cout << "\n";
  cout << "  GEN_LAGUERRE_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Alpha           Estimate       Exact            Error\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";

      f = new double[order];
      w = new double[order];
      x = new double[order];

      gen_laguerre_compute ( order, alpha, x, w );
 
      for ( n = 0; n <= 2 * order + 2; n = n + 1 )
      {
        exact = gen_laguerre_integral ( n, alpha );

        if ( n == 0 )
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = 1.0;
          }
        }
        else
        {
          for ( i = 0; i < order; i++ )
          {
            f[i] = pow ( x[i], n );
          }
        }
        estimate = 0.0;
        for ( i = 0; i < order; i++ )
        {
          estimate = estimate + w[i] * f[i];
        }

        error = r8_abs ( exact - estimate );
  
        cout << "  " << setw(8)  << order
             << "  " << setw(8)  << n
             << "  " << setprecision(6) << setw(14) << alpha
             << "  " << setprecision(6) << setw(14) << estimate
             << "  " << setprecision(6) << setw(14) << exact
             << "  " << setprecision(6) << setw(14) << error << "\n";
      }
      delete [] f;
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test10 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST10 tests GEN_LAGUERRE_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST10\n";
  cout << "  GEN_LAGUERRE_COMPUTE computes a generalized Gauss-Laguerre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) x^alpha exp(-x) dx.\n";

  for ( test = 0; test < TEST_NUM; test++ )
  {
    alpha = alpha_test[test];

    for ( order = 1; order <= order_max; order++ )
    {
      cout << "\n";
      cout << "  Order = " << order << "\n";
      cout << "  ALPHA = " << alpha << "\n";

      w = new double[order];
      x = new double[order];

      gen_laguerre_compute ( order, alpha, x, w );
 
      for ( i = 0; i < order; i =i + 1 )
      {
        cout << "  " << setw(8) << i
             << "  " << setprecision(16) << setw(24) << x[i]
             << "  " << setprecision(16) << setw(24) << w[i] << "\n";
      }
      delete [] w;
      delete [] x;
    }
  }
 
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test11 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST11 tests HERMITE_COMPUTE against HERMITE_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST11\n";
  cout << "  HERMITE_COMPUTE computes a Gauss-Hermite rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -oo < x < +oo ) f(x) exp(-x*x) dx.\n";
  cout << "\n";
  cout << "  HERMITE_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Estimate       Exact            Error\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";

    f = new double[order];
    w = new double[order];
    x = new double[order];

    hermite_compute ( order, x, w );

    for ( n = 0; n <= 2 * order + 2; n = n + 1 )
    {
      exact = hermite_integral ( n );
 
      if ( n == 0 )
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = 1.0;
        }
      }
      else
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = pow ( x[i], n );
        }
      }
      estimate = 0.0;
      for ( i = 0; i < order; i++ )
      {
        estimate = estimate + w[i] * f[i];
      }

      error = r8_abs ( exact - estimate );
  
      cout << "  " << setw(8)  << order
           << "  " << setw(8)  << n
           << "  " << setprecision(6) << setw(14) << estimate
           << "  " << setprecision(6) << setw(14) << exact
           << "  " << setprecision(6) << setw(14) << error << "\n";
    }
    delete [] f;
    delete [] w;
    delete [] x;
  }
 
  return;
}
//****************************************************************************80

void test12 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST12 tests HERMITE_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST12\n";
  cout << "  HERMITE_COMPUTE computes a Gauss-Hermite rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -oo < x < +oo ) f(x) exp(-x*x) dx.\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";
    cout << "  Order = " << order << "\n";

    w = new double[order];
    x = new double[order];

    hermite_compute ( order, x, w );
 
    for ( i = 0; i < order; i =i + 1 )
    {
      cout << "  " << setw(8) << i
           << "  " << setprecision(16) << setw(24) << x[i]
           << "  " << setprecision(16) << setw(24) << w[i] << "\n";
    }
    delete [] w;
    delete [] x;
  }
 
  return;
}
//****************************************************************************80

void test13 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST13 tests JACOBI_COMPUTE against JACOBI_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double beta;
  double beta_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  int test1;
  int test2;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST13\n";
  cout << "  JACOBI_COMPUTE computes a Gauss-Jacobi rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) (1-x)^alpha (1+x)^beta dx.\n";
  cout << "\n";
  cout << "  JACOBI_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Alpha           Beta            Estimate       Exact            Error\n";

  for ( test1 = 0; test1 < TEST_NUM; test1++ )
  {
    alpha = alpha_test[test1];

    for ( test2 = 0; test2 < TEST_NUM; test2++ )
    {
      beta = beta_test[test2];

      for ( order = 1; order <= order_max; order++ )
      {
        cout << "\n";

        f = new double[order];
        w = new double[order];
        x = new double[order];

        jacobi_compute ( order, alpha, beta, x, w );

        for ( n = 0; n <= 2 * order + 2; n = n + 1 )
        {
          exact = jacobi_integral ( n, alpha, beta );
 
          if ( n == 0 )
          {
            for ( i = 0; i < order; i++ )
            {
              f[i] = 1.0;
            }
          }
          else
          {
            for ( i = 0; i < order; i++ )
            {
              f[i] = pow ( x[i], n );
            }
          }
          estimate = 0.0;
          for ( i = 0; i < order; i++ )
          {
            estimate = estimate + w[i] * f[i];
          }
 
          error = r8_abs ( exact - estimate );
  
          cout << "  " << setw(8)  << order
               << "  " << setw(8)  << n
               << "  " << setprecision(6) << setw(14) << alpha
               << "  " << setprecision(6) << setw(14) << beta
               << "  " << setprecision(6) << setw(14) << estimate
               << "  " << setprecision(6) << setw(14) << exact
               << "  " << setprecision(6) << setw(14) << error << "\n";
        }
        delete [] f;
        delete [] w;
        delete [] x;
      }
    }
  }
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test14 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST14 tests JACOBI_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
# define TEST_NUM 3

  double alpha;
  double alpha_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  double beta;
  double beta_test[TEST_NUM] = { 0.5, 1.0, 2.5 };
  int i;
  int order;
  int order_max = 10;
  int test1;
  int test2;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST14\n";
  cout << "  JACOBI_COMPUTE computes a Gauss-Jacobi rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) (1-x)^alpha (1+x)^beta dx.\n";

  for ( test1 = 0; test1 < TEST_NUM; test1++ )
  {
    alpha = alpha_test[test1];

    for ( test2 = 0; test2 < TEST_NUM; test2++ )
    {
      beta = beta_test[test2];

      for ( order = 1; order <= order_max; order++ )
      {
        cout << "\n";
        cout << "  Order = " << order << "\n";
        cout << "  ALPHA = " << alpha << "\n";
        cout << "  BETA = "  << beta << "\n";

        w = new double[order];
        x = new double[order];

        jacobi_compute ( order, alpha, beta, x, w );

        for ( i = 0; i < order; i =i + 1 )
        {
          cout << "  " << setw(8) << i
               << "  " << setprecision(16) << setw(24) << x[i]
               << "  " << setprecision(16) << setw(24) << w[i] << "\n";
        }
        delete [] w;
        delete [] x;
      }
    }
  }
  return;
# undef TEST_NUM
}
//****************************************************************************80

void test15 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST15 tests LAGUERRE_COMPUTE against LAGUERRE_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST15\n";
  cout << "  LAGUERRE_COMPUTE computes a Gauss-Laguerre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) exp(-x) dx.\n";
  cout << "\n";
  cout << "  LAGUERRE_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Estimate       Exact            Error\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";

    f = new double[order];
    w = new double[order];
    x = new double[order];

    laguerre_compute ( order, x, w );

    for ( n = 0; n <= 2 * order + 2; n = n + 1 )
    {
      exact = laguerre_integral ( n );

      if ( n == 0 )
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = 1.0;
        }
      }
      else
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = pow ( x[i], n );
        }
      }
      estimate = 0.0;
      for ( i = 0; i < order; i++ )
      {
        estimate = estimate + w[i] * f[i];
      }

      error = r8_abs ( exact - estimate );
  
      cout << "  " << setw(8)  << order
           << "  " << setw(8)  << n
           << "  " << setprecision(6) << setw(14) << estimate
           << "  " << setprecision(6) << setw(14) << exact
           << "  " << setprecision(6) << setw(14) << error << "\n";
    }
    delete [] f;
    delete [] w;
    delete [] x;
  }
  return;
}
//****************************************************************************80

void test16 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST16 tests LAGUERRE_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double alpha;
  int i;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST16\n";
  cout << "  LAGUERRE_COMPUTE computes a generalized Gauss-Laguerre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( 0 <= x < +oo ) f(x) exp(-x) dx.\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";
    cout << "  Order = " << order << "\n";

    w = new double[order];
    x = new double[order];

    laguerre_compute ( order, x, w );
 
    for ( i = 0; i < order; i =i + 1 )
    {
      cout << "  " << setw(8) << i
           << "  " << setprecision(16) << setw(24) << x[i]
           << "  " << setprecision(16) << setw(24) << w[i] << "\n";
    }
    delete [] w;
    delete [] x;
  }
 
  return;
}
//****************************************************************************80

void test17 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST17 tests LEGENDRE_COMPUTE against LEGENDRE_INTEGRAL.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  double error;
  double estimate;
  double exact;
  double *f;
  int i;
  int n;
  int order;
  int order_max = 10;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST17\n";
  cout << "  LEGENDRE_COMPUTE computes a Gauss-Legendre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x)  dx.\n";
  cout << "\n";
  cout << "  LEGENDRE_INTEGRAL determines the exact value of\n";
  cout << "  this integal when f(x) = x^n.\n";
  cout << "\n";
  cout << "  A rule of order ORDER should be exact for monomials X^N\n";
  cout << "  up to N = 2*ORDER-1\n";
  cout << "\n";
  cout << "  In the following table, for each order, the LAST THREE estimates\n";
  cout << "  are made on monomials that exceed the exactness limit for the rule.\n";
  cout << "\n";
  cout << "     Order         N       Estimate       Exact            Error\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";

    f = new double[order];
    w = new double[order];
    x = new double[order];

    legendre_compute ( order, x, w );

    for ( n = 0; n <= 2 * order + 2; n = n + 1 )
    {
      exact = legendre_integral ( n );

      if ( n == 0 )
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = 1.0;
        }
      }
      else
      {
        for ( i = 0; i < order; i++ )
        {
          f[i] = pow ( x[i], n );
        }
      }
      estimate = 0.0;
      for ( i = 0; i < order; i++ )
      {
        estimate = estimate + w[i] * f[i];
      }

      error = r8_abs ( exact - estimate );
  
      cout << "  " << setw(8)  << order
           << "  " << setw(8)  << n
           << "  " << setprecision(6) << setw(14) << estimate
           << "  " << setprecision(6) << setw(14) << exact
           << "  " << setprecision(6) << setw(14) << error << "\n";
    }
    delete [] f;
    delete [] w;
    delete [] x;
  }
  return;
}
//****************************************************************************80

void test18 ( void )

//****************************************************************************80
//
//  Purpose:
//
//    TEST18 tests LEGENDRE_COMPUTE.
//
//  Licensing:
//
//    This code is distributed under the GNU LGPL license. 
//
//  Modified:
//
//    26 February 2008
//
//  Author:
//
//    John Burkardt
//
{
  int i;
  int order;
  int order_max = 10;
  int test;
  double *w;
  double *x;

  cout << "\n";
  cout << "TEST18\n";
  cout << "  LEGENDRE_COMPUTE computes a Gauss-Legendre rule\n";
  cout << "  which is appropriate for integrands of the form\n";
  cout << "    Integral ( -1 <= x <= +1 ) f(x) dx.\n";

  for ( order = 1; order <= order_max; order++ )
  {
    cout << "\n";
    cout << "  Order = " << order << "\n";

    w = new double[order];
    x = new double[order];

    legendre_compute ( order, x, w );
 
    for ( i = 0; i < order; i =i + 1 )
    {
      cout << "  " << setw(8) << i
           << "  " << setprecision(16) << setw(24) << x[i]
           << "  " << setprecision(16) << setw(24) << w[i] << "\n";
    }
    delete [] w;
    delete [] x;
  }
 
  return;
}
