/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2007, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file ResponseSet.h
 *
 * Defines the colin::ResponseSet class.
 **/
 
#ifndef colin_ResponseSet_h
#define colin_ResponseSet_h

#include <list>
 
//#include <acro_config.h>
//#include <utilib/std_headers.h>
//#include <colin/real.h>


namespace colin {

namespace ERR {
const int ResponseSet_NULL       = -1001;
const int ResponseSet_NoResponse = -1002;
};


/** A class for storing a set of responses calculated for a single
 *  problem state vector, and for calculating statistics on the
 *  responses.
 *
 *  This class makes use of a mix of the envelope/letter and smart
 *  pointer idioms so that responses can be bantered around using
 *  shallow copies, but without the function overhead of the "standard"
 *  envelope/letter or Cheshire Cat idioms.
 */
template<class DomainT, class ResponseT>
class ResponseSet 
  {
  public:
    //
    //- Heading: Class-specific interface typedefs & enums
    //
    typedef unsigned long   id_t;

    enum Statistics
      {
      NEWEST,
      MEAN,
      MEDIAN,
      DEVIATION
      };

  public:
    //
    //- Heading: Constructors, destructors, and whatnot
    //

    /// default constructor (initializes an empty envelope)
    ResponseSet()
      : m_members(NULL)
      {}

    /// default destructor (delete letter if this is the last reference)
    ~ResponseSet()
      {
      if ( m_members == NULL )
        { return; }

      if ( --(m_members->refCount) == 0 )
        { delete m_members; }
      }

    /// Standard envelope constructor: initializes to a new point
    ResponseSet(DomainT &point, double convergence_factor = 1.0)
      : m_members(new ImplData)
      {
      ImplData &m = *m_members;
      m.id = nextID();
      m.refCount = 1;
      m.point = point;
      m.convergence_factor = convergence_factor;
      }

    /// Copy constructor (copy Impl pointer & increment reference)
    ResponseSet(const ResponseSet &src)
      {
      if ( (m_members = src.m_members) != NULL )
        { ++(m_members->refCount); }
      }

    /// Assignment operator (transfer envelope to a new letter)
    ResponseSet& operator=(const ResponseSet& src)
      {
      // forget old letter
      if ( m_members != NULL )
        {
        if ( --(m_members->refCount) == 0 )
          { delete m_members; }
        }
      // grab the new letter
      if ( (m_members = src.m_members) != NULL )
        { ++(m_members->refCount); }

      return *this;
      }


    //
    //- Heading: Comparison operators
    //
    bool operator== (ResponseSet i) const
      { return (m_members == i.m_members) && (m_members != NULL); }
    bool operator!= (ResponseSet i) const
      { return m_members != i.m_members || (m_members == NULL); }
    bool operator< (ResponseSet i) const
      { return (( m_members == NULL ? 0 : m_members->id ) 
                < ( i.m_members == NULL ? 0 : i.m_members->id )); }
    bool operator<= (ResponseSet i) const
      { return (( m_members == NULL ? 0 : m_members->id ) 
                <= ( i.m_members == NULL ? 0 : i.m_members->id )); }
    bool operator> (ResponseSet i) const
      { return (( m_members == NULL ? 0 : m_members->id ) 
                > ( i.m_members == NULL ? 0 : i.m_members->id )); }
    bool operator>= (ResponseSet i) const
      { return (( m_members == NULL ? 0 : m_members->id ) 
                >= ( i.m_members == NULL ? 0 : i.m_members->id )); }

    //
    //- Heading: Comparison operators
    //
    /** I would make this a normal operator<<, but for some (unknown)
     *  reason, that causes StdOptSolver<>::debug_io() to fail to
     *  compile on an "ambiguous overload" error.
     */
    std::ostream& write( std::ostream &os )
      { 
      if ( m_members == NULL )
        {
        os << "Invalid Sesponse Set" << std::endl;
        return os;
        }
      ImplData &m = *m_members;
  
      os << "Response Set id " << m.id << "  (" << m.refCount << " refs)" 
         << std::endl << m.point << std::endl
         << m.responses.size() << " responses" << std::endl;
      
      typename std::list<ResponseT>::iterator it = m.responses.begin();
      typename std::list<ResponseT>::iterator itEnd = m.responses.end();
      for( ; it != itEnd; os << *it++ );
      return os;
      }


    //
    //- Heading: Member functions
    //

    /// Return true if this set is NOT initialized to a valid point
    bool invalid()
      { return m_members == NULL; }

    /// Return a unique ID for this response set (0 if not valid)
    id_t id()
      { 
      assert(m_members != NULL);
      return m_members->id; 
      }

    /// Return the state point for this response set
    const DomainT& point()
      { 
      assert(m_members != NULL);
      return m_members->point; 
      }

    /// Return the convergence factor for this response set
    double convergenceFactor()
      { 
      assert(m_members != NULL);
      return m_members->convergence_factor; 
      }

    /// Store a new response into this response set
    void new_response(ResponseT& response)
      { 
      assert(m_members != NULL);
      m_members->responses.push_back(response); 
      }

    /// Return the number of responses in the response set
    size_t response_count()
      { 
      assert(m_members!=NULL);
      return m_members->responses.size(); 
      }

    /// Calculate statistics on the response set & return result in 'response'
    int stat(ResponseT &response, Statistics stat = NEWEST)
      {
      assert((stat==NEWEST)&&"ResponseSet currently only supports NEWEST");

      if ( m_members == NULL )
        { return ERR::ResponseSet_NULL; }

      ImplData &m = *m_members;
      if ( m.responses.empty() )
        { return ERR::ResponseSet_NoResponse; }

      if ( stat == NEWEST )
        {
        // return the response most recently added
        response = m.responses.back();
        }

      return 0;
      }

  private:
    static id_t nextID()
      {
      static id_t lastID = 0;
      return ++lastID;
      }

    struct ImplData {
      id_t                  id;
      int                   refCount;
      double                convergence_factor;
      DomainT               point;
      std::list<ResponseT>  responses;
      };
    
    /// pointer to member data (i.e. the letter contents)
    ImplData    *m_members;
  };


};


#endif // colin_ResponseSet_h
