/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- class:       Dakota::Map<T1, T2>
//- Description: Inherits from the STL map class in order to add Dakota
//-              specific methods.
//-
//- Assumptions: 
//-              
//- Owner:       Mike Eldred
//- Version: $Id

#ifndef DAKOTA_MAP_H
#define DAKOTA_MAP_H

#include "template_defs.h"
#include "global_defs.h"
#include "MPIPackBuffer.H"


namespace Dakota {

/// Template class for the Dakota bookkeeping map

/** The Map template class is a sorted associative container class for
    Dakota.  It inherits from the STL map class in order to add Dakota
    specific methods. */

template<typename T1, typename T2>
class Map: public DAKOTA_BASE_MAP<T1, T2>
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  Map();                     ///< Default constructor
  Map(const Map<T1, T2>& a); ///< Copy constructor
  ~Map();                    ///< Destructor

#ifdef HAVE_MEMBER_TEMPLATES
  /// Range constructor (member template)
  template <class InputIter>
  Map(InputIter first, InputIter last);
#endif // HAVE_MEMBER_TEMPLATES

  /// assignment operator
  Map<T1, T2>& operator=(const Map<T1, T2>& a);

  //
  //- Heading: Member functions 
  //

  /// Writes a Map to an output stream
  void write(ostream& s) const;

  /// Reads a Map from an MPIUnpackBuffer after an MPI receive
  void read(MPIUnpackBuffer& s);

  /// Writes a Map to a MPIPackBuffer prior to an MPI send
  void write(MPIPackBuffer& s) const;

  // Functions needed by DAKOTA and not supported by STL maps:

  /// Returns the number of items that are currently in the map
  size_t entries() const;

  // Removes and returns the item at the specified index
  //T removeAt(size_t index);

  // Removes the specified item from the map
  //bool remove(const T1& a);
    
  // Returns TRUE if map contains object a, returns FALSE otherwise
  //bool contains(const T1& a) const;
};



template <typename T1, typename T2> 
inline Map<T1, T2>::Map()
{}


template <typename T1, typename T2> 
inline Map<T1, T2>::Map(const Map<T1, T2>& a)
{ 
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_MAP<T1, T2>::operator=(a); // base class assignment operator
}


template <typename T1, typename T2> 
inline Map<T1, T2>::~Map()
{}


#ifdef HAVE_MEMBER_TEMPLATES
template <typename T1, typename T2> template <class InputIter>
Map<T1, T2>::Map(InputIter first, InputIter last):
  DAKOTA_BASE_MAP<T1, T2>(first, last)
{}
#endif // HAVE_MEMBER_TEMPLATES


template <typename T1, typename T2> 
inline Map<T1, T2>& Map<T1, T2>::operator=(const Map<T1, T2>& a)
{
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_MAP<T1, T2>::operator=(a); // base class assignment operator
  return *this;
}


template <typename T1, typename T2>
void Map<T1, T2>::write(ostream& s) const
{
  for (typename Map<T1, T2>::const_iterator cit
	 = DAKOTA_BASE_MAP<T1, T2>::begin();
       cit != DAKOTA_BASE_MAP<T1, T2>::end(); cit++)
    s << "                     " << *cit << '\n';
}


/// global ostream insertion operator for Map
template <typename T1, typename T2>
inline ostream& operator<<(ostream& s, const Map<T1, T2>& data)
{
  data.write(s);
  return s;
}


template <typename T1, typename T2>
void Map<T1, T2>::read(MPIUnpackBuffer& s)
{
  DAKOTA_BASE_MAP<T1, T2>::clear();
  size_t len;
  s >> len;
  for (size_t i=0; i<len; i++){
    T data; // fresh allocation needed in case T is ref-counted
    s >> data; 
    DAKOTA_BASE_MAP<T1, T2>::push_back(data);
  }
}


template <typename T1, typename T2>
void Map<T1, T2>::write(MPIPackBuffer& s) const
{
  size_t len = DAKOTA_BASE_MAP<T1, T2>::size();
  s << len;
  for (typename Map<T1, T2>::const_iterator cit
	 = DAKOTA_BASE_MAP<T1, T2>::begin();
       cit != DAKOTA_BASE_MAP<T1, T2>::end(); cit++)
    s << *cit;
}


/// global MPIUnpackBuffer extraction operator for Map
template <typename T1, typename T2>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, Map<T1, T2>& data)
{
  data.read(s);
  return s;
}


/// global MPIPackBuffer insertion operator for Map
template <typename T1, typename T2>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const Map<T1, T2>& data)
{
  data.write(s);
  return s;
}


template <typename T1, typename T2> 
inline size_t Map<T1, T2>::entries() const
{ return DAKOTA_BASE_MAP<T1, T2>::size(); }


/** Removes the item at the index specified.  Uses the STL advance()
    function to step to the appropriate position in the map and then
    calls the map::erase() method. */
template <typename T1, typename T2> 
T Map<T1, T2>::removeAt(size_t index)
{
  if (index >= DAKOTA_BASE_MAP<T1, T2>::size()) {
    Cerr << "Error: index passed to Map<T1, T2>::remoteAt() is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  typename Map<T1, T2>::iterator it = DAKOTA_BASE_MAP<T1, T2>::begin();
  advance(it, index);
  T tempT(*it);
  DAKOTA_BASE_MAP<T1, T2>::erase(it);
  return tempT;
}


/** Removes the first instance matching object a from the map (and
    therefore differs from the STL map::remove() which removes all
    instances).  Uses the STL find() algorithm to find the object and
    the map::erase() method to perform the remove.*/
template <typename T1, typename T2> 
bool Map<T1, T2>::remove(const T& a)
{
  typename Map<T1, T2>::iterator it
    = std::find(DAKOTA_BASE_MAP<T1, T2>::begin(),
		DAKOTA_BASE_MAP<T1, T2>::end(), a);
  if (it != DAKOTA_BASE_MAP<T1, T2>::end()) {
    DAKOTA_BASE_MAP<T1, T2>::erase(it);
    return true;
  }
  else
    return false;
}


/** Uses the STL find() algorithm to locate the first instance of
    object a.  Returns true if an instance is found. */
template <typename T1, typename T2>
inline bool Map<T1, T2>::contains(const T& a) const
{
  return (std::find(DAKOTA_BASE_MAP<T1, T2>::begin(),
		    DAKOTA_BASE_MAP<T1, T2>::end(), a)
	  != DAKOTA_BASE_MAP<T1, T2>::end()) ? true : false;
}

} // namespace Dakota

#endif //DAKOTA_MAP_H
