/*  _________________________________________________________________________
 *
 *  Coliny: A Library of COLIN optimizers
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top Coliny directory.
 *  _________________________________________________________________________
 */

//
// APPS.cpp
//
 
#include <acro_config.h>
#ifdef ACRO_USING_APPSPACK

#include <coliny/Factory.h>
#include <utilib/_math.h>
#include <coliny/APPS.h>
using namespace std;
#include "APPSPACK_Executor_Interface.hpp"
#include "APPSPACK_Parameter_List.hpp"
#include "APPSPACK_Constraints_Linear.hpp"
#include "APPSPACK_Solver.hpp"
#include "APPSPACK_Float.hpp"

#define COORDINATE_BASIS                 0
#define SIMPLEX_BASIS                    1
#define POSITIVE_COORDINATE_BASIS        2
#define SHIFTED_SIMPLEX_BASIS            3


namespace coliny  {

namespace ColinyAPPS {

/// Interface to objective function
class FevalMgr : public APPSPACK::Executor::Interface {

public:

  //! Constructor
  FevalMgr(APPS* _apps_ptr) : apps_ptr(_apps_ptr) {}

  //! Destructor
  virtual ~FevalMgr() { }

  //
  bool isWaiting() const
        {
        for (size_type i=0; i<task.size(); i++)
          if (task[i].id == -1) return true;
        return false;
        }

  // See base class
  bool spawn(const APPSPACK::Vector& x, int tag_in );

  // See base class
  int recv(int& tag_out, APPSPACK::Vector& f_out, string& msg_out);

  //! Pointer to COLIN problem
  colin::OptProblem<vector<double> >* problem;

  ///
  vector<double> tmp_x;

  ///
  vector<TaskStatus> task;

  ///
  APPS* apps_ptr;

  ///
  void reset();
};

bool FevalMgr::spawn(const APPSPACK::Vector& x, int tag_in)
{
//
// Get the next free task
//
int ndx=-1;
for (unsigned int i=0; i<task.size(); i++) {
  if (task[i].free) {
     ndx=i;
     break;
     }
  }
//
// Return false if all workers are busy
//
if (ndx == -1)
   return false;
//
// Setup Task
//
for (int i=0; i<x.size(); i++)
  tmp_x[i] = x[i];
if (problem->numNonlinearConstraints() == 0) {
   problem->AsyncEvalF(tmp_x, &(task[ndx].fval));
   task[ndx].id = problem->last_id()+1;
   }
else {
   int priority=1;
   if (problem->numNonlinearConstraints() > 0)
      problem->AsyncEval(tmp_x, priority, task[ndx].response, colin::mode_f|colin::mode_cf);
   else
      problem->AsyncEval(tmp_x, priority, task[ndx].response,colin::mode_f);
   task[ndx].id = task[ndx].response->info->id + 1;
   }
task[ndx].free = false;
task[ndx].tag = tag_in;

return true;
}


int FevalMgr::recv(int& tag_out, APPSPACK::Vector& f_out, string& msg_out)
{
//
// Get the id of the next evaluated spawn
//
int id = 1+ problem->next_eval();
if (id == 0) return 0;
//
// Find the index of this id
//
int ndx=-1;
for (unsigned int i=0; i<task.size(); i++) {
  if (task[i].id == id) {
     ndx=i;
     break;
     }
  }
if (ndx == -1)
   EXCEPTION_MNGR(runtime_error, "FevalMngr::recv - no tasks have id=" << id);
//
// Process the return values
//
if (task[ndx].failed) {
   f_out.resize(0);
   }
else {
   f_out.resize(1);
   if (problem->numNonlinearConstraints() == 0)
      f_out[0] = task[ndx].fval;
   else {
      real tmp_f,tmp_violation;
      //cerr << task[ndx].response->function_value() << endl;
      //cerr << task[ndx].response->constraint_values();
      //cerr << endl;

      apps_ptr->compute_response_info(*(task[ndx].response), problem->state->constraint_lower_bounds,  problem->state->constraint_upper_bounds, tmp_f, tmp_violation);
      f_out[0] = tmp_f;
   
      //cerr << tmp_f << endl;
      }
   }

task[ndx].free=true;
task[ndx].id = -1;
tag_out = task[ndx].tag;
return id;
}


void FevalMgr::reset()
{
tmp_x.resize(problem->num_real_params());
int nevals = max((unsigned int)1,problem->num_evaluation_servers());
if (task.size() != static_cast<unsigned int>(nevals)) {
   task.resize(nevals);
   for (size_type i=0; i<task.size(); i++)
     if (! task[i].response)
        task[i].response = new colin::AppResponse<>(1,
					problem->numNonlinearConstraints(),
					problem->num_real_params());
   }
}


} // namespace ColinyAPPS



APPS::~APPS()
{
delete feval_mgr;
delete params;
}

APPS::APPS()
 : auto_rescale(true),
   step_tolerance(1.0e-5),
   alpha(0.0),
   //pattern("coordinate"),
   num_search_directions(-1),
   //num_active_directions(-1),
   profile(0),
   //inc_flag(true),
   contraction_factor(0.5),
   initial_step(1.0),
   min_step_allowed(2.0e-5),
   synch_flag(true),
   batch_str("sync")
{
opt_name="APPS";

feval_mgr = new ColinyAPPS::FevalMgr(this);
feval_mgr->problem = &problem;

params = new APPSPACK::Parameter::List;
disable_parameter("ftol");
disable_parameter("max_iters");
disable_parameter("max_time");
disable_parameter("constant_constraint_penalty");

utilib::ParameterSet::create_parameter("batch_eval",batch_str,
"<str>","sequential",
"Defines how the algorithm is parallelized when two or more\n"
"\t  function evaluations could be computed simultaneously.\n"
"\t  sync: All function evaluations are performed synchronously.\n"
"\t  async: Function evaluations are performed asynchronously.\n"
"\t     Evaluations are recorded until an improving point is generated or\n"
"\t     all of them have completed.\n"
);

ParameterSet::create_categorized_parameter("auto_rescale",auto_rescale,
        "<bool>","true",
        "If true, then automatically rescale the search for bound-constrained\n""\t  problems. The initial scale is 10% of the range in each dimension.",
        "Step Length Control");

ParameterSet::create_categorized_parameter("initial_step",initial_step, 
	"<double>","1.0",
	"Initial step length",
	"Step Length Control");
ParameterSet::alias_parameter("initial_step","initial_stepsize");
ParameterSet::alias_parameter("initial_step","initial_steplength");

ParameterSet::create_categorized_parameter("step_tolerance",step_tolerance, 
	"<double>","1e-5",
	"Convergence tolerance step length",
	"Termination");

ParameterSet::create_categorized_parameter("contraction_factor",
	contraction_factor,
	"<double>","0.5",
	"Contraction factor",
	"Step Length Control");

ParameterSet::create_categorized_parameter("min_step_allowed",min_step_allowed,
	"<double>","2x tol",
	"Minimum step length allowed",
	"Step Length Control");

ParameterSet::create_categorized_parameter("alpha",alpha, "<double>","0.0",
	"Sufficient decrease parameter",
	"Step Length Control");

ParameterSet::create_parameter("profile",profile, 
	"<int>","0",
	"Profile level");

}


// publish options to APPS through command line arguments.
void APPS::reset()
{
if (!problem) return;

colin::OptSolver<vector<double> >::reset();

if (debug > 1000) profile=10;

int tmp_neval;
if (max_neval > 0) {
   tmp_neval = max(max_neval-problem.neval(),0);
   if (max_neval_curr != 0)
      tmp_neval = min(tmp_neval, max_neval_curr);
   }
else
   tmp_neval = max_neval_curr;

if (tmp_neval > 0)
   params->setParameter("Maximum Evaluations",tmp_neval);
params->setParameter("Function Tolerance",accuracy);
params->setParameter("Sufficient Decrease Factor",alpha);
params->setParameter("Contraction Factor",contraction_factor);
params->setParameter("Initial Step",initial_step);
params->setParameter("Debug",profile);
params->setParameter("Minimum Step",min_step_allowed);
params->setParameter("Step Tolerance",step_tolerance);
params->setParameter("Bounds Tolerance",constraint_tolerance);
///params->setParameter("Solution File","appspack_solution_file");

if (batch_str == "sync")
   synch_flag = true;
else if (batch_str == "async")
   synch_flag = false;
else
   EXCEPTION_MNGR(runtime_error,"APPS::reset - bad value for batch_eval parameter: " << batch_str);
params->setParameter("Synchronous",synch_flag);

//params->setParameter("inc",inc_flag);
//params->setParameter("Maximum Step",max_step_allowed);
//params->setParameter("profile",profile);
//params->setParameter("tol",step_tolerance);

if (problem.num_real_params() > 0) {
   APPSPACK::Vector  isLower, isUpper;
   isLower.resize(problem.num_real_params());
   isUpper.resize(problem.num_real_params());

   APPSPACK::Vector scale(problem.num_real_params());
   for (int i=0; i<scale.size(); i++)
     scale[i] = 1.0;
   if (problem.enforcing_bounds()) {
      APPSPACK::Vector l_bound, u_bound;
      l_bound.resize(problem.num_real_params());
      u_bound.resize(problem.num_real_params());
      utilib::pvector<real> L_bound, U_bound;
      double DNE = APPSPACK::dne();
      for (int i=0; i<l_bound.size(); i++) {
        l_bound[i] = DNE;
        u_bound[i] = DNE;
        }
      problem.get_real_bounds(L_bound,U_bound);

      for (int i=0; i<isLower.size(); i++) {
        l_bound[i] = L_bound[i];
        u_bound[i] = U_bound[i];
        if ((L_bound[i] == real::negative_infinity) ||
	    (problem.real_lower_bound_type(i) != colin::hard_bound))
           isLower[i] = 0.0;
        else
           isLower[i] = 1.0;
        if ((U_bound[i] == real::positive_infinity) ||
	    (problem.real_upper_bound_type(i) != colin::hard_bound))
           isUpper[i] = 0.0;
        else
           isUpper[i] = 1.0;
        if (auto_rescale && (isLower[i] == 1.0) && (isUpper[i] == 1.0))
           scale[i] = max((u_bound[i]-l_bound[i])*0.1,1e-5);
        }
      //params->setParameter("Is Lower",isLower);
      //params->setParameter("Is Upper",isUpper);
      params->setParameter("Lower",l_bound);
      params->setParameter("Upper",u_bound);
      }
   params->setParameter("Scaling",scale);
   }
   

if ((num_search_directions == -1) && (problem.num_real_params() > 0))
   num_search_directions = problem.num_real_params()*2;
   params->setParameter("search",num_search_directions);

/*
if (pattern != "coordinate")
   EXCEPTION_MNGR(runtime_error,"APPS::reset - must specify coorindate pattern for bound-constrained problems.")

if (pattern == "coordinate")
   params->setParameter("pattern",0);
else if (pattern == "simplex")
   params->setParameter("pattern",1);
else if (pattern == "positive-coordinate")
   params->setParameter("pattern",2);
else if (pattern == "shifted-simplex")
   params->setParameter("pattern",3);
else
   EXCEPTION_MNGR(runtime_error,"APPS::reset - bad pattern: " << pattern)
*/

feval_mgr->reset();

colin::AppResponseAnalysis::initialize(problem.numNonlinearIneqConstraints(),constraint_tolerance);
}



void APPS::minimize()
{
//
// Call the optimizer's initializer
//
opt_init();
if (!initial_point_flag)
   EXCEPTION_MNGR(runtime_error,"APPS::minimize - no initial point specified.");
if (problem.num_real_params() != best().point.size())
   EXCEPTION_MNGR(runtime_error,"APPS::minimize - problem has " <<
        problem.num_real_params() << " real params, but initial point has " << best().point.size() );
if (best().point.size() == 0) {
   best().termination_info = "No-Real-Params";
   return;
   }

if (problem.numNonlinearConstraints() > 0)
   problem.Eval(best().point,best().response,colin::mode_f|colin::mode_cf);
else
   problem.Eval(best().point,best().response,colin::mode_f);
compute_response_info(best().response,problem.state->constraint_lower_bounds,  problem.state->constraint_upper_bounds, best().value(),best().constraint_violation);
//
// Call APPS
//
APPSPACK::Vector init_point(problem.num_real_params());
for (int i=0; i<init_point.size(); i++)
  init_point[i] = best().point[i];
params->setParameter("Initial X",init_point);
APPSPACK::Constraints::Linear linear(*params);
APPSPACK::Solver solver(*params, *feval_mgr, linear);
APPSPACK::Solver::State state = solver.solve();
//
// Set the minimum point and value
//
best().point = solver.getBestX();
if (problem.numNonlinearConstraints() > 0)
   problem.Eval(best().point,best().response,colin::mode_f|colin::mode_cf);
else
   problem.Eval(best().point,best().response,colin::mode_f);
compute_response_info(best().response,problem.state->constraint_lower_bounds,  problem.state->constraint_upper_bounds, best().value(),best().constraint_violation);
//
//
//
if (state == APPSPACK::Solver::StepConverged)
   best().termination_info = "Step-Length";

else if (state == APPSPACK::Solver::FunctionConverged)
   best().termination_info = "Accuracy";

else if (state == APPSPACK::Solver::EvaluationsExhausted)
   best().termination_info = "Max-Num-Evals";

else
   best().termination_info = "Unknown";
}


void APPS::write(ostream& os) const
{
colin::OptSolver<vector<double> >::write(os);
}

} // namespace coliny

FACTORY_REGISTER(apps, colin::OptSolver<std::vector<double> >*, return new coliny::APPS();, "An alias to APPS")
FACTORY_REGISTER(APPS, colin::OptSolver<std::vector<double> >*, return new coliny::APPS();, "The APPSPACK pattern search optimizer")

#endif
