/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Definition of class DesignTarget.

    NOTES:

        See notes under section "Class Definition" of this file.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Mon May 19 10:20:53 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the definition of the DesignTarget class.
 */




/*
================================================================================
Prevent Multiple Inclusions
================================================================================
*/
#ifndef JEGA_UTILITIES_DESIGNTARGET_HPP
#define JEGA_UTILITIES_DESIGNTARGET_HPP







/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <vector>
#include <utilities/include/int_types.hpp>






/*
================================================================================
Pre-Namespace Forward Declares
================================================================================
*/







/*
================================================================================
Namespace Using Directives - DISCOURAGED!!
================================================================================
*/







/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {





/*
================================================================================
In-Namespace Forward Declares
================================================================================
*/
class Design;
class DesignTarget;
class RegionOfSpace;
class ConstraintInfo;
class DesignDVSortSet;
class DesignVariableInfo;
class ObjectiveFunctionInfo;




/*
================================================================================
In Namespace File Scope Typedefs
================================================================================
*/
/**
 * \brief A container for all the information objects describing a set of
 *        Design varaibles.
 */
typedef
std::vector<DesignVariableInfo*>
DesignVariableInfoVector;

/**
 * \brief A container for all the information objects describing a set of
 *        Constraints.
 */
typedef
std::vector<ConstraintInfo*>
ConstraintInfoVector;

/**
 * \brief A container for all the information objects describing a set of
 *        objective functions.
 */
typedef
std::vector<ObjectiveFunctionInfo*>
ObjectiveFunctionInfoVector;



/*
================================================================================
Class Definition
================================================================================
*/

/// An interface class providing information about a problem.
/**
 * This class is used by many of the classes in this application.  It provides
 * needed information about a problem such as counts and information objects.
 * It also provides capabilities such as new Design creation, feasibility
 * checking, discarded design storage, etc.
 */
class JEGA_SL_IEDECL DesignTarget
{
    /*
    ===========================================================================
    Class Scope Typedefs
    ===========================================================================
    */
    public:


    protected:


    private:


    /*
    ===========================================================================
    Member Data Declarations
    ===========================================================================
    */
    private:



    /*
    ===========================================================================
    Mutators
    ===========================================================================
    */
    public:

        virtual
        void
        SetTrackDiscards(
            bool use
            ) = 0;


    /*
    ===========================================================================
    Accessors
    ===========================================================================
    */
    public:

        virtual
        bool
        GetTrackDiscards(
            ) const = 0;



    /*
    ===========================================================================
    Public Methods
    ===========================================================================
    */
    public:





    /*
    ===========================================================================
    Subclass Visible Methods
    ===========================================================================
    */
    protected:





    /*
    ===========================================================================
    Subclass Overridable Methods
    ===========================================================================
    */
    public:

        /**
         * \brief Override to provide protected access to the discarded
         *        Design's stored in this.
         *
         * The protection should last until a call to CheckinDiscards is
         * issued.
         *
         * \return The set of Designs passed to this Target by way of the
         *         TakeDesign* methods.
         */
        virtual
        const DesignDVSortSet&
        CheckoutDiscards(
            ) const = 0;

        /**
         * \brief Override to end protection of the discards installed at a
         *        call to CheckoutDiscards.
         */
        virtual
        void
        CheckinDiscards(
            ) const = 0;

        /**
         * \brief Override to return the vector of DesignVariableInfo's for
         *        this problem.
         *
         * \return The vector of all design variable information objects.
         */
        virtual
        const DesignVariableInfoVector&
        GetDesignVariableInfos(
            ) const = 0;

        /**
         * \brief Override to return the vector of ObjectiveFunctionInfo's for
         *        this problem.
         *
         * \return The vector of all objective function information objects.
         */
        virtual
        const ObjectiveFunctionInfoVector&
        GetObjectiveFunctionInfos(
            ) const = 0;

        /**
         * \brief Override to return the vector of ConstraintInfo's for
         *        this problem.
         *
         * \return The vector of all constraint information objects.
         */
        virtual
        const ConstraintInfoVector&
        GetConstraintInfos(
            ) const = 0;

        /// Override to return a Design* properly allocated for this target
        /**
         * \return A new Design object properly allocated for this target.
         */
        virtual
        Design*
        GetNewDesign(
            ) const = 0;

        /**
         * \brief Override to return a Design* properly allocated for this
         *        target which is a duplicate of "copy".
         *
         * \param copy An existing Design to duplicate.
         * \return A new Design copy constructed from "copy".
         */
        virtual
        Design*
        GetNewDesign(
            const Design& copy
            ) const = 0;

        /// Override to return the number of design variables for this problem.
        /**
         * \return The number of design variables in the problem represented by
         *         this DesignTarget.
         */
        virtual
        std::size_t
        GetNDV(
            ) const = 0;

        /**
         * \brief Override to return the number of objective functions for this
         *        problem.
         *
         * \return The number of objective functions in the problem represented
         *         by this DesignTarget.
         */
        virtual
        std::size_t
        GetNOF(
            ) const = 0;

        /// Override to return the number of constraints for this problem.
        /**
         * \return The number of constraints in the problem represented by
         *         this DesignTarget.
         */
        virtual
        std::size_t
        GetNCN(
            ) const = 0;

        /**
         * \brief Override to do whatever is necessary to determine the
         *        feasibility of "des".
         *
         * A design is considered feasible if all constraints (including side
         * constraints) are satisfied.  This generally requires checking for
         * bound and regular constraint violations and tagging the Design
         * object with appropriate attributes.
         *
         * \param des The Design of which to test the feasibility.
         * \return true if "des" is feasible and false otherwise.
         */
        virtual
        bool
        CheckFeasibility(
            Design& des
            ) const = 0;

        /**
         * \brief Override to return true if all variable values in "des" are
         *        in bounds.
         *
         * \param des The Design of which to test compilance with the side
         *            constraints.
         * \return true if "des" satisfies all side constraints and false
         *         otherwise.
         */
        virtual
        bool
        CheckSideConstraints(
            Design& des
            ) const = 0;

        /**
         * \brief Override to return true if all non-side constraints are
         *        satisfied by "des".
         *
         * \param des The Design of which to test compilance with the
         *            functional constraints of the problem.
         * \return true if "des" satisfies all non-side constraints and false
         *         otherwise.
         */
        virtual
        bool
        CheckNonSideConstraints(
            Design& des
            ) const = 0;

        /// This method should be used to discard a no-longer-wanted Design.
        /**
         * The target assumes that if you call this method, you are completely
         * done with the Design.  It may choose to destroy it and so, if you
         * try to use it again, crash.
         *
         * \param des The no longer wanted Design.
         */
        virtual
        void
        TakeDesign(
            Design* des
            ) = 0;

        /// This method should be used to re-claim what was an unwanted Design.
        /**
         * Since the target stores discards under certain circumstances, it is
         * possible to get them back.  It is only safe to call this for designs
         * that you have discovered in the discards group after a checkout and
         * prior to a checkin.  Note that it is NOT safe to call this method
         * while iterating because it will invalidate the iterator pointing to
         * \a des.
         *
         * \param des The now again wanted Design.
         * \return True if the design is found in the discards and remove and
         *         false otherwise.
         */
        virtual
        bool
        ReclaimDesign(
            const Design& des
            ) = 0;

        /// This method should be used to discard no-longer-wanted Designs.
        /**
         * The target assumes that if you call this method, you are completely
         * done with these Designs.  It may choose to destroy them and so if
         * you try to use them again, crash.
         *
         * DesCont This type must adhere to STL conventions for a simple
         * forward iteratable container including support for const_iterators,
         * begin(), and end().
         *
         * \param cont The container of no longer wanted Designs.
         */
        template <typename DesCont>
        void
        TakeDesigns(
            const DesCont& cont
            )
        {
            for(typename DesCont::const_iterator it(cont.begin());
                it!=cont.end(); ++it) TakeDesign(*it);
        }

        /**
         * \brief Override to record all constraint violations of "des" with
         *        the corresponding info objects.
         *
         * \param des The design whose constraint violations are to be
         *               recorded.
         */
        virtual
        void
        RecordAllConstraintViolations(
            const Design& des
            ) const = 0;

        /**
         * \brief Override to return the complete design space represented as
         *        a RegionOfSpace object.
         */
        virtual
        RegionOfSpace
        GetDesignSpace(
            ) const = 0;

    protected:


    private:





    /*
    ===========================================================================
    Private Methods
    ===========================================================================
    */
    private:





    /*
    ===========================================================================
    Structors
    ===========================================================================
    */
    public:

        /// Constructs a DesignTarget object.
        DesignTarget(
            ) {}

        /// Destructs a DesignTarget object.
        virtual
        ~DesignTarget(
            ) {}

}; // class DesignTarget



/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA







/*
================================================================================
Include Inlined Methods File
================================================================================
*/
// no inline file for interface class.







/*
================================================================================
End of Multiple Inclusion Check
================================================================================
*/
#endif // JEGA_UTILITIES_DESIGNTARGET_HPP
