/*  _________________________________________________________________________
 *
 *  COLIN: A Common Optimization Library INterface
 *  Copyright (c) 2003, Sandia National Laboratories.
 *  This software is distributed under the GNU Lesser General Public License.
 *  For more information, see the README.html file in the top COLIN directory.
 *  _________________________________________________________________________
 */

/**
 * \file OptProblem.h
 *
 * Defines the colin::OptProblem class.
 */

#ifndef colin_OptProblem_h
#define colin_OptProblem_h

#include <acro_config.h>
#include <utilib/std_headers.h>
#include <utilib/stl_auxillary.h>
#include <utilib/CommonIO.h>
#include <utilib/SmartPtr.h>
#include <colin/OptDomainTraits.h>
#include <colin/OptApplication.h>
#include <colin/AppResponse.h>
#include <colin/OptProblemAppWrapper.h>

#include <colin/OptProblemFrag_LinearConstraints.h>
#include <colin/OptProblemFrag_RealParams.h>
#include <colin/OptProblemFrag_IntParams.h>

namespace colin {

//============================================================================
//============================================================================
// Class OptProblem
//============================================================================
//============================================================================

/**
 *  Base class for defining an optimization or search problem.
 *
 *  The class \c OptProblem provides an abstract base class for all
 *  optimization problems.   This class defines a letter-envelope idiom that
 *  leaves the specific details of how the response values are computed an
 *  \c OptApplication object.
 * 
 *  The methods for \c OptProblem define envelope operations.
 *  The \c app method provides access to the application object.
 */
template <class DomainT, class ResponseT=AppResponse<> >
class OptProblem : 
  public OptProblemInterface_LinearConstraints<DomainT,ResponseT,OptDomainTrait_linearconstraints<DomainT>::value>,
  public OptProblemInterface_IntParams<DomainT,ResponseT,OptDomainTrait_integers<DomainT>::value>,
  public OptProblemInterface_RealParams<DomainT,ResponseT,OptDomainTrait_reals<DomainT>::value>,
  virtual public OptProblemFragment_Base<DomainT,ResponseT>,
  public utilib::CommonIO
{ 
public:
 
  #if !defined(DOXYGEN)
  typedef typename ResponseT::vector_t vector_t;
  typedef typename ResponseT::matrix_t matrix_t;
  typedef ResponseT response_t;
  #endif

  /// Generic constructor
  OptProblem() : default_priority(-1)
		{}

  /// Copy constructor
  OptProblem(OptProblem<DomainT,ResponseT>& prob)
		{*this &= prob;}

  /// Destructor
  virtual ~OptProblem()
		{}

  /// Operator to share the problem handle between two problem classes
  OptProblem<DomainT,ResponseT>& operator&=(const OptProblem<DomainT,ResponseT>& prob)
		{
		if (this->handle != prob.handle) {
	   	   this->handle = prob.handle;
	   	   this->init_appinfo();
	   	   }
		return *this;
		}
  
  /// Operator to share the problem handle between two problem classes
  template <class LDomainT, class LResponseT>
  OptProblem<DomainT,ResponseT>& operator&=(OptProblem<LDomainT,LResponseT>& problem)
	{
	if (problem.handle) {
	   this->handle = new OptProblemAppWrapper<DomainT,ResponseT,LDomainT,LResponseT>(problem.handle);
	   this->init_appinfo();
           }
	return *this;
	}

  /// If keep_state is true, then keep the state from the current
  /// application object
  /// TODO: clean up semantics for deletion.
  ///   	By default, this OWNS the application ... which is weird.
  void set_application(OptApplication<DomainT,ResponseT>* app_,
		  				bool keep_state=false)
	{
	if (app_) {
	   if (keep_state) {
	      OptProblemAppBody<DomainT,ResponseT>* handle_ = new OptProblemAppBody<DomainT,ResponseT>();
	      handle_ -> App = app_;
	      this->handle->extract_state(handle_);
	      this->handle = handle_;
	   }
	   else {
	      this->handle = new OptProblemAppBody<DomainT,ResponseT>
			(new OptProblemState<DomainT,vector_t,matrix_t>,app_);
	   }
	   this->init_appinfo();
	   }
	}

  /// Setup application information in this problem
  template <class LDomainT, class LResponseT>
  void set_application(OptApplication<LDomainT,LResponseT>* app_)
	{
	if (app_) {
	   this->handle = new OptProblemAppWrapper<DomainT,ResponseT,LDomainT,LResponseT>(
		   new OptProblemAppBody<LDomainT,LResponseT>
 			(new OptProblemState<DomainT,vector_t,matrix_t>,app_));
	   this->init_appinfo();
	   }
	}

  /// Returns true if the application is defined
  operator bool() const 
		{return this->handle;}

  /// Write out the information about the objective function
  virtual void write(std::ostream& output) const;

  /// The default output for a generic domain is empty
  template <typename PointT>
  void write_domain_info(std::ostream& ) const {}

  /// Read in the information about the objective function
  void read(std::istream& ins) 
		{this->app->read(ins);}

  /// Initialize the problem with argc/argv values
  void initialize(int* argc, char*** argv)
		{if (this->app) this->app->initialize(argc,argv);}

  /// Finish up the problem information.
  void finalize()
		{if (this->app) this->app->finalize();}
 
  /**@name Evaluation Controls */
  //@{
  /// Returns the application mode information
  int app_mode()
		{return this->app->app_mode;}

  /// This synchronizes the calculations requested
  void synchronize() 
		{this->app->synchronize();}

  /// Returns the id of the next asynchronous evaluation that terminates
  /// Returns -1 if no evaluation is available
  int next_eval()
		{ return this->app->next_eval(); }

  /// Note that this is -1 if the last evaluation was 
  /// executed synchronously, and this is geq 0 otherwise.
  int& last_id()
		{return this->state->last_id;}

  /// Stops an asynchronous evaluation for the given id.
  /// Returns true if the termination was successful.
  bool terminate_eval(const int id)
		{
		/*
		 * TODO: does this make sense?  If we tell the user
		 * that termination has failed, should they expect it
		 * to eventually complete?  What can they assume about the
		 * results of that evaluation?
		 * */
		bool status = this->app->terminate_eval(id);
		return status;
		}

  /// Ignore this evaluation id.  Treat it as a dead value that is discarded.
  void ignore_eval(const int id)
		{this->app->ignore_eval(id);}

  /// Return the number of queued evaluations
  unsigned int num_queued_evaluations()
		{ return this->app->num_queued_evaluations(); }

  ///
  virtual void Eval(std::vector<int>& asv, ResponseT& response, int mode=mode_f)
		{
		response.resize(this->numObjectives(), 
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->Eval(asv,mode,response);
		}

  ///
  virtual void Eval(const DomainT& point, std::vector<int>& asv, ResponseT& response,
                	int mode=mode_f)
		{
		this->handle->set_point(point);
		Eval(asv,response,mode);
		}

  ///
  virtual void Eval(ResponseT& response, int mode=mode_f)
		{
		response.resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->Eval(mode,response);
		}

  ///
  virtual void Eval(const DomainT& point, ResponseT& response, int mode=mode_f)
		{
		this->handle->set_point(point);
		response.resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->Eval(mode,response);
		}

  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void Eval(const LArrayT<DomainT>& point, std::vector<int>& asv,
			LArrayT<ResponseT>& response, int mode=mode_f)
#else
  void Eval(const std::vector<DomainT>& point, std::vector<int>& asv,
			std::vector<ResponseT>& response, int mode=mode_f)
#endif
		{
		for (size_type i=0; i<point.size(); i++)
                  Eval(point[i], asv, response[i], mode);
		}

#if 0
  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void Eval(const LArrayT<DomainT>& point,
			LArrayT<ResponseT>& response, int mode=mode_f)
#else
  void Eval(const std::vector<DomainT>& point, std::vector<int>& asv,
			std::vector<ResponseT>& response, int mode=mode_f)
#endif
		{
		for (size_type i=0; i<point.size(); i++)
                  Eval(point[i], response[i], mode);
		}
#endif

  ///
  virtual void AsyncEval(std::vector<int>& asv, 
			int& priority, 
			ResponseT* response, 
			int mode=mode_f,
			int tag=-1)
		{
		response->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->AsyncEval(asv,mode,priority,response,tag);
		}

  ///
  virtual void AsyncEval(const DomainT& point, std::vector<int>& asv, int& priority,
			ResponseT* response, int mode=mode_f, int tag=-1)
		{
		response->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->set_point(point);
		this->handle->AsyncEval(asv,mode,priority,response,tag);
		}

  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void AsyncEval(const LArrayT<DomainT>& point, std::vector<int>& asv, 
			LArrayT<int*>& priority, LArrayT<ResponseT*>& response, 
			int mode, LArrayT<int>& tag)
#else
  void AsyncEval(const std::vector<DomainT>& point, std::vector<int>& asv, 
			std::vector<int*>& priority, std::vector<ResponseT*>& response, 
			int mode, std::vector<int>& tag)
#endif
		{
		this->app->start_async_batch();
		for (size_type i=0; i<point.size(); i++) {
		  response[i]->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		  this->handle->set_point(point[i]);
		  this->handle->AsyncEval(asv,mode,*(priority[i]),response[i],tag[i]);
		  }
		this->app->end_async_batch();
		}

  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void AsyncEval(const LArrayT<DomainT>& point, std::vector<int>& asv, 
			LArrayT<int*>& priority, LArrayT<ResponseT*>& response, 
			int mode)
#else
  void AsyncEval(const std::vector<DomainT>& point, std::vector<int>& asv, 
			std::vector<int*>& priority, std::vector<ResponseT*>& response, 
			int mode)
#endif
		{
		this->app->start_async_batch();
		for (size_type i=0; i<point.size(); i++) {
		  response[i]->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		  this->handle->set_point(point[i]);
		  this->handle->AsyncEval(asv,mode,*(priority[i]),response[i]);
		  }
		this->app->end_async_batch();
		}

  ///
  virtual void AsyncEval(int& priority, ResponseT* response, int mode=mode_f,
				int tag=-1)
		{
		response->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->AsyncEval(mode,priority,response,tag);
		}

  ///
  virtual void AsyncEval(const DomainT& point, 
			int& priority,
			ResponseT* response, 
                	int mode=mode_f, int tag=-1)
		{
		response->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		this->handle->set_point(point);
		this->handle->AsyncEval(mode,priority,response,tag);
		}

  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void AsyncEval(const LArrayT<DomainT>& point,
			LArrayT<int*>& priority, LArrayT<ResponseT*>& response, 
			int mode, LArrayT<int>& tag)
#else
  void AsyncEval(const std::vector<DomainT>& point,
			std::vector<int*>& priority, std::vector<ResponseT*>& response, 
			int mode, std::vector<int>& tag)
#endif
		{
		this->app->start_async_batch();
		for (size_type i=0; i<point.size(); i++) {
		  response[i]->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		  this->handle->set_point(point[i]);
		  this->handle->AsyncEval(mode,*(priority[i]),(response[i]),tag[i]);
		  }
		this->app->end_async_batch();
		}

  ///
#ifdef ACRO_HAVE_TEMPLATES_AS_TEMPLATE_ARGUMENTS
  template <template <class TYPE> class LArrayT>
  void AsyncEval(const LArrayT<DomainT>& point,
			LArrayT<int*>& priority, LArrayT<ResponseT*>& response, 
			int mode)
#else
  void AsyncEval(const std::vector<DomainT>& point,
			std::vector<int*>& priority, std::vector<ResponseT*>& response, 
			int mode)
#endif
		{
		this->app->start_async_batch();
		for (size_type i=0; i<point.size(); i++) {
		  response[i]->resize(this->numObjectives(),
				this->numNonlinearConstraints(),
				this->num_real_params(), 
				(mode|mode_g)||(mode|mode_cg),
				(mode|mode_h)||(mode|mode_ch));
		  this->handle->set_point(point[i]);
		  ucout<< "\n222++++++in OptProblem batch async eval method, "
		      << "address of response obj == "<<response[i]<<std::endl;
		  this->handle->AsyncEval(mode,*(priority[i]),response[i]);
		  }
		this->app->end_async_batch();
		}

  ///
  void EvalF(const DomainT& point, typename ResponseT::realarray_t& values)
		{
		this->handle->set_point(point);
		this->handle->EvalF(values);
		}

  ///
  void EvalF(const DomainT& point, real& value)
		{
		this->handle->set_point(point);
		this->handle->EvalF(value);
		}

  ///
  void AsyncEvalF(const DomainT& point, real* value, int& priority, int tag=-1)
		{
		this->handle->set_point(point);
		this->handle->AsyncEvalF(value,priority,tag);
		}

  ///
  void AsyncEvalF(const DomainT& point, real* value, int tag=-1)
		{
		AsyncEvalF(point,value,this->default_priority,tag);
		}

  ///
  void AsyncEvalF(const DomainT& point, typename ResponseT::realarray_t* values, int& priority, int tag=-1)
		{
		this->handle->set_point(point);
		this->handle->AsyncEvalF(values,priority,tag);
		}

  ///
  void AsyncEvalF(const DomainT& point, typename ResponseT::realarray_t* values, int tag=-1)
		{
		AsyncEvalF(point,values,this->default_priority,tag);
		}

  /// Evaluate the constraint functions at the given point.
  void EvalCF(const DomainT& point, vector_t& value)
		{
		if (this->state->numConstraints == 0) {
		   value.resize(0);
   		   return;
		   }
		this->handle->set_point(point);
		this->handle->EvalCF(value);
		}

  /// Asynchronously evaluate the constraint functions at the given point.
  void AsyncEvalCF(const DomainT& point, vector_t* value, int& priority, 
				int tag=-1)
		{
		if (this->state->numConstraints == 0) {
		   value.resize(0);
   		   return;
		   }
		this->handle->set_point(point);
		this->handle->AsyncEvalCF(value,priority,tag);
		}

  /// Asynchronously evaluate the constraint functions at the given point
  ///   with a default priority
  void AsyncEvalCF(const DomainT& point, vector_t* value, int tag=-1)
		{
		AsyncEvalCF(point,value,this->default_priority,tag);
		}
  //@}

  /**@name Response History Controls */
  //@{
  /// Load the database of prior evaluations.
  virtual void read_db(std::string& /*fname*/) {}

  /// Save the database of evaluations.
  virtual void write_db(std::string& /*fname*/) {}
  //@}

  /// The number of function evaluations
  int neval() const 
		{return this->app->neval_ctr;}

  /// Reset the number of function evaluations
  void reset_neval(int val=0) 
		{this->app->neval_ctr = val;}

  /// Reset the IDs used for function evaluations
  void reset_neval_request_id(int val=0) 
		{this->app->neval_requested = val;}


  /**@name Parameter Manipulation Routines */
  //@{
#if 0
  There is a bug here.  We cannot count on the rep->data()->point object
  existing, and if it does, its on loan.

  /// Write out the current point in the objective function.
  virtual void write_vars(ostream& output) const
			{output << *(rep->data()->point);}

  /// Read in the current point in the objective function.
  virtual void read_vars(istream& input)
			{input >> *(rep->data()->point);}

#ifdef ACRO_HAVE_MPI
  /// Write out the current point in the objective function.
  virtual void write_vars(PackBuffer& output) const
			{output << this->handle->point;}

  /// Read in the current point in the objective function.
  virtual void read_vars(UnPackBuffer& input)
			{input >> this->handle->point; this->handle->set_point(point);}
#endif
#endif
  //@}

  ///
  colin::optimizationSense sense()
			{ return this->app->sense; }

protected:

  /// Copy operator
  OptProblem<DomainT,ResponseT>& operator=(const OptProblem<DomainT,ResponseT>& prob);

  /// Default priority for evaluations
  int default_priority;

};


//=======================================================================
//
//
template <class DomainT, class ResponseT>
void OptProblem<DomainT,ResponseT>::write(std::ostream& os) const
{
os << "##------------------------------------------------------------------------------" << std::endl;
os << "## Optimization Problem Information" << std::endl;
os << "##------------------------------------------------------------------------------" << std::endl;
if (this->app) {
   this->app->write(os);
   os << "#\n# Constraint Information\n#\n";
   os << "Total Num Constraints\t\t" << this->numConstraints() << std::endl;

   if ((this->state->num_real_params + this->state->num_int_params +
	this->state->num_binary_params) > 0) {
      os << "Num Linear Equality\t\t" << this->state->numLinearEqConstraints << std::endl;
      os << "Num Linear Inequality\t\t" << this->state->numLinearIneqConstraints << std::endl;
      }

   os << "Num Nonlinear Equality\t\t" << this->numNonlinearEqConstraints() << std::endl;
   os << "Num Nonlinear Inequality\t" << this->numNonlinearIneqConstraints() << std::endl;

   if ((this->state->num_real_params != 0) ||
       (this->state->num_int_params != 0) ||
       (this->state->num_binary_params != 0))
      os << "enforcing_bounds " << (this->state->enforcing_bound_constraints) << std::endl;

   if (this->state->num_real_params != 0) {
         os << "Num Real Variables: " << this->state->num_real_params << std::endl;
#if defined(CYGWIN) || defined(COUGAR) || defined(TFLOPS_SERVICE)
         os << "Real    Lower    ";
	 os << this->state->real_lower_bounds << std::endl;
         os << "Real    Upper    ";
	 os << this->state->real_upper_bounds << std::endl;
         if (this->state->enforcing_bound_constraints) {
            os << "Real Lower Type ";
	    os << this->state->real_lower_bound_type << std::endl;
            os << "Real Upper Type ";
	    os << this->state->real_upper_bound_type << std::endl;
            }
#else
         os << "Real    Lower    " << this->state->real_lower_bounds << std::endl;
         os << "Real    Upper    " << this->state->real_upper_bounds << std::endl;
         if (this->state->enforcing_bound_constraints) {
            os << "Real Lower Type " << this->state->real_lower_bound_type << std::endl;
            os << "Real Upper Type " << this->state->real_upper_bound_type << std::endl;
            }
#endif
      }

   if (this->state->num_int_params != 0) {
     
         os << "Num Integer Variables: " << this->state->num_int_params << std::endl;
#if defined(CYGWIN) || defined(COUGAR) || defined(TFLOPS_SERVICE)
         os << "Integer Lower " << this->state->int_lower_bounds << std::endl;
         os << "Integer Upper " << this->state->int_upper_bounds << std::endl;
         if (this->state->enforcing_bound_constraints) {
            os << "Integer Lower Type " << this->state->int_lower_bound_type << std::endl;
            os << "Integer Upper Type " << this->state->int_upper_bound_type << std::endl;
	    }
#else
         os << "Integer Lower " << this->state->int_lower_bounds << std::endl;
         os << "Integer Upper " << this->state->int_upper_bounds << std::endl;
         if (this->state->enforcing_bound_constraints) {
            os << "Integer Lower Type " << this->state->int_lower_bound_type << std::endl;
            os << "Integer Upper Type " << this->state->int_upper_bound_type << std::endl;
	    }
#endif
      }

   if (this->state->num_binary_params != 0) {
         os << "Num Binary  Variables: " << this->state->num_binary_params << std::endl;
      }

   write_domain_info<DomainT>(os);
   }
else {
   os << "#\n# No optimization application specified.\n#" << std::endl;
   }
}

//=======================================================================
//
//
template <class DomainT, class ResponseT>
OptProblem<DomainT,ResponseT>& OptProblem<DomainT,ResponseT> ::operator=(const OptProblem<DomainT,ResponseT>& prob)
{
EXCEPTION_MNGR(std::runtime_error,"OptProblem::operator= - undefined!");
return *this;
}

} // namespace colin

//=======================================================================
//
//
/// Write a colin::OptProblem object.
template <class DomainT, class ResponseT>
inline std::ostream& operator<< (std::ostream& output, const colin :: OptProblem<DomainT,ResponseT>& prob)
{ prob.write(output); return(output); }


//=======================================================================
//
//
/// Read a colin::OptProblem object.
template <class DomainT, class ResponseT>
inline std::istream& operator>> (std::istream& input, colin :: OptProblem<DomainT,ResponseT>& prob)
{ prob.read(input); return(input); }


#endif
