/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        InterpPolyApproximation
//- Description:  Implementation code for InterpPolyApproximation class
//-               
//- Owner:        Mike Eldred

#include "InterpPolyApproximation.H"
#include "LagrangeInterpPolynomial.H"
#include "NonDQuadrature.H"
#include "NonDSparseGrid.H"
#ifdef DAKOTA_QUADRATURE
#include "sparse_grid_mixed.H"
#endif // DAKOTA_QUADRATURE
#include "ProblemDescDB.H"

//#define DEBUG

namespace Dakota {


InterpPolyApproximation::
InterpPolyApproximation(const ProblemDescDB& problem_db, const size_t& num_acv):
  BasisPolyApproximation(problem_db, num_acv), numCollocPts(0)
{ }


int InterpPolyApproximation::min_coefficients() const
{
  // return the minimum number of data instances required to build the 
  // surface of multiple dimensions
  return (expansionCoeffFlag || expansionGradFlag) ? 1 : 0;
}


void InterpPolyApproximation::find_coefficients()
{
  if (!expansionCoeffFlag && !expansionGradFlag) {
    Cerr << "Warning: neither expansion coefficients nor expansion coefficient "
	 << "gradients\n         are active in InterpPolyApproximation::find_"
	 << "coefficients().\n         Bypassing approximation construction."
	 << endl;
    return;
  }

  // For testing of anchorPoint logic:
  //anchorPoint = currentPoints.front();
  //currentPoints.pop_front();

  size_t i, j, offset = 0;
  numCollocPts = currentPoints.entries();
  // anchorPoint, if present, is the first expansionSample.
  if (!anchorPoint.is_null()) {
    offset        = 1;
    numCollocPts += offset;
  }
  if (!numCollocPts) {
    Cerr << "Error: nonzero number of sample points required in "
	 << "InterpPolyApproximation::find_coefficients()." << endl;
    abort_handler(-1);
  }

  if (expansionCoeffFlag && expansionCoeffs.empty())
    expansionCoeffs.reshape(numCollocPts);
  if (expansionGradFlag  && expansionCoeffGrads.empty())
    expansionCoeffGrads.reshape(numCollocPts);

  if (!anchorPoint.is_null()) {
    if (expansionCoeffFlag)
      expansionCoeffs[0]     = anchorPoint.response_function();
    if (expansionGradFlag)
      expansionCoeffGrads[0] = anchorPoint.response_gradient();
  }

  List<SurrogateDataPoint>::iterator it = currentPoints.begin();
  for (i=offset; i<numCollocPts; i++, it++) {
    if (expansionCoeffFlag)
      expansionCoeffs[i]     = it->response_function();
    if (expansionGradFlag)
      expansionCoeffGrads[i] = it->response_gradient();
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE: {
    NonDQuadrature* nond_quad = (NonDQuadrature*)integrationRep;
    const UShortArray& quad_order = nond_quad->quadrature_order();

    // verify total number of collocation pts (should not include anchorPoint)
    size_t num_gauss_pts = 1;
    for (i=0; i<numVars; i++)
      num_gauss_pts *= quad_order[i];
    if (num_gauss_pts != numCollocPts) {
      Cerr << "Error: inconsistent total Gauss point count in "
	   << "InterpPolyApproximation::find_coefficients()" << endl;
      abort_handler(-1);
    }

    // size and initialize polynomialBasis, one interpolant per variable
    if (polynomialBasis.empty()) {
      polynomialBasis.reshape(numVars);
      const RealVector2DArray& gauss_pts_1d = nond_quad->gauss_points_array();
      for (i=0; i<numVars; i++) {
	polynomialBasis[i].reshape(1);
	bool found = false;
	for (j=0; j<i; j++)
	  if (gauss_pts_1d[i][0] == gauss_pts_1d[j][0])
	    { found = true; break; }
	if (found) // reuse previous instance via shared representation
	  polynomialBasis[i][0] = polynomialBasis[j][0];
	else { // instantiate a new unique instance
	  polynomialBasis[i][0] = BasisPolynomial(LAGRANGE);
	  polynomialBasis[i][0].interpolation_points(gauss_pts_1d[i][0]);
	}
      }
    }

    // only one set of Smolyak indices, all 0's, indicating usage of
    // polynomialBasis[variable][0], with order as specified by quadrature spec.
    if (smolyakMultiIndex.empty()) {
      smolyakMultiIndex.reshape(1);
      smolyakMultiIndex[0].reshape(numVars);
      smolyakMultiIndex[0] = 0;
    }

    // define mapping from 1:numCollocPts to set of 1d interpolation indices
    if (collocKey.empty()) {
      collocKey.reshape(1);
      collocKey[0].reshape(numCollocPts);
      UShortArray gauss_indices(numVars, 0);
      for (i=0; i<numCollocPts; i++) {
	collocKey[0][i] = gauss_indices;
#ifdef DEBUG
	Cout << "collocKey[0][" << i << "]:\n" << collocKey[0][i] <<endl;
#endif // DEBUG
	// increment the n-dimensional gauss point index set
	size_t increment_index = 0;
	gauss_indices[increment_index]++;
	while (increment_index < numVars &&
	       gauss_indices[increment_index] >= quad_order[increment_index]) {
	  gauss_indices[increment_index] = 0;
	  increment_index++;
	  if (increment_index < numVars)
	    gauss_indices[increment_index]++;
	}
      }
    }

    // map indices i to i for the lone tensor product 
    if (expansionCoeffIndices.empty()) {
      expansionCoeffIndices.reshape(1);
      expansionCoeffIndices[0].reshape(numCollocPts);
      for (i=0; i<numCollocPts; i++)
	expansionCoeffIndices[0][i] = i;
    }
    break;
  }
  case SPARSE_GRID: {
    NonDSparseGrid* nond_sparse = (NonDSparseGrid*)integrationRep;
    unsigned short w = nond_sparse->sparse_grid_level()[0];

    // size and initialize polynomialBasis, multiple interpolants per variable
    if (polynomialBasis.empty()) {
      polynomialBasis.reshape(numVars);
      const RealVector2DArray& gauss_pts_1d = nond_sparse->gauss_points_array();
      // j range is 0:w inclusive; i range is 1:w+1 inclusive
      size_t num_levels_per_var = w + 1;
      for (i=0; i<numVars; i++) {
	polynomialBasis[i].reshape(num_levels_per_var);
	// isotropic level (for now) simplifies reuse logic:
	// just check highest order for duplication
	bool found = false;
	size_t k, last_lev = num_levels_per_var - 1;
	for (k=0; k<i; k++)
	  if (gauss_pts_1d[i][last_lev] == gauss_pts_1d[k][last_lev])
	    { found = true; break; }
	for (j=0; j<num_levels_per_var; j++) {
	  if (found) // reuse previous instances via shared representations
	    polynomialBasis[i][j] = polynomialBasis[k][j]; // shared rep
	  else { // instantiate new unique instances
	    polynomialBasis[i][j] = BasisPolynomial(LAGRANGE);
	    polynomialBasis[i][j].interpolation_points(gauss_pts_1d[i][j]);
	  }
	}
      }
    }

    // Populate smolyakMultiIndex for use in forming interpolants.  Identifies
    // use of polynomialBasis[variable][index] based on index 0:num_levels-1.
    // Implementation is adapted from OrthogPolyApproximation::multiIndex.
    if (smolyakMultiIndex.empty()) {
      // w = q - N = dimension-independent level
      //   w + 1 <= |i| <= w + N for i starts at 1 (used for index set defn.)
      //   w - N + 1 <= |j| <= w for j = i - 1 starts at 0 (used for generation)
      size_t i, num_terms = (w < numVars) ?
	(size_t)(   BasisPolynomial::factorial_ratio(w+numVars,w) /
		    BasisPolynomial::factorial(numVars) ) :
	(size_t)( ( BasisPolynomial::factorial_ratio(w+numVars,w) -
		    BasisPolynomial::factorial_ratio(w,w-numVars) ) /
		    BasisPolynomial::factorial(numVars) );
      smolyakMultiIndex.reshape(num_terms);
      size_t cntr = 0;
      // special logic required for order_nd < 2 due to prev_index defn below
      int wmNp1 = w-numVars+1;
      if (wmNp1 <= 0) { // && w >= 0) {
	smolyakMultiIndex[cntr].reshape(numVars);
	smolyakMultiIndex[cntr] = 0; // j index set is 0-based
	cntr++;
      }
      if (wmNp1 <= 1 && w >= 1) {
	for (i=0; i<numVars; i++, cntr++) {
	  smolyakMultiIndex[cntr].reshape(numVars);
	  smolyakMultiIndex[cntr] = 0;    // j index set is 0-based
	  smolyakMultiIndex[cntr][i] = 1; // j index set is 0-based
	}
      }
      for (unsigned short order_nd=max(wmNp1,2); order_nd<=w; order_nd++) {
	UShortArray terms(order_nd, 1); // # of indices = current order
	bool order_complete = false;
	while (!order_complete) {
	  // this is the inner-most loop w/i the nested looping managed by terms
	  size_t k, last_index = order_nd - 1, prev_index = order_nd - 2;
	  for (terms[last_index]=1; terms[last_index]<=terms[prev_index]; 
	       terms[last_index]++, cntr++) {
	    smolyakMultiIndex[cntr].reshape(numVars);
	    for (k=0; k<numVars; k++) // j index set is 0-based => j = i - 1
	      smolyakMultiIndex[cntr][k] = terms.count(k+1);
	  }
	  // increment term hierarchy
	  bool increment_complete = false;
	  while (!increment_complete) {
	    terms[last_index] = 1;
	    terms[prev_index]++;
	    if (prev_index == 0) {
	      increment_complete = true;
	      if (terms[prev_index] > numVars)
		order_complete = true;
	    }
	    else {
	      last_index = prev_index;
	      prev_index--;
	      if (terms[last_index]<=terms[prev_index])
		increment_complete = true;
	    }
	  }
	}
      }
      smolyakCoeffs.reshape(num_terms);
      for (i=0; i<num_terms; i++) {
	const UShortArray& index_set_i = smolyakMultiIndex[i];
	int wpNmi = w; // w + N - |i| = w - |j|
	for (j=0; j<numVars; j++) // subtract 1-norm of index set
	  wpNmi -= index_set_i[j];
	smolyakCoeffs[i] = pow(-1., wpNmi)
	  * BasisPolynomial::n_choose_k(numVars - 1, wpNmi);
      }
#ifdef DEBUG
      Cout << "num Smolyak terms = " << num_terms << endl;
      for (i=0; i<num_terms; i++)
	Cout << "smolyakMultiIndex[" << i << "]:\n" << smolyakMultiIndex[i]
	     << "smolyakCoeffs[" << i << "] = " << smolyakCoeffs[i] << endl;
#endif // DEBUG
    }

    // define mapping from 1:numCollocPts to set of 1d interpolation indices
    if (collocKey.empty() || expansionCoeffIndices.empty()) {
      size_t i, j, k, num_smolyak_indices = smolyakMultiIndex.length();
      collocKey.reshape(num_smolyak_indices);
      expansionCoeffIndices.reshape(num_smolyak_indices);
      UShortArray gauss_indices(numVars), order(numVars);
      IntArray key(2*numVars);
      unsigned short closed_order_max;
      nond_sparse->level_to_order_closed(w, closed_order_max);
      //const IntArraySizetMap& index_map
      //  = nond_sparse->sparse_grid_index_map();
      const Pecos::ShortArray& ran_var_types
	= nond_sparse->integrated_variable_types();
      // TO DO: drop size() call once num_total_pts no longer needed
      const IntArray& rules = nond_sparse->integration_rules();
      int num_total_pts
	= webbur::sparse_grid_mixed_size_total(numVars, w, rules);
      int* unique_mapping = new int [num_total_pts];
      webbur::sparse_grid_mixed_unique_index(numVars, w, rules, num_total_pts,
	unique_mapping);
      size_t cntr = 0;
      for (i=0; i<num_smolyak_indices; i++) {
	const UShortArray& sm_index_i = smolyakMultiIndex[i];
	SizetArray&    coeff_map_i  = expansionCoeffIndices[i];
	UShort2DArray& colloc_key_i = collocKey[i];
	size_t num_tp_pts = 1;
	for (j=0; j<numVars; j++) {
	  switch (ran_var_types[j]) { // j -> order
	  case NORMAL: case EXPONENTIAL:
	    nond_sparse->level_to_order_open(sm_index_i[j], order[j]);   break;
	  case UNIFORM:
	    nond_sparse->level_to_order_closed(sm_index_i[j], order[j]); break;
	  }
	  num_tp_pts *= order[j];
	}
	colloc_key_i.reshape(num_tp_pts);
	coeff_map_i.reshape(num_tp_pts);
        gauss_indices = 0;
	for (j=0; j<num_tp_pts; j++) {
	  // update collocKey
	  colloc_key_i[j] = gauss_indices;
	  coeff_map_i[j]  = unique_mapping[cntr++];

	  /*
	  // update expansionCoeffIndices: sparse grid nesting of points may be
	  // tracked by demoting each point to its lowest order representation
	  // or by promoting each point to the highest order grid.
	  for (k=0; k<numVars; k++) {
	    switch (ran_var_types[k]) {
	    // For open weakly-nested rules (Gauss-Hermite, Gauss-Legendre),
	    // demote to order=1,index=0 if a center pt for a particular order>1
	    case NORMAL:
	      if (numVars > 1 && order[k] > 1 &&
		  gauss_indices[k] == (order[k]-1)/2) // demoted base/index
		{ key[k] = 1;        key[k+numVars] = 0; }
	      else                                    // original base/index
		{ key[k] = order[k]; key[k+numVars] = gauss_indices[k]; }
	      break;
	    // For closed nested rules (Clenshaw-Curtis), base is a dummy and
	    // index is promoted to the highest order grid
	    case UNIFORM:
	      key[k] = closed_order_max; // promoted base
	      if (sm_index_i[k] == 0)
		key[k+numVars] = (closed_order_max-1)/2;      // promoted index
	      else {
		key[k+numVars] = gauss_indices[k];
		unsigned short delta_w = w - sm_index_i[k];
		if (delta_w)
		  key[k+numVars] *= (size_t)pow(2., delta_w); // promoted index
	      }
	      break;
	    // For open non-nested rules (Gauss-Laguerre), no modification reqd
	    case EXPONENTIAL: // original base/index
	      key[k] = order[k]; key[k+numVars] = gauss_indices[k];
	      break;
	    }
	  }
#ifdef DEBUG
	  Cout << "lookup key:\n" << key << endl;
#endif // DEBUG
	  IntArraySizetMap::const_iterator cit = index_map.find(key);
	  if (cit == index_map.end()) {
	    Cerr << "Error: lookup on sparse grid index map failed in "
		 << "InterpPolyApproximation::find_coefficients()" << endl;
	    abort_handler(-1);
	  }
	  else
	    coeff_map_i[j] = cit->second;
	  */
#ifdef DEBUG
	  Cout << "collocKey[" << i << "][" << j << "]:\n" << colloc_key_i[j]
	       << "expansionCoeffIndices[" << i << "][" << j << "] = "
	       << coeff_map_i[j] << endl;
#endif // DEBUG

	  // increment the n-dimensional gauss point index set
	  size_t increment_index = 0;
	  gauss_indices[increment_index]++;
	  while (increment_index < numVars &&
		 gauss_indices[increment_index] >= order[increment_index]) {
	    gauss_indices[increment_index] = 0;
	    increment_index++;
	    if (increment_index < numVars)
	      gauss_indices[increment_index]++;
	  }
	}
      }
      delete [] unique_mapping;
    }
  }
  }
}


const Real& InterpPolyApproximation::
tensor_product_value(const RealVector& x, size_t tp_index)
{
  tpValue = 0.;
  const UShort2DArray& key         = collocKey[tp_index];
  const UShortArray&   sm_index    = smolyakMultiIndex[tp_index];
  const SizetArray&    coeff_index = expansionCoeffIndices[tp_index];
  size_t i, j, num_colloc_pts = key.length();
  for (i=0; i<num_colloc_pts; i++) {
    const UShortArray& key_i = key[i];
    Real L_i = 1.0;
    for (j=0; j<numVars; j++)
      L_i *= polynomialBasis[j][sm_index[j]].get_value(x[j], key_i[j]);
    tpValue += expansionCoeffs[coeff_index[i]] * L_i;
  }
  return tpValue;
}


const RealBaseVector& InterpPolyApproximation::
tensor_product_gradient(const RealVector& x, size_t tp_index)
{
  if (tpGradient.length() != numVars)
    tpGradient.reshape(numVars);
  tpGradient = 0.;

  const UShort2DArray& key         = collocKey[tp_index];
  const UShortArray&   sm_index    = smolyakMultiIndex[tp_index];
  const SizetArray&    coeff_index = expansionCoeffIndices[tp_index];
  size_t i, j, k, num_colloc_pts = key.length();
  for (i=0; i<num_colloc_pts; i++) {
    const UShortArray& key_i   = key[i];
    const Real&        coeff_i = expansionCoeffs[coeff_index[i]];
    for (j=0; j<numVars; j++) {
      Real term_i_grad_j = 1.0;
      for (k=0; k<numVars; k++)
	term_i_grad_j *= (k == j) ?
	  polynomialBasis[k][sm_index[k]].get_gradient(x[k], key_i[k]) :
	  polynomialBasis[k][sm_index[k]].get_value(x[k],    key_i[k]);
      tpGradient[j] += coeff_i * term_i_grad_j;
    }
  }
  return tpGradient;
}


const RealBaseVector& InterpPolyApproximation::
tensor_product_gradient(const RealVector& x, size_t tp_index,
			const UIntArray& dvv)
{
  size_t num_deriv_vars = dvv.length();
  if (tpGradient.length() != num_deriv_vars)
    tpGradient.reshape(num_deriv_vars);
  tpGradient = 0.;

  const UShort2DArray& key         = collocKey[tp_index];
  const UShortArray&   sm_index    = smolyakMultiIndex[tp_index];
  const SizetArray&    coeff_index = expansionCoeffIndices[tp_index];
  size_t i, j, k, deriv_index, num_colloc_pts = key.length();
  for (i=0; i<num_colloc_pts; i++) {
    const UShortArray& key_i   = key[i];
    const Real&        coeff_i = expansionCoeffs[coeff_index[i]];
    for (j=0; j<num_deriv_vars; j++) {
      deriv_index = dvv[j] - 1; // requires an "All" view
      Real term_i_grad_j = 1.0;
      for (k=0; k<numVars; k++)
	term_i_grad_j *= (k == deriv_index) ?
	  polynomialBasis[k][sm_index[k]].get_gradient(x[k], key_i[k]) :
	  polynomialBasis[k][sm_index[k]].get_value(x[k],    key_i[k]);
      tpGradient[j] += coeff_i * term_i_grad_j;
    }
  }
  return tpGradient;
}


const Real& InterpPolyApproximation::
tensor_product_mean(const RealVector& x, size_t tp_index)
{
  tpMean = 0.;

  const UShort2DArray&      key = collocKey[tp_index];
  const UShortArray&   sm_index = smolyakMultiIndex[tp_index];
  const SizetArray& coeff_index = expansionCoeffIndices[tp_index];
  const RealVector2DArray& gauss_wts_1d = integrationRep->gauss_weights_array();

  size_t i, j, num_colloc_pts = key.length();
  for (i=0; i<num_colloc_pts; i++) {
    const UShortArray& key_i = key[i];
    Real prod_i = 1.;
    for (j=0; j<numVars; j++)
      prod_i *= (randomVarsKey[j]) ? gauss_wts_1d[j][sm_index[j]][key_i[j]] :
	polynomialBasis[j][sm_index[j]].get_value(x[j], key_i[j]);
    tpMean += expansionCoeffs[coeff_index[i]] * prod_i;
  }
  return tpMean;
}


const RealBaseVector& InterpPolyApproximation::
tensor_product_mean_gradient(const RealVector& x, size_t tp_index,
			     const UIntArray& dvv)
{
  size_t i, j, k, deriv_index, num_deriv_vars = dvv.length(),
    cntr = 0; // insertions carried in order within tpCoeffGrads
  if (tpMeanGrad.length() != num_deriv_vars)
    tpMeanGrad.reshape(num_deriv_vars);
  tpMeanGrad = 0.;

  const UShort2DArray&      key = collocKey[tp_index];
  const UShortArray&   sm_index = smolyakMultiIndex[tp_index];
  const SizetArray& coeff_index = expansionCoeffIndices[tp_index];
  const RealVector2DArray& gauss_wts_1d = integrationRep->gauss_weights_array();

  // -------------------------------------------------------------------
  // Mixed variable key:
  //   xi = ran vars, sa = augmented des vars, si = inserted design vars
  // Active variable expansion:
  //   R(xi, sa, si) = Sum_i r_i(sa, si) L_i(xi)
  //   mu(sa, si)    = Sum_i r_i(sa, si) wt_prod_i
  //   dmu/ds        = Sum_i dr_i/ds wt_prod_i
  // All variable expansion:
  //   R(xi, sa, si) = Sum_i r_i(si) L_i(xi, sa)
  //   mu(sa, si)    = Sum_i r_i(si) Lsa_i wt_prod_i
  //   dmu/dsa       = Sum_i r_i(si) dLsa_i/dsa wt_prod_i
  //   dmu/dsi       = Sum_i dr_i/dsi Lsa_i wt_prod_i
  // -------------------------------------------------------------------
  size_t num_colloc_pts = key.length();
  SizetList::iterator it;
  for (i=0; i<num_deriv_vars; i++) {
    deriv_index = dvv[i] - 1; // OK since we are in an "All" view
    // Error check for required data
    if (randomVarsKey[deriv_index] && !expansionGradFlag) {
      Cerr << "Error: expansion coefficient gradients not defined in "
	   << "InterpPolyApproximation::get_mean_gradient()." << endl;
      abort_handler(-1);
    }
    else if (!randomVarsKey[deriv_index] && !expansionCoeffFlag) {
      Cerr << "Error: expansion coefficients not defined in "
	   << "InterpPolyApproximation::get_mean_gradient()" << endl;
      abort_handler(-1);
    }
    for (j=0; j<num_colloc_pts; j++) {
      const UShortArray& key_j = key[j];
      Real wt_prod_j = 1.;
      for (it=randomIndices.begin(); it!=randomIndices.end(); it++) {
	k = *it;
	wt_prod_j *= gauss_wts_1d[k][sm_index[k]][key_j[k]];
      }
      if (randomVarsKey[deriv_index]) {
	// -----------------------------------------------------------
	// derivative of All variable expansion w.r.t. random variable
	// (design variable insertion)
	// -----------------------------------------------------------
	Real Lsa_j = 1.;
	for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	  k = *it;
	  Lsa_j *= polynomialBasis[k][sm_index[k]].get_value(x[k], key_j[k]);
	}
	tpMeanGrad[i]
	  += wt_prod_j * Lsa_j * expansionCoeffGrads[coeff_index[j]][cntr];
      }
      else {
	// --------------------------------------------------------------
	// derivative of All variable expansion w.r.t. nonrandom variable
	// (design variable augmentation)
	// --------------------------------------------------------------
	Real dLsa_j_dsa_i = 1.;
	for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	  k = *it;
	  dLsa_j_dsa_i *= (k == deriv_index) ?
	    polynomialBasis[k][sm_index[k]].get_gradient(x[k], key_j[k]) :
	    polynomialBasis[k][sm_index[k]].get_value(x[k],    key_j[k]);
	}
	tpMeanGrad[i]
	  += expansionCoeffs[coeff_index[j]] * wt_prod_j * dLsa_j_dsa_i;
      }
    }
    if (randomVarsKey[deriv_index]) // deriv w.r.t. design var insertion
      cntr++;
  }

  return tpMeanGrad;
}


const Real& InterpPolyApproximation::
tensor_product_variance(const RealVector& x, size_t tp_index)
{
  tpVariance = 0.;
  const UShort2DArray&      key = collocKey[tp_index];
  const UShortArray&   sm_index = smolyakMultiIndex[tp_index];
  const SizetArray& coeff_index = expansionCoeffIndices[tp_index];
  const RealVector2DArray& gauss_wts_1d = integrationRep->gauss_weights_array();

  size_t i, j, k, num_colloc_pts = key.length();
  Real mean = 0.;
  SizetList::iterator it;
  for (i=0; i<num_colloc_pts; i++) {
    const UShortArray& key_i = key[i];
    Real wt_prod_i = 1., Ls_prod_i = 1.;
    for (k=0; k<numVars; k++)
      if (randomVarsKey[k])
	wt_prod_i *= gauss_wts_1d[k][sm_index[k]][key_i[k]];
      else
	Ls_prod_i *= polynomialBasis[k][sm_index[k]].get_value(x[k], key_i[k]);
    const Real& exp_coeff_i = expansionCoeffs[coeff_index[i]];
    mean += exp_coeff_i * wt_prod_i * Ls_prod_i;
    for (j=0; j<num_colloc_pts; j++) {
      const UShortArray& key_j = key[j];
      // to include the ij-th term, colloc pts xi_i must be the same as xi_j
      // for the ran var subset.  In this case, wt_prod_i may be reused.
      bool include = true;
      for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	if (key_i[*it] != key_j[*it])
	  { include = false; break; }
      if (include) {
	Real Ls_prod_j = 1.;
	for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	  k = *it;
	  Ls_prod_j *= polynomialBasis[k][sm_index[k]].get_value(x[k],key_j[k]);
	}
	tpVariance += wt_prod_i * Ls_prod_i * Ls_prod_j * exp_coeff_i
	  * expansionCoeffs[coeff_index[j]];
      }
    }
  }
  tpVariance -= mean*mean;
  return tpVariance;
}


const RealBaseVector& InterpPolyApproximation::
tensor_product_variance_gradient(const RealVector& x, size_t tp_index,
				 const UIntArray& dvv)
{
  size_t i, j, k, l, deriv_index, num_deriv_vars = dvv.length(),
    cntr = 0; // insertions carried in order within expansionCoeffGrads
  if (tpVarianceGrad.length() != num_deriv_vars)
    tpVarianceGrad.reshape(num_deriv_vars);
  tpVarianceGrad = 0.;

  const UShort2DArray&      key = collocKey[tp_index];
  const UShortArray&   sm_index = smolyakMultiIndex[tp_index];
  const SizetArray& coeff_index = expansionCoeffIndices[tp_index];
  const RealVector2DArray& gauss_wts_1d = integrationRep->gauss_weights_array();

  // -------------------------------------------------------------------
  // Mixed variable key:
  //   xi = ran vars, sa = augmented des vars, si = inserted design vars
  // Active variable expansion:
  //   R(xi, sa, si) = Sum_i r_i(sa, si) L_i(xi)
  //   var(sa, si)   = Sum_i Sum_j r_i(sa, si) r_j(sa, si) wt_prod_ij - mu^2
  //   dvar/ds       = Sum_i Sum_j (r_i dr_j/ds + dr_i/ds r_j) wt_prod_ij -
  //                   2 mu dmu/ds
  // All variable expansion:
  //   R(xi, sa, si) = Sum_i r_i(si) L_i(xi, sa)
  //   var(sa, si)   = Sum_i Sum_j r_i(si) r_j(si) Lsa_i Lsa_j wt_prod_ij -
  //                   mu^2
  //   dvar/dsa      = Sum_i Sum_j r_i(si) r_j(si) (Lsa_i dLsa_j/dsa +
  //                   dLsa_i/dsa Lsa_j) wt_prod_ij - 2 mu dmu/dsa
  //   dvar/dsi      = Sum_i Sum_j (r_i dr_j/dsi + dr_i/dsi r_j)
  //                   Lsa_i Lsa_j wt_prod_ij - 2 mu dmu/dsi
  // -------------------------------------------------------------------
  Real mean = 0.;
  size_t num_colloc_pts = key.length();
  SizetList::iterator it;
  for (i=0; i<num_deriv_vars; i++) {
    deriv_index = dvv[i] - 1; // OK since we are in an "All" view
    if (randomVarsKey[deriv_index] && !expansionGradFlag) { // Error check 
      Cerr << "Error: expansion coefficient gradients not defined in "
	   << "InterpPolyApproximation::get_variance_gradient()." << endl;
      abort_handler(-1);
    }
    Real mean_grad_i = 0.;
    // first loop of double sum
    for (j=0; j<num_colloc_pts; j++) {
      const UShortArray& key_j = key[j];
      // compute wt_prod_j and Lsa_j
      Real wt_prod_j = 1., Lsa_j = 1., dLsa_j_dsa_i = 1.;
      for (k=0; k<numVars; k++)
	if (randomVarsKey[k])
	  wt_prod_j *= gauss_wts_1d[k][sm_index[k]][key_j[k]];
	else
	  Lsa_j *= polynomialBasis[k][sm_index[k]].get_value(x[k], key_j[k]);
      // update mean (once) and mean_grad_i
      if (i == 0)
	mean += expansionCoeffs[coeff_index[j]] * wt_prod_j * Lsa_j;
      if (randomVarsKey[deriv_index])
	mean_grad_i
	  += wt_prod_j * Lsa_j * expansionCoeffGrads[coeff_index[j]][cntr];
      else {
	for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	  k = *it;
	  dLsa_j_dsa_i *= (k == deriv_index) ?
	    polynomialBasis[k][sm_index[k]].get_gradient(x[k], key_j[k]) :
	    polynomialBasis[k][sm_index[k]].get_value(x[k],    key_j[k]);
	}
	mean_grad_i
	  += wt_prod_j * dLsa_j_dsa_i * expansionCoeffs[coeff_index[j]];
      }
      // second loop of double sum
      for (k=0; k<num_colloc_pts; k++) {
	const UShortArray& key_k = key[k];
	// to include the jk-th term, colloc pts xi_j must be the same as xi_k
	// for the random var subset.  In this case, wt_prod_j may be reused.
	bool include = true;
	for (it=randomIndices.begin(); it!=randomIndices.end(); it++)
	  if (key_j[*it] != key_k[*it])
	    { include = false; break; }
	if (include) {
	  Real Lsa_k = 1.;
	  for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++) {
	    l = *it;
	    Lsa_k *= polynomialBasis[l][sm_index[l]].get_value(x[l], key_k[l]);
	  }
	  if (randomVarsKey[deriv_index])
	    // -----------------------------------------------------------
	    // derivative of All variable expansion w.r.t. random variable
	    // (design variable insertion)
	    // -----------------------------------------------------------
	    tpVarianceGrad[i] += wt_prod_j * Lsa_j * Lsa_k *
	      (expansionCoeffs[coeff_index[j]] *
	       expansionCoeffGrads[coeff_index[k]][cntr] +
	       expansionCoeffGrads[coeff_index[j]][cntr] *
	       expansionCoeffs[coeff_index[k]]);
	  else {
	    // ---------------------------------------------------------
	    // derivative of All variable expansion w.r.t. nonrandom var
	    // (design variable augmentation)
	    // ---------------------------------------------------------
	    Real dLsa_k_dsa_i = 1.;
	    for (it=nonRandomIndices.begin(); it!=nonRandomIndices.end(); it++){
	      l = *it;
	      dLsa_k_dsa_i *= (l == deriv_index) ?
		polynomialBasis[l][sm_index[l]].get_gradient(x[l], key_k[l]):
		polynomialBasis[l][sm_index[l]].get_value(x[l],    key_k[l]);
	    }
	    tpVarianceGrad[i] +=
	      wt_prod_j * expansionCoeffs[coeff_index[j]] *
	      expansionCoeffs[coeff_index[k]] *
	      (Lsa_j * dLsa_k_dsa_i + dLsa_j_dsa_i * Lsa_k);
	  }
	}
      }
    }

    // subtract 2 mu dmu/ds
    tpVarianceGrad[i] -= 2. * mean * mean_grad_i;

    if (randomVarsKey[deriv_index]) // deriv w.r.t. design var insertion
      cntr++;
  }
  return tpVarianceGrad;
}


const Real& InterpPolyApproximation::get_value(const RealVector& x)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_value()" << endl;
    abort_handler(-1);
  }

  // sum expansion to get response prediction
  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_value(x, 0); // first and only collocKey
    break;
  case SPARSE_GRID: {
    // Smolyak recursion of anisotropic tensor products
    approxValue = 0.;
    size_t i, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++)
      approxValue += smolyakCoeffs[i] * tensor_product_value(x, i);
    return approxValue;
    break;
  }
  }
}


const RealBaseVector& InterpPolyApproximation::get_gradient(const RealVector& x)
{
  // this could define a default_dvv and call get_gradient(x, dvv),
  // but we want this fn to be as fast as possible

  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_gradient()" << endl;
    abort_handler(-1);
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_gradient(x, 0); // first and only collocKey
    break;
  case SPARSE_GRID: {
    if (approxGradient.length() != numVars)
      approxGradient.reshape(numVars);
    approxGradient = 0.;
    // Smolyak recursion of anisotropic tensor products
    size_t i, j, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++) {
      const Real&           coeff   = smolyakCoeffs[i];
      const RealBaseVector& tp_grad = tensor_product_gradient(x, i);
      for (j=0; j<numVars; j++)
	approxGradient[j] += coeff * tp_grad[j];
    }
    return approxGradient;
    break;
  }
  }
}


const RealBaseVector& InterpPolyApproximation::
get_gradient(const RealVector& x, const UIntArray& dvv)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_gradient()" << endl;
    abort_handler(-1);
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_gradient(x, 0, dvv); // first and only collocKey
    break;
  case SPARSE_GRID: {
    size_t num_deriv_vars = dvv.length();
    if (approxGradient.length() != num_deriv_vars)
      approxGradient.reshape(num_deriv_vars);
    approxGradient = 0.;
    // Smolyak recursion of anisotropic tensor products
    size_t i, j, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++) {
      const Real&           coeff   = smolyakCoeffs[i];
      const RealBaseVector& tp_grad = tensor_product_gradient(x, i, dvv);
      for (j=0; j<num_deriv_vars; j++)
	approxGradient[j] += coeff * tp_grad[j];
    }
    return approxGradient;
    break;
  }
  }
}


/** In this case, all expansion variables are random variables and the
    mean of the expansion is simply the sum over i of r_i w_i. */
const Real& InterpPolyApproximation::get_mean()
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_mean()" << endl;
    abort_handler(-1);
  }

  expansionMean = 0.;
  const RealVector& wt_products = integrationRep->weight_products();
  for (size_t i=0; i<numCollocPts; i++)
    expansionMean += expansionCoeffs[i] * wt_products[i];
#ifdef DEBUG
  Cout << "Active Vars: Mean = " << expansionMean << '\n';
#endif // DEBUG
  return expansionMean;
}


/** In this case, a subset of the expansion variables are random
    variables and the mean of the expansion involves evaluating the
    expectation over this subset. */
const Real& InterpPolyApproximation::get_mean(const RealVector& x)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_mean()" << endl;
    abort_handler(-1);
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_mean(x, 0);
    break;
  case SPARSE_GRID: {
    expansionMean = 0.;
    size_t i, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++)
      expansionMean += smolyakCoeffs[i] * tensor_product_mean(x, i);
    return expansionMean;
    break;
  }
  }
#ifdef DEBUG
  Cout << "All Vars: Mean = " << expansionMean << '\n';
#endif // DEBUG
}


/** In this function, all expansion variables are random variables and
    any design/state variables are omitted from the expansion.  In
    this case, the derivative of the expectation is the expectation of
    the derivative.  The mixed derivative case (some design variables
    are inserted and some are augmented) requires no special treatment. */
const RealBaseVector& InterpPolyApproximation::get_mean_gradient()
{
  // d/ds <R> = <dR/ds>

  // Error check for required data
  if (!expansionGradFlag) {
    Cerr << "Error: expansion coefficient gradients not defined in "
	 << "InterpPolyApproximation::get_mean_gradient()." << endl;
    abort_handler(-1);
  }

  const RealVector& wt_products = integrationRep->weight_products();
  size_t i, j, num_deriv_vars = expansionCoeffGrads.num_columns();
  if (expansionMeanGrad.length() != num_deriv_vars)
    expansionMeanGrad.reshape(num_deriv_vars);
  expansionMeanGrad = 0.;
  for (i=0; i<numCollocPts; i++) {
    const Real&           wt_prod_i        = wt_products[i];
    const RealBaseVector& exp_coeff_grad_i = expansionCoeffGrads[i];
    for (j=0; j<num_deriv_vars; j++)
      expansionMeanGrad[j] += exp_coeff_grad_i[j] * wt_prod_i;
  }
  return expansionMeanGrad;
}


/** In this function, a subset of the expansion variables are random
    variables and any augmented design/state variables (i.e., not
    inserted as random variable distribution parameters) are included
    in the expansion.  In this case, the mean of the expansion is the
    expectation over the random subset and the derivative of the mean
    is the derivative of the remaining expansion over the non-random
    subset.  This function must handle the mixed case, where some
    design/state variables are augmented (and are part of the
    expansion: derivatives are evaluated as described above) and some
    are inserted (derivatives are obtained from expansionCoeffGrads). */
const RealBaseVector& InterpPolyApproximation::
get_mean_gradient(const RealVector& x, const UIntArray& dvv)
{
  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_mean_gradient(x, 0, dvv);
    break;
  case SPARSE_GRID:
    size_t num_deriv_vars = dvv.length();
    if (expansionMeanGrad.length() != num_deriv_vars)
      expansionMeanGrad.reshape(num_deriv_vars);
    expansionMeanGrad = 0.;
    // Smolyak recursion of anisotropic tensor products
    size_t i, j, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++) {
      const Real& coeff = smolyakCoeffs[i];
      const RealBaseVector& tpm_grad = tensor_product_mean_gradient(x, i, dvv);
      for (j=0; j<num_deriv_vars; j++)
	expansionMeanGrad[j] += coeff * tpm_grad[j];
    }
    return expansionMeanGrad;
    break;
  }
}


/** In this case, all expansion variables are random variables and the
    variance of the expansion is the sum over all but the first term
    of the coefficients squared times the polynomial norms squared. */
const Real& InterpPolyApproximation::get_variance()
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_variance()" << endl;
    abort_handler(-1);
  }

  expansionVariance = 0.;
  Real mean = 0.;
  const RealVector& wt_products = integrationRep->weight_products();
  for (size_t i=0; i<numCollocPts; i++) {
    const Real& exp_coeff_i = expansionCoeffs[i];
    const Real& wt_prod_i   = wt_products[i];
    Real coeff_wt_i = exp_coeff_i * wt_prod_i;
    mean              += coeff_wt_i;
    expansionVariance += exp_coeff_i * coeff_wt_i;
  }
  expansionVariance -= mean*mean;
#ifdef DEBUG
  Cout << "Active Vars: Variance = " << expansionVariance
       << " mean = " << mean << '\n';
#endif // DEBUG
  return expansionVariance;
}


/** In this case, a subset of the expansion variables are random
    variables and the variance of the expansion involves summations
    over this subset. */
const Real& InterpPolyApproximation::get_variance(const RealVector& x)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_variance()" << endl;
    abort_handler(-1);
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_variance(x, 0);
    break;
  case SPARSE_GRID:
    expansionVariance = 0.;
    size_t i, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++)
      expansionVariance	+= smolyakCoeffs[i] * tensor_product_variance(x, i);
    return expansionVariance;
    break;
  }
#ifdef DEBUG
  Cout << "All Vars: Variance = " << expansionVariance << '\n';
#endif // DEBUG
}


/** In this function, all expansion variables are random variables and
    any design/state variables are omitted from the expansion.  The
    mixed derivative case (some design variables are inserted and some
    are augmented) requires no special treatment. */
const RealBaseVector& InterpPolyApproximation::get_variance_gradient()
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_variance_gradient()" << endl;
    abort_handler(-1);
  }
  if (!expansionGradFlag) {
    Cerr << "Error: expansion coefficient gradients not defined in "
	 << "InterpPolyApproximation::get_variance_gradient()." << endl;
    abort_handler(-1);
  }

  const RealVector& wt_products = integrationRep->weight_products();
  size_t i, j, num_deriv_vars = expansionCoeffGrads.num_columns();
  if (expansionVarianceGrad.length() != num_deriv_vars)
    expansionVarianceGrad.reshape(num_deriv_vars);
  expansionVarianceGrad = 0.;

  Real mean = 0.;
  for (i=0; i<numCollocPts; i++)
    mean += expansionCoeffs[i] * wt_products[i];
  for (i=0; i<numCollocPts; i++) {
    const RealBaseVector& exp_coeff_grad_i = expansionCoeffGrads[i];
    Real term_i = 2. * (expansionCoeffs[i] - mean) * wt_products[i];
    for (j=0; j<num_deriv_vars; j++)
      expansionVarianceGrad[j] += term_i * exp_coeff_grad_i[j];
  }

  return expansionVarianceGrad;
}


/** In this function, a subset of the expansion variables are random
    variables and any augmented design/state variables (i.e., not
    inserted as random variable distribution parameters) are included
    in the expansion.  This function must handle the mixed case, where
    some design/state variables are augmented (and are part of the
    expansion) and some are inserted (derivatives are obtained from
    expansionCoeffGrads). */
const RealBaseVector& InterpPolyApproximation::
get_variance_gradient(const RealVector& x, const UIntArray& dvv)
{
  // Error check for required data
  if (!expansionCoeffFlag) {
    Cerr << "Error: expansion coefficients not defined in "
	 << "InterpPolyApproximation::get_variance_gradient()" << endl;
    abort_handler(-1);
  }

  switch (expCoeffsSolnApproach) {
  case QUADRATURE:
    return tensor_product_variance_gradient(x, 0, dvv);
    break;
  case SPARSE_GRID:
    size_t num_deriv_vars = dvv.length();
    if (expansionVarianceGrad.length() != num_deriv_vars)
      expansionVarianceGrad.reshape(num_deriv_vars);
    expansionVarianceGrad = 0.;
    // Smolyak recursion of anisotropic tensor products
    size_t i, j, num_smolyak_indices = smolyakMultiIndex.length();
    for (i=0; i<num_smolyak_indices; i++) {
      const Real& coeff = smolyakCoeffs[i];
      const RealBaseVector& tpv_grad
	= tensor_product_variance_gradient(x, i, dvv);
      for (j=0; j<num_deriv_vars; j++)
	expansionVarianceGrad[j] += coeff * tpv_grad[j];
    }
    return expansionVarianceGrad;
    break;
  }
}

} // namespace Dakota
