/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        SurfpackApproximation
//- Description:  Class implementation of Surfpack response surface 
//-               
//- Owner:        Mark Richards, Sandia National Laboratories

#include "surfpack_config.h"
#include <stdexcept>
#include <string>
#include <typeinfo>

#include "SurfpackApproximation.H"
#include "ProblemDescDB.H"

// Headers from Surfpack
#include "SurfData.h"
#include "ModelFactory.h"
#include "ModelFitness.h"
#include "LinearRegressionModel.h"

namespace Dakota {

/// \todo The dakota data structures like RealVector inherit from std::vector.
/// and are passed as is to Surfpack methods and functions-- which expect
/// structures of type std::vector.  This is legal as long as the types 
/// Surfpack expects really are superclasses of their Dakota counterparts.
/// It will fail if a Real is #defined in Dakota to be a float,
/// because then it will be a subclass of std::vector<float> being passed in
/// as a std::vector<double>.  The possible solutions, I think, are:
/// 1. Have the same typedef feature in Surfpack so that everything can
/// be configured to use floats or doubles
/// 2. Explicitly cast each vector component as it passes over
/// the Dakota/Surfpack boundary

/** Initialize the embedded Surfpack surface object and configure it
    using the specifications from the input file.  Data for the
    surface is created later. */

using surfpack::toString;
using surfpack::fromVec;
SurfpackApproximation::
SurfpackApproximation(const ProblemDescDB& problem_db, const size_t& num_acv):
  Approximation(BaseConstructor(), problem_db, num_acv),
//  surface(NULL), 
  surfData(NULL), model(NULL), factory(NULL)
{
    ParamMap args;
    args["ndims"] = toString<size_t>(num_acv);

    // For Polynomial surface fits
    if (approxType == "global_polynomial") {
      args["type"] = "polynomial";
      approxOrder = problem_db.get_short("model.surrogate.polynomial_order");
      args["order"] = toString<short>(approxOrder);
    }
       
    // For Kriging surface fits
    else if (approxType == "global_kriging") {
      args["type"] = "kriging";
      const RealVector& conmin_seed 
	= problem_db.get_drv("model.surrogate.kriging_conmin_seed");
      if (!conmin_seed.empty()) {
        Cout << "There are conmin_seed values" << endl;
        args["conmin_seed"] = fromVec<Real>(conmin_seed);
      }
      const RealVector& correlation_vector
        = problem_db.get_drv("model.surrogate.kriging_correlations");
      if (!correlation_vector.empty()) {
        Cout << "There are correlation values" << endl;
        args["correlations"] = fromVec<Real>(correlation_vector);
      }
     
      const RealVector& max_correlations 
        = problem_db.get_drv("model.surrogate.kriging_max_correlations");
      if (!max_correlations.empty()) {
        Cout << "There are max_correlation values" << endl;
        args["max_correlations"] = fromVec<Real>(max_correlations);
      }
      const RealVector& min_correlations 
        = problem_db.get_drv("model.surrogate.kriging_min_correlations");
      if (!min_correlations.empty()) {
        args["min_correlations"] = fromVec<Real>(min_correlations);
      }
      const short& max_iter
	= problem_db.get_short("model.surrogate.kriging_max_trials");
      if (max_iter > 0) {
        args["max_iter"] = toString<short>(max_iter);
      }
   } //

    // For ANN surface fits
    else if (approxType == "global_neural_network") {
      args["type"] = "ann";
      const short& random_weight
	= problem_db.get_short("model.surrogate.neural_network_random_weight");
      if (random_weight > 0) {
        args["random_weight"] = toString<short>(random_weight);
      }
      const short& nodes
	= problem_db.get_short("model.surrogate.neural_network_nodes");
      if (nodes > 0) {
        args["nodes"] = toString<short>(nodes);
      }
      const Real& range
	= problem_db.get_real("model.surrogate.neural_network_range");
      if (range > 0) {
        args["range"] = toString<Real>(range);
      }
    }

    //// For moving least squares
    else if (approxType == "global_moving_least_squares") {
      args["type"] = "mls";
      const short& weight
	= problem_db.get_short("model.surrogate.mls_weight_function");
      if (weight > 0) {
        args["weight"] = toString<short>(weight);
      }
      const short& order
	= problem_db.get_short("model.surrogate.mls_poly_order");
      if (order > 0) {
        args["order"] = toString<short>(order);
      }
    }

    //// For radial basis function networks
    else if (approxType == "global_radial_basis") {
      args["type"] = "rbf";
      const short& bases = problem_db.get_short("model.surrogate.rbf_bases");
      if (bases > 0) {
        args["bases"] = toString<short>(bases);
      }
      const short& min_partition
	= problem_db.get_short("model.surrogate.rbf_min_partition");
      if (min_partition > 0) {
        args["min_partition"] = toString<short>(min_partition);
      }
      const short& max_subsets
	= problem_db.get_short("model.surrogate.rbf_max_subsets");
      if (max_subsets > 0) {
        args["max_iter"] = toString<short>(max_subsets);
      }
      const short& max_pts
	= problem_db.get_short("model.surrogate.rbf_max_pts");
      if (max_pts > 0) {
        args["max_pts"] = toString<short>(max_pts);
      }
    }

    //// For Mars surface fits
    else if (approxType == "global_mars") {
      args["type"] = "mars";
      const short& max_bases
	= problem_db.get_short("model.surrogate.mars_max_bases");
      if (max_bases > 0) {
        args["max_bases"] = toString<short>(max_bases);
      }
      const String& interpolation
	= problem_db.get_string("model.surrogate.mars_interpolation");
      if (interpolation != "") {
        args["interpolation"] = interpolation; 
      }
    }
   //Cout << "PARAMETERS: " << endl;
   //for (ParamMap::iterator itr = args.begin();
   //     itr != args.end(); itr++) {
   //   Cout << "     " << itr->first << ": " << itr->second << std::endl;
   // }

    factory = ModelFactory::createModelFactory(args);


  //}
  //catch(...) {
  //  Cout << "Exception caught in attempt to create Surface object" << endl;
  //  abort_handler(-1);
  //}
}


// Embedded Surfpack objects will need to be deleted
SurfpackApproximation::~SurfpackApproximation()
{
  delete surfData;
  delete model;
  delete factory;
}


int SurfpackApproximation::min_coefficients() const
{

  if (approxType == "global_polynomial") {
    // bypass eqConRHS in Surfpack PolynomialSurface::minPointsRequired()
    switch (approxOrder) {
    case 3:
      return (numVars*numVars*numVars + 6*numVars*numVars + 11*numVars + 6)/6;
      break;
    case 2:
      return (numVars+1)*(numVars+2)/2;
      break;
    case 1:
      return numVars+1;
      break;
    default:
      Cerr << "Error: bad approximation order for global polynomial in "
	   << "SurfpackApproximation::min_coefficients()." << endl;
      abort_handler(-1);
    }
  }
  else {
    assert(factory);
    return factory->minPointsRequired();
  }
}


int SurfpackApproximation::recommended_coefficients() const
{
  if (approxType == "global_polynomial") {
    return min_coefficients();
  }
  else {
    assert(factory);
    return factory->recommendedNumPoints();
  }
}


void SurfpackApproximation::find_coefficients()
{
  // Surface object should have been created in constructor
  if (!factory) { 
    Cerr << "Error: surface is null in find_coefficients" << endl;  
    abort_handler(-1);
  }

  /// surfData will be deleted in dtor
  /// \todo Right now, we're completely deleting the old data and then
  /// recopying the current data into a SurfData object.  This was just
  /// the easiest way to arrive at a solution that would build and run.
  /// This function is frequently called from addPoint rebuild, however,
  /// and it's not good to go through this whole process every time one
  /// more data point is added.
  try {
    if (surfData) {
      delete surfData;
      surfData = NULL;
    }
    surfData = surrogates_to_surf_data();
    if (approxType == "global_polynomial")
      checkForEqualityConstraints();
    model = factory->Build(*surfData); 
    if (outputLevel > NORMAL_OUTPUT) { 

      Cout << model->asString();

      //      ModelFitness* SS_fitness = ModelFitness::Create("sum_squared");
      //Cout << "Sum-squared goodness of fit =  " 
      //   << (*SS_fitness)(*model,*surfData) << "\n" ;
      //delete SS_fitness;

      //ModelFitness* R2_fitness = ModelFitness::Create("rsquared");
      //Cout << "R-squared goodness of fit =  " 
      //   << (*R2_fitness)(*model,*surfData) << "\n" ;
      //delete R2_fitness;

    }
    // TO DO: extract coefficients array
  }
  catch (std::runtime_error& e) {
    Cerr << e.what() << endl;
    Cerr << typeid(e).name() << endl;
    abort_handler(-1);
  }
  catch (std::string& e) {
    Cerr << "Error: exception with no recourse caught trying to build model:\n"
	 << e << endl;
    abort_handler(-1);
  }
  catch (...) {
    Cerr << "Error: exception caught trying to build model" << endl;
    abort_handler(-1);
  }
}


const Real& SurfpackApproximation::get_value(const RealVector& x)
{ 
  static int times_called = 0;
  if (!model) { 
    Cerr << "Error: surface is null in get_value" << endl;  
    abort_handler(-1);
  }
  // check incoming x for correct length
  else if (x.length() != numVars) {
    Cerr << "Error: bad parameter vector length in SurfpackApproximation::"
	 << "get_value" << endl;
    abort_handler(-1);
  } 
  approxValue = (*model)(x);
  //Cout << "SurfpackApproximation::get_value times called " << ++times_called
  //     << endl; 
  return approxValue;
}


const RealBaseVector& SurfpackApproximation::get_gradient(const RealVector& x)
{
  assert(surfData);
  approxGradient.resize(x.size());
  try {
      VecDbl local_grad = model->gradient(x);
      for (unsigned i = 0; i < surfData->xSize(); i++) {
        *(approxGradient.begin() + i) = local_grad[i];
      }
  }
  catch (...) {
    Cerr << "Error: get_gradient() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }
  return approxGradient;
}


const RealMatrix& SurfpackApproximation::get_hessian(const RealVector& x)
{
  approxHessian.reshape_2d(x.size(),x.size());
  try {
    if (approxType == "global_moving_least_squares") {
      Cerr << "Have not implemented analytical hessians in this surfpack class"
	   << endl;
      abort_handler(-1);
    }
    MtxDbl sm = model->hessian(x);
    ///\todo Make this acceptably efficient
    for (size_t i = 0; i < x.size(); i++)
      for(size_t j = 0; j < x.size(); j++)
        approxHessian[i][j] = sm(i,j);
  }
  catch (...) {
    Cerr << "Error: get_hessian() not available for this approximation type."
	 << endl;
    abort_handler(-1);
  }
  return approxHessian;
}

const bool SurfpackApproximation::diagnostics_available()
{
    return true;
}

const Real& SurfpackApproximation::get_diagnostic(const String& metric_type)
{ 
  static int times_called = 0;
  if (!model) { 
    Cerr << "Error: surface is null in get_diagnostic" << endl;  
    abort_handler(-1);
  }

  ModelFitness* SS_fitness = ModelFitness::Create(metric_type);
  approxDiagnostic = (*SS_fitness)(*model,*surfData);
  Cout << "The " << metric_type << " goodness of fit =  " 
       << approxDiagnostic  << "\n" ;
  delete SS_fitness;
     
  return approxDiagnostic;
}

/** Copy the data stored in Dakota-style SurrogateDataPoint objects into
    Surfpack-style SurfPoint and SurfData objects. */
SurfData* SurfpackApproximation::surrogates_to_surf_data()
{
  SurfData* surf_data = new SurfData();
  if (!anchorPoint.is_null() && approxType != "global_polynomial")
    surf_data->addPoint(SurfPoint(anchorPoint.continuous_variables(),
				  anchorPoint.response_function()));
  for (List<SurrogateDataPoint>::iterator it=currentPoints.begin();
       it!=currentPoints.end(); it++)
    surf_data->addPoint(SurfPoint(it->continuous_variables(),
				  it->response_function()));
  return surf_data;
}


/** If there is an anchor point, add an equality constraint for its response
    value.  Also add constraints for gradient and hessian, if applicable. */
void SurfpackApproximation::checkForEqualityConstraints()
{
  assert(factory);
  if (!anchorPoint.is_null()) {
    // We will use logical OR to accumulate which constraints are present:
    // 1 for a response value,
    // 2 for a gradient vector,
    // 4 for a hessian matrix

    // Print out the continuous design variables
    vector<double> vars = anchorPoint.continuous_variables();
    if (outputLevel > NORMAL_OUTPUT) {
      Cout << "Anchor point continuous vars\n";
      anchorPoint.continuous_variables().write(Cout);
    }

    // At a minimum, there should be a response value
    int requested_constraints = 1;
    double response_value = anchorPoint.response_function();
    if (outputLevel > NORMAL_OUTPUT)
      Cout << "Anchor response: " << response_value << '\n';

    // Check for gradient in anchor point
    vector<double> gradient = anchorPoint.response_gradient();
    // Print the gradient out, if present
    if (!gradient.empty()) {
      requested_constraints |= 2;
      if (outputLevel > NORMAL_OUTPUT) {
        Cout << "Anchor gradient:\n";
        copy(gradient.begin(),gradient.end(),
	     ostream_iterator<double>(Cout," "));
        Cout << '\n';
      }
    }
    
    // Check for hessian in anchor point
    MtxDbl hessian;
    if (!(anchorPoint.response_hessian().empty())) {
       requested_constraints |= 4;
       const RealMatrix& rm = anchorPoint.response_hessian();
       hessian.reshape(rm.size(),rm[0].size());
       ///\todo improve efficiency of conversion
       for (size_t i = 0; i < rm.size();i++) {
         for (size_t j = 0; j < rm[i].size();i++) {
           if (outputLevel > NORMAL_OUTPUT)
	     Cout << "hessian[" << i << "][" << j << "]: " << rm[i][j] << " ";
           hessian(i,j) = rm[i][j];
         }
         if (outputLevel > NORMAL_OUTPUT)
	   Cout << '\n';
       }
    }
    SurfPoint sp(vars,response_value);
    if (outputLevel > NORMAL_OUTPUT)
      Cout << "Requested contraints vector: " << requested_constraints << '\n';
    dynamic_cast<LinearRegressionModelFactory*>(factory)->setEqualityConstraints
      (requested_constraints, sp, response_value, &gradient, &hessian);
  }
}

} // namespace Dakota
