/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:        DistinctConstraints
//- Description:  Class implementation
//- Owner:        Mike Eldred

#include "DistinctConstraints.H"
#include "ProblemDescDB.H"

static const char rcsId[]="@(#) $Id";


namespace Dakota {

/** In this class, the distinct approach (design, uncertain, and state
    types are distinct) is used.  Most iterators/strategies use this
    approach, which is the default in Constraints::get_constraints(). */
DistinctConstraints::
DistinctConstraints(const ProblemDescDB& problem_db,
		    const pair<short,short>& view):
  Constraints(BaseConstructor(), problem_db, view),
  continuousDesignLowerBnds(
    problem_db.get_drv("variables.continuous_design.lower_bounds")),
  continuousDesignUpperBnds(
    problem_db.get_drv("variables.continuous_design.upper_bounds")),
  discreteDesignLowerBnds(
    problem_db.get_div("variables.discrete_design.lower_bounds")),
  discreteDesignUpperBnds(
    problem_db.get_div("variables.discrete_design.upper_bounds")),
  uncertainLowerBnds(
    problem_db.get_drv("variables.uncertain.lower_bounds")),
  uncertainUpperBnds(
    problem_db.get_drv("variables.uncertain.upper_bounds")),
  continuousStateLowerBnds(
    problem_db.get_drv("variables.continuous_state.lower_bounds")),
  continuousStateUpperBnds(
    problem_db.get_drv("variables.continuous_state.upper_bounds")),
  discreteStateLowerBnds(
    problem_db.get_div("variables.discrete_state.lower_bounds")),
  discreteStateUpperBnds(
    problem_db.get_div("variables.discrete_state.upper_bounds"))
{
#ifdef REFCOUNT_DEBUG
  Cout << "Letter instantiated: variablesView active = " << variablesView.first
       << " inactive = " << variablesView.second << endl;
#endif

  // Manage linear constraints.  Verify specification sanity and set defaults
  // if needed.
  if (variablesView.first == MIXED_DISTINCT_UNCERTAIN) {
    if ( problem_db.get_drv("method.linear_inequality_constraints").length()
      || problem_db.get_drv("method.linear_equality_constraints").length() )
      Cerr << "Warning: linear constraints not supported with nondeterministic "
	   << "iterators\n         Input will be ignored.";
    numLinearIneqCons = numLinearEqCons = 0;
  }
  else
    manage_linear_constraints(problem_db);
}


void DistinctConstraints::read(istream& s)
{
  continuousDesignLowerBnds.read(s);
  discreteDesignLowerBnds.read(s);
  uncertainLowerBnds.read(s);
  continuousStateLowerBnds.read(s);
  discreteStateLowerBnds.read(s);

  continuousDesignUpperBnds.read(s);
  discreteDesignUpperBnds.read(s);
  uncertainUpperBnds.read(s);
  continuousStateUpperBnds.read(s);
  discreteStateUpperBnds.read(s);
}


void DistinctConstraints::write(ostream& s) const
{
  continuousDesignLowerBnds.write(s);
  discreteDesignLowerBnds.write(s);
  uncertainLowerBnds.write(s);
  continuousStateLowerBnds.write(s);
  discreteStateLowerBnds.write(s);

  continuousDesignUpperBnds.write(s);
  discreteDesignUpperBnds.write(s);
  uncertainUpperBnds.write(s);
  continuousStateUpperBnds.write(s);
  discreteStateUpperBnds.write(s);
}


void DistinctConstraints::copy_rep(const Constraints* con_rep)
{
  // copy derived class attributes
  continuousDesignLowerBnds
    = ((DistinctConstraints*)con_rep)->continuousDesignLowerBnds;
  continuousDesignUpperBnds
    = ((DistinctConstraints*)con_rep)->continuousDesignUpperBnds;
  discreteDesignLowerBnds
    = ((DistinctConstraints*)con_rep)->discreteDesignLowerBnds;
  discreteDesignUpperBnds
    = ((DistinctConstraints*)con_rep)->discreteDesignUpperBnds;
  uncertainLowerBnds = ((DistinctConstraints*)con_rep)->uncertainLowerBnds;
  uncertainUpperBnds = ((DistinctConstraints*)con_rep)->uncertainUpperBnds;
  continuousStateLowerBnds
    = ((DistinctConstraints*)con_rep)->continuousStateLowerBnds;
  continuousStateUpperBnds
    = ((DistinctConstraints*)con_rep)->continuousStateUpperBnds;
  discreteStateLowerBnds
    = ((DistinctConstraints*)con_rep)->discreteStateLowerBnds;
  discreteStateUpperBnds
    = ((DistinctConstraints*)con_rep)->discreteStateUpperBnds;
}


void DistinctConstraints::reshape_rep(const Sizet2DArray& vars_comps)
{
  size_t num_cdv = vars_comps[0][1], num_ddv = vars_comps[0][2],
         num_uv  = vars_comps[1][0],
         num_csv = vars_comps[2][1], num_dsv = vars_comps[2][2];
  continuousDesignLowerBnds.reshape(num_cdv);
  continuousDesignUpperBnds.reshape(num_cdv);
  discreteDesignLowerBnds.reshape(num_ddv);
  discreteDesignUpperBnds.reshape(num_ddv);
  uncertainLowerBnds.reshape(num_uv);
  uncertainUpperBnds.reshape(num_uv);
  continuousStateLowerBnds.reshape(num_csv);
  continuousStateUpperBnds.reshape(num_csv);
  discreteStateLowerBnds.reshape(num_dsv);
  discreteStateUpperBnds.reshape(num_dsv);
}

} // namespace Dakota
