/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       Iterator
//- Description: Implementation code for the Iterator class
//- Owner:       Mike Eldred
//- Checked by:

#include "DakotaIterator.H"
#include "ParamStudy.H"
#include "NonDPolynomialChaos.H"
#include "NonDStochCollocation.H"
#include "NonDLocalReliability.H"
#include "NonDGlobalReliability.H"
#include "NonDLHSSampling.H"
#include "NonDIncremLHSSampling.H"
#include "SurrBasedLocalMinimizer.H"
#include "SurrBasedGlobalMinimizer.H"
#include "EffGlobalMinimizer.H"
#ifdef DAKOTA_EVIDENCE
#include "NonDEvidence.H"
#endif
#ifdef DAKOTA_DDACE
#include "DDACEDesignCompExp.H"
#endif
#ifdef DAKOTA_FSUDACE
#include "FSUDesignCompExp.H"
#endif
#ifdef DAKOTA_PSUADE
#include "PSUADEDesignCompExp.H"
#endif
#ifdef DAKOTA_DOT
#include "DOTOptimizer.H"
#endif
#ifdef DAKOTA_CONMIN
#include "CONMINOptimizer.H"
#endif
#ifdef DAKOTA_DL_SOLVER
#include "DLSolver.H"
#endif
#ifdef DAKOTA_NPSOL
#include "NPSOLOptimizer.H"
#include "NLSSOLLeastSq.H"
#endif
#ifdef DAKOTA_NLPQL
#include "NLPQLPOptimizer.H"
#endif
#ifdef DAKOTA_NL2SOL
#include "NL2SOLLeastSq.H"
#endif
#ifdef DAKOTA_RSQP
#include "rSQPOptimizer.H"
#endif
#ifdef DAKOTA_OPTPP
#include "SNLLOptimizer.H"
#include "SNLLLeastSq.H"
#endif
#ifdef DAKOTA_COLINY
#include "COLINOptimizer.H"
#endif
#ifdef DAKOTA_APPS
#include "APPSOptimizer.H"
#endif
#ifdef DAKOTA_NCSU
#include "NCSUOptimizer.H"
#endif
#ifdef DAKOTA_JEGA
#include "JEGAOptimizer.H"
#endif
#include "ProblemDescDB.H"
#include "ParallelLibrary.H"
#include "DakotaGraphics.H"

static const char rcsId[]="@(#) $Id: DakotaIterator.C 5412 2008-10-23 03:49:52Z mseldre $";

namespace Dakota {

extern ProblemDescDB dummy_db;    // defined in global_defs.C


/** This constructor builds the base class data for all inherited
    iterators.  get_iterator() instantiates a derived class and the
    derived class selects this base class constructor in its
    initialization list (to avoid the recursion of the base class
    constructor calling get_iterator() again).  Since the letter IS
    the representation, its representation pointer is set to NULL (an
    uninitialized pointer causes problems in ~Iterator). */
Iterator::Iterator(BaseConstructor, Model& model):
  probDescDB(model.problem_description_db()),
  methodName(probDescDB.get_string("method.algorithm")),
  convergenceTol(probDescDB.get_real("method.convergence_tolerance")),
  maxIterations(probDescDB.get_int("method.max_iterations")),
  maxFunctionEvals(probDescDB.get_int("method.max_function_evaluations")),
  maxConcurrency(model.derivative_concurrency()),
  numFunctions(model.num_functions()), numContinuousVars(model.cv()),
  numDiscreteVars(model.dv()), activeSet(model.current_response().active_set()),
  subIteratorFlag(false),
  gradientType(probDescDB.get_string("responses.gradient_type")),
  methodSource(probDescDB.get_string("responses.method_source")),
  intervalType(probDescDB.get_string("responses.interval_type")),
  hessianType(probDescDB.get_string("responses.hessian_type")),
  fdGradStepSize(0.001), fdHessByGradStepSize(0.001), fdHessByFnStepSize(0.002),
  // Output verbosity is observed within Iterator (algorithm verbosity), 
  // Model (synchronize/estimate_derivatives verbosity), Interface 
  // (map/synch verbosity), Approximation (global data fit coefficient
  // reporting), and AnalysisCode (file operations verbosity) as follows:
  //               iterator     model     interface   approx    file ops
  //   "silent" :   silent      silent     silent     quiet      quiet 
  //   "quiet"  :    quiet      quiet       quiet     quiet      quiet
  //   "normal" :   normal      normal     normal     quiet      quiet 
  //   "verbose":   verbose     normal     verbose    verbose    verbose
  //   "debug"  :    debug      normal      debug     verbose    verbose
  // where "silent," "quiet", "verbose" and "debug" must be user specified and
  // "normal" is the default for no user specification.  Note that iterators
  // and interfaces have the most granularity in verbosity.
  outputLevel(probDescDB.get_short("method.output")),
  idMethod(probDescDB.get_string("method.id")), iteratorRep(NULL),
  referenceCount(1)
{
  // Check for active variables
  if ( numContinuousVars <= 0 && numDiscreteVars <= 0 ) {
    Cerr << "\nError: no active variables available in selected method.  Please"
	 << "\n       select variables that are active for " << methodName
	 << "." << endl;
    abort_handler(-1);
  }
  // Check for response functions
  if ( numFunctions <= 0 ) {
    Cerr << "\nError: number of response functions must be greater than zero."
	 << endl;
    abort_handler(-1);
  }

  // Populate gradient/Hessian attributes for use within the iterator hierarchy.
  // Note: the fd step size arrays specialize by variable, whereas the mixed
  // grads/Hessians specialize by function.
  Cout << "methodName = " << methodName << "\ngradientType = "
       << gradientType << '\n';
  if (gradientType == "numerical") {
    if (methodSource == "vendor") {
      const RealVector& fdgss
	= probDescDB.get_drv("responses.fd_gradient_step_size");
      if (fdgss.length()) // else use default from initializer list
	fdGradStepSize = fdgss[0];
    }
    Cout << "Numerical gradients using " << intervalType
	 << " differences\nto be calculated by the " << methodSource
	 << " finite difference routine.\n";
  }
  else if (gradientType == "mixed") {
    // Vendor numerical is no good in mixed mode except maybe for NPSOL/NLSSOL.
    if (methodSource == "vendor") {
      Cerr << "Error: Mixed gradient specification not currently valid with "
           << "vendor numerical.\nSelect dakota as method_source instead."
	   << endl;
      abort_handler(-1);
    }
    const IntList& mixed_grad_analytic_ids
      = probDescDB.get_dil("responses.gradients.mixed.id_analytic");
    const IntList& mixed_grad_numerical_ids
      = probDescDB.get_dil("responses.gradients.mixed.id_numerical");
    Cout << "Mixed gradients: analytic gradients for functions { ";
    for (ILCIter cit=mixed_grad_analytic_ids.begin();
	 cit!=mixed_grad_analytic_ids.end(); cit++)
      Cout << *cit << ' ';
    Cout << "} and\nnumerical gradients for functions { ";
    for (ILCIter cit=mixed_grad_numerical_ids.begin();
	 cit!=mixed_grad_numerical_ids.end(); cit++)
      Cout << *cit << ' ';
    Cout << "} using " << intervalType << " differences\ncalculated by the "
         << methodSource << " routine.\n";
  }

  Cout << "hessianType = " << hessianType << '\n';
  if ( hessianType == "numerical" || ( hessianType == "mixed" &&
      !probDescDB.get_dil("responses.hessians.mixed.id_numerical").empty() ) ) {
    const RealVector& fdhss
      = probDescDB.get_drv("responses.fd_hessian_step_size");
    if (fdhss.length()) // else use defaults from initializer list
      fdHessByGradStepSize = fdHessByFnStepSize = fdhss[0];
  }

#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(BaseConstructor, model) called to "
       << "build letter base class\n";
#endif
}


/** This alternate constructor builds base class data for inherited
    iterators.  It is used for on-the-fly instantiations for which DB
    queries cannot be used.  Therefore it only sets attributes taken
    from the incoming model. */
Iterator::Iterator(NoDBBaseConstructor, Model& model):
  probDescDB(dummy_db), convergenceTol(0.0001), maxIterations(100),
  maxFunctionEvals(1000), maxConcurrency(model.derivative_concurrency()),
  numFunctions(model.num_functions()), numContinuousVars(model.cv()),
  numDiscreteVars(model.dv()), activeSet(model.current_response().active_set()),
  subIteratorFlag(false), gradientType(model.gradient_type()),
  methodSource(model.method_source()), intervalType(model.interval_type()),
  hessianType(model.hessian_type()), fdGradStepSize(0.001),
  fdHessByGradStepSize(0.001), fdHessByFnStepSize(0.002),
  outputLevel(NORMAL_OUTPUT), idMethod("NO_DB_METHOD"), iteratorRep(NULL),
  referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(NoDBBaseConstructor, model) called "
       << "to build letter base class\n";
#endif
}


/** This alternate constructor builds base class data for inherited
    iterators.  It is used for on-the-fly instantiations for which DB
    queries cannot be used. It has no incoming model, so only sets up
    a minimal set of defaults. However, its use is preferable to the
    default constructor, which should remain as minimal as possible. */
Iterator::Iterator(NoDBBaseConstructor): probDescDB(dummy_db),
  convergenceTol(0.0001), maxIterations(100), maxFunctionEvals(1000),
  maxConcurrency(1), subIteratorFlag(false), gradientType("none"),
  hessianType("none"), fdGradStepSize(0.001), fdHessByGradStepSize(0.001),
  fdHessByFnStepSize(0.002), outputLevel(NORMAL_OUTPUT),
  idMethod("NO_DB_METHOD"), iteratorRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(NoDBBaseConstructor) called to build letter base "
       << "class\n";
#endif
}


/** The default constructor is used in Vector<Iterator>
    instantiations and for initialization of Iterator objects
    contained in Strategy derived classes (see derived class
    header files).  iteratorRep is NULL in this case (a populated
    problem_db is needed to build a meaningful Iterator object).
    This makes it necessary to check for NULL pointers in the copy
    constructor, assignment operator, and destructor. */
Iterator::Iterator(): probDescDB(dummy_db), maxConcurrency(1),
  iteratorRep(NULL), referenceCount(1)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator() called to build empty envelope "
       << "base class object." << endl;
#endif
}


/** Used in iterator instantiations within strategy constructors.
    Envelope constructor only needs to extract enough data to properly
    execute get_iterator(), since letter holds the actual base class data. */
Iterator::Iterator(Model& model):
  //iteratedModel(model), // no Model copy for envelope
  probDescDB(model.problem_description_db()), 
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(Model&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate iterator type
  iteratorRep = get_iterator(model);
  if ( !iteratorRep ) // bad name or insufficient memory
    abort_handler(-1);
}


/** Used only by the envelope constructor to initialize iteratorRep to the 
    appropriate derived type, as given by the methodName attribute. */
Iterator* Iterator::get_iterator(Model& model)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter: Getting iterator " << methodName 
       << endl;
#endif

  // These instantiations will NOT recurse on the Iterator(model) 
  // constructor due to the use of BaseConstructor.

  const String& method_name = probDescDB.get_string("method.algorithm");
  if (method_name.ends("_parameter_study"))
    return new ParamStudy(model);
  else if (method_name == "nond_polynomial_chaos")
    return new NonDPolynomialChaos(model);
  else if (method_name == "nond_stoch_collocation")
    return new NonDStochCollocation(model);
  else if (method_name == "nond_sampling") {
    const String& sample_type = probDescDB.get_string("method.sample_type");
    if (sample_type.begins("incremental"))
      return new NonDIncremLHSSampling(model);
    else  
      return new NonDLHSSampling(model);
  }
  else if (method_name == "surrogate_based_local")
    return new SurrBasedLocalMinimizer(model);
  else if (method_name == "surrogate_based_global")
    return new SurrBasedGlobalMinimizer(model);
  else if (method_name == "efficient_global")
    return new EffGlobalMinimizer(model);
  else if (method_name == "nond_local_reliability")
    return new NonDLocalReliability(model);
  else if (method_name == "nond_global_reliability")
    return new NonDGlobalReliability(model);
#ifdef DAKOTA_EVIDENCE
  else if (method_name == "nond_evidence")
    return new NonDEvidence(model);
#endif
#ifdef DAKOTA_OPTPP
  else if (method_name == "optpp_cg"        || method_name == "optpp_q_newton"
        || method_name == "optpp_fd_newton" || method_name == "optpp_newton"
        || method_name == "optpp_pds")
    return new SNLLOptimizer(model);
  else if (method_name == "optpp_g_newton")
    return new SNLLLeastSq(model);
#endif
#ifdef DAKOTA_APPS
  else if (method_name == "asynch_pattern_search")
    return new APPSOptimizer(model);
#endif
#ifdef DAKOTA_COLINY
#ifdef DAKOTA_3PO
  else if (method_name == "coliny_cobyla")
    return new COLINOptimizer<coliny::Cobyla>(model);
#endif
  else if (method_name == "coliny_direct")
    return new COLINOptimizer<coliny::DIRECT>(model);
  else if (method_name == "coliny_pattern_search")
    return new COLINOptimizer<coliny::PatternSearch>(model);
  else if (method_name == "coliny_multi_start")
    return new
      COLINOptimizer<coliny::MultiStart<utilib::BasicArray<double> > >(model);
  else if (method_name == "coliny_solis_wets")
    return new COLINOptimizer<coliny::SolisWets>(model);
  //else if (method_name == "coliny_pga_real")
  //  return new COLINOptimizer<coliny::PEAreal>(model);
  else if (method_name == "coliny_ea")
    return new COLINOptimizer<coliny::EAminlp>(model);
#endif
#ifdef DAKOTA_JEGA
  else if (method_name == "moga" || method_name == "soga")
    return new JEGAOptimizer(model);
#endif
#ifdef DAKOTA_DL_SOLVER
  else if (method_name == "dl_solver")
    return new DLSolver(model);
#endif
#ifdef DAKOTA_NPSOL
  else if (method_name == "npsol_sqp")
    return new NPSOLOptimizer(model);
  else if (method_name == "nlssol_sqp")
    return new NLSSOLLeastSq(model);
#endif
#ifdef DAKOTA_NLPQL
  else if (method_name == "nlpql_sqp")
    return new NLPQLPOptimizer(model);
#endif
#ifdef DAKOTA_NL2SOL
  else if (method_name == "nl2sol")
    return new NL2SOLLeastSq(model);
#endif
//#ifdef DAKOTA_RSQP
//  else if (method_name == "reduced_sqp")
//    return new rSQPOptimizer(model);
//#endif
#ifdef DAKOTA_DOT
  else if (method_name.begins("dot_"))
    return new DOTOptimizer(model);
#endif
#ifdef DAKOTA_CONMIN
  else if (method_name.begins("conmin_"))
    return new CONMINOptimizer(model);
#endif
#ifdef DAKOTA_DDACE
  else if (method_name == "dace")
    return new DDACEDesignCompExp(model);
#endif
#ifdef DAKOTA_FSUDACE
  else if (method_name.begins("fsu_"))
    return new FSUDesignCompExp(model);
#endif
#ifdef DAKOTA_PSUADE
  else if (method_name.begins("psuade_"))
    return new PSUADEDesignCompExp(model);
#endif
#ifdef DAKOTA_NCSU
  else if (method_name == "ncsu_direct")
    return new NCSUOptimizer(model);
#endif
  else {
    Cerr << "Invalid iterator: " << method_name << " not available." << endl;
    return NULL;
  }
}


/** Used in sub-iterator instantiations within iterator constructors.
    Envelope constructor only needs to extract enough data to properly
    execute get_iterator(), since letter holds the actual base class data. */
Iterator::Iterator(const String& method_name, Model& model):
  //iteratedModel(model), // no Model copy for envelope
  probDescDB(model.problem_description_db()), 
  referenceCount(1) // not used since this is the envelope, not the letter
{
#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(Model&) called to instantiate "
       << "envelope." << endl;
#endif

  // Set the rep pointer to the appropriate iterator type
  iteratorRep = get_iterator(method_name, model);
  if ( !iteratorRep ) // bad name or insufficient memory
    abort_handler(-1);
}


/** Used only by the envelope constructor to initialize iteratorRep to the 
    appropriate derived type, as given by the passed method_name. */
Iterator* Iterator::get_iterator(const String& method_name, Model& model)
{
#ifdef REFCOUNT_DEBUG
  Cout << "Envelope instantiating letter: Getting iterator " <<  method_name
       << " by name." << endl;
#endif

  // These instantiations will NOT recurse on the Iterator(model) 
  // constructor due to the use of BaseConstructor.

  //if (method_name == "surrogate_based_local")
  //  return new SurrBasedLocalMinimizer(NoDBBaseConstructor(), model);
  //else if (method_name == "surrogate_based_global")
  //  return new SurrBasedGlobalMinimizer(NoDBBaseConstructor(), model);
  //else if (method_name == "efficient_global")
  //  return new EffGlobalMinimizer(NoDBBaseConstructor(), model);

  if (false) { } // dummy anchor for else blocks to avoid issues with #ifdef's
#ifdef DAKOTA_OPTPP
  else if (method_name == "optpp_cg"   || method_name == "optpp_q_newton" ||
      method_name == "optpp_fd_newton" || method_name == "optpp_newton"   ||
      method_name == "optpp_pds")
    return new SNLLOptimizer(method_name, model);
  else if (method_name == "optpp_g_newton")
    return new SNLLLeastSq(method_name, model);
#endif
#ifdef DAKOTA_APPS
  else if (method_name == "asynch_pattern_search")
    return new APPSOptimizer(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_COLINY
#ifdef DAKOTA_3PO
  else if (method_name == "coliny_cobyla")
    return new COLINOptimizer<coliny::Cobyla>(NoDBBaseConstructor(), model);
#endif
  else if (method_name == "coliny_direct")
    return new COLINOptimizer<coliny::DIRECT>(NoDBBaseConstructor(), model);
  else if (method_name == "coliny_pattern_search")
    return new COLINOptimizer<coliny::PatternSearch>(NoDBBaseConstructor(),
    model);
  else if (method_name == "coliny_multi_start")
    return new
      COLINOptimizer<coliny::MultiStart<utilib::BasicArray<double> > >(
      NoDBBaseConstructor(), model);
  else if (method_name == "coliny_solis_wets")
    return new COLINOptimizer<coliny::SolisWets>(NoDBBaseConstructor(), model);
  //else if (method_name == "coliny_pga_real")
  //  return new COLINOptimizer<coliny::PEAreal>(NoDBBaseConstructor(), model);
  else if (method_name == "coliny_ea")
    return new COLINOptimizer<coliny::EAminlp>(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_JEGA
  //else if (method_name == "moga" || method_name == "soga")
  //  return new JEGAOptimizer(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_DL_SOLVER
  //else if (method_name == "dl_solver")
  //  return new DLSolver(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_NPSOL
  else if (method_name == "npsol_sqp")
    return new NPSOLOptimizer(NoDBBaseConstructor(), model);
  else if (method_name == "nlssol_sqp")
    return new NLSSOLLeastSq(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_NLPQL
  else if (method_name == "nlpql_sqp")
    return new NLPQLPOptimizer(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_NL2SOL
  else if (method_name == "nl2sol")
    return new NL2SOLLeastSq(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_DOT
  else if (method_name.begins("dot_"))
    return new DOTOptimizer(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_CONMIN
  else if (method_name.begins("conmin_"))
    return new CONMINOptimizer(NoDBBaseConstructor(), model);
#endif
#ifdef DAKOTA_NCSU
  else if (method_name == "ncsu_direct")
    return new NCSUOptimizer(NoDBBaseConstructor(), model);
#endif
  else {
    Cerr << "Invalid iterator: " << method_name << " not available by name."
	 << endl;
    return NULL;
  }
}


/** Copy constructor manages sharing of iteratorRep and incrementing
    of referenceCount. */
Iterator::Iterator(const Iterator& iterator):
  probDescDB(iterator.problem_description_db())
{
  // Increment new (no old to decrement)
  iteratorRep = iterator.iteratorRep;
  if (iteratorRep) // Check for an assignment of NULL
    iteratorRep->referenceCount++;

#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::Iterator(Iterator&)" << endl;
  if (iteratorRep)
    Cout << "iteratorRep referenceCount = " << iteratorRep->referenceCount
	 << endl;
#endif
}


/** Assignment operator decrements referenceCount for old iteratorRep, assigns
    new iteratorRep, and increments referenceCount for new iteratorRep. */
Iterator Iterator::operator=(const Iterator& iterator)
{
  if (iteratorRep != iterator.iteratorRep) { // normal case: old != new
    // Decrement old
    if (iteratorRep) // Check for NULL
      if ( --iteratorRep->referenceCount == 0 ) 
	delete iteratorRep;
    // Assign and increment new
    iteratorRep = iterator.iteratorRep;
    if (iteratorRep) // Check for NULL
      iteratorRep->referenceCount++;
  }
  // else if assigning same rep, then do nothing since referenceCount
  // should already be correct

#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::operator=(Iterator&)" << endl;
  if (iteratorRep)
    Cout << "iteratorRep referenceCount = " << iteratorRep->referenceCount
	 << endl;
#endif

  return *this; // calls copy constructor since returned by value
}


/** Destructor decrements referenceCount and only deletes iteratorRep
    when referenceCount reaches zero. */
Iterator::~Iterator()
{ 
  if (iteratorRep) { // Check for NULL pointer 
    --iteratorRep->referenceCount;
#ifdef REFCOUNT_DEBUG
    Cout << "iteratorRep referenceCount decremented to " 
         << iteratorRep->referenceCount << endl;
#endif
    if (iteratorRep->referenceCount == 0) {
#ifdef REFCOUNT_DEBUG
      Cout << "deleting iteratorRep" << endl;
#endif
      delete iteratorRep;
    }
  }
}


/** Similar to the assignment operator, the assign_rep() function
    decrements referenceCount for the old iteratorRep and assigns the
    new iteratorRep.  It is different in that it is used for
    publishing derived class letters to existing envelopes, as opposed
    to sharing representations among multiple envelopes (in
    particular, assign_rep is passed a letter object and operator= is
    passed an envelope object).  Letter assignment supports two models as 
    governed by ref_count_incr:

    \li ref_count_incr = true (default): the incoming letter belongs to 
    another envelope.  In this case, increment the reference count in the 
    normal manner so that deallocation of the letter is handled properly.

    \li ref_count_incr = false: the incoming letter is instantiated on the
    fly and has no envelope.  This case is modeled after get_iterator():
    a letter is dynamically allocated using new and passed into assign_rep,
    the letter's reference count is not incremented, and the letter is not
    remotely deleted (its memory management is passed over to the envelope). */
void Iterator::assign_rep(Iterator* iterator_rep, bool ref_count_incr)
{
  if (iteratorRep == iterator_rep) {
    // if ref_count_incr = true (rep from another envelope), do nothing as
    // referenceCount should already be correct (see also operator= logic).
    // if ref_count_incr = false (rep from on the fly), then this is an error.
    if (!ref_count_incr) {
      Cerr << "Error: duplicated iterator_rep pointer assignment without "
	   << "reference count increment in Iterator::assign_rep()." << endl;
      abort_handler(-1);
    }
  }
  else { // normal case: old != new
    // Decrement old
    if (iteratorRep) // Check for NULL
      if ( --iteratorRep->referenceCount == 0 ) 
	delete iteratorRep;
    // Assign new
    iteratorRep = iterator_rep;
    // Increment new
    if (iteratorRep && ref_count_incr) // Check for NULL & honor ref_count_incr
      iteratorRep->referenceCount++;
  }

#ifdef REFCOUNT_DEBUG
  Cout << "Iterator::assign_rep(Iterator*)" << endl;
  if (iteratorRep)
    Cout << "iteratorRep referenceCount = " << iteratorRep->referenceCount
	 << endl;
#endif
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This non-virtual function is one form of the overloaded run_iterator
    function which automates the pre-run/run/post-run portions of the
    progression.  This form accepts an ostream and executes verbosely. */
void Iterator::run_iterator(ostream& s)
{
  if (iteratorRep)
    iteratorRep->run_iterator(s); // envelope fwd to letter
  else {
    // Since iterator executions are mixed with direct model evaluations in
    // some methods/strategies (e.g., SBO and local/global reliability), avoid
    // having to reset the parallel configuration for the direct model evals
    // by eliminating configuration modifications within an iterator execution.
    //ParallelLibrary& parallel_lib = iteratedModel.parallel_library();
    //ParConfigLIter prev_pc = parallel_lib.parallel_configuration_iterator();

    pre_run(s);
    s << "\n>>>>> Running "  << methodName <<" iterator.\n";
    run();
    s << "\n<<<<< Iterator " << methodName <<" completed.\n";
    post_run(s);

    //parallel_lib.parallel_configuration_iterator(prev_pc); // reset
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This non-virtual function is one form of the overloaded run_iterator
    function which automates the pre-run/run/post-run portions of the
    progression.  This form does not accept an ostream and executes quietly. */
void Iterator::run_iterator()
{
  if (iteratorRep)
    iteratorRep->run_iterator(); // envelope fwd to letter
  else {
    // Since iterator executions are mixed with direct model evaluations in
    // some methods/strategies (e.g., SBO and local/global reliability), avoid
    // having to reset the parallel configuration for the direct model evals
    // by eliminating configuration modifications within an iterator execution.
    //ParallelLibrary& parallel_lib = iteratedModel.parallel_library();
    //ParConfigLIter prev_pc = parallel_lib.parallel_configuration_iterator();

    pre_run();
    run();
    post_run();

    //parallel_lib.parallel_configuration_iterator(prev_pc); // reset
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This function is one form of the overloaded pre-run function.  This form
    accepts an ostream and executes verbosely.  It is used for standard
    stand-alone iterator executions.  This function is not virtual: derived
    portions are defined in derived_pre_run(). */
void Iterator::pre_run(ostream& s)
{
  if (iteratorRep)
    iteratorRep->pre_run(s); // envelope fwd to letter
  else {

    // Verify that iteratedModel is not null (default ctor and some
    // NoDBBaseConstructor ctors leave iteratedModel uninitialized).
    if (!iteratedModel.is_null()) {
      // update context data that is outside scope of local DB specifications.
      // This is needed for reused objects.
      //iteratedModel.db_scope_reset(); // TO DO: need better name?
      // Set the reference points for the evaluation counters
      iteratedModel.set_evaluation_reference();
      // Set the active parallel configuration within the Model
      iteratedModel.set_communicators(maxConcurrency);
    }

    // Handle late availability of model's asynchEvalFlag in this way
    // (previously initialized in Iterator constructor, this info is
    // now unavailable until after model.set_communicators())
    asynchFlag = iteratedModel.asynch_flag();

    // Perform any derived class portions of pre_run
    derived_pre_run();
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This function is one form of the overloaded pre-run function.  This form
    does not accept an ostream and executes quietly.  It is commonly used in
    sub-iterator executions.  This function is not virtual: derived portions
    are defined in derived_pre_run(). */
void Iterator::pre_run()
{
  if (iteratorRep)
    iteratorRep->pre_run(); // envelope fwd to letter
  else {

    // Verify that iteratedModel is not null (default ctor and some
    // NoDBBaseConstructor ctors leave iteratedModel uninitialized).
    if (!iteratedModel.is_null()) {
      // update context data that is outside scope of local DB specifications.
      // This is needed for reused objects.
      //iteratedModel.db_scope_reset(); // TO DO: need better name?

      // Do not reset the evaluation reference (sub-iterator execution)

      // Set the active parallel configuration within the Model
      iteratedModel.set_communicators(maxConcurrency);
    }

    // Handle late availability of model's asynchEvalFlag in this way
    // (previously initialized in Iterator constructor, this info is
    // now unavailable until after model.set_communicators())
    asynchFlag = iteratedModel.asynch_flag();

    // Perform any derived class portions of pre_run
    derived_pre_run();
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct
    progression.  This function is the virtual derived class portion
    of pre_run().  Redefinition by derived classes is optional. */
void Iterator::derived_pre_run()
{
  if (iteratorRep)
    iteratorRep->derived_pre_run(); // envelope fwd to letter
  // else base class default behavior is no-op
}


/** Iterator supports a construct/pre-run/run/post-run/destruct
    progression.  This function is the virtual run function for the
    iterator class hierarchy.  All derived classes need to redefine it. */
void Iterator::run()
{
  if (iteratorRep)
    iteratorRep->run(); // envelope fwd to letter
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual run() function.\n"
         << "No default iteration defined at base class." << endl;
    abort_handler(-1);
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This function is one form of the overloaded post-run function.  This form
    accepts an ostream and executes verbosely.  This function is not virtual:
    derived portions are defined in derived_post_run(). */
void Iterator::post_run(ostream& s)
{
  if (iteratorRep)
    iteratorRep->post_run(s); // envelope fwd to letter
  else {
    // Perform any derived class portions of post_run
    derived_post_run();

    // Print the function evaluation summary for all Iterators
    if (!iteratedModel.is_null())
      iteratedModel.print_evaluation_summary(s); // full hdr, relative counts

    // The remaining final results output varies by iterator branch
    print_results(s);
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct progression.
    This function is one form of the overloaded post-run function.  This form
    does not accept an ostream and executes quietly.  This function is not
    virtual: derived portions are defined in derived_post_run(). */
void Iterator::post_run()
{
  if (iteratorRep)
    iteratorRep->post_run(); // envelope fwd to letter
  else {
    // Perform any derived class portions of post_run
    derived_post_run();
  }
}


/** Iterator supports a construct/pre-run/run/post-run/destruct
    progression.  This function is the virtual derived class portion
    of post_run().  Redefinition by derived classes is optional. */
void Iterator::derived_post_run()
{
  if (iteratorRep)
    iteratorRep->derived_post_run(); // envelope fwd to letter
  // else base class default behavior is no-op
}


const Variables& Iterator::variables_results() const
{
  if (!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Warning: letter class does not redefine variables_results() "
         << "virtual fn.\nDefault behavior returns current variables.\n";
    return iteratedModel.current_variables();
  }

  return iteratorRep->variables_results(); // envelope fwd to letter
}

const VariablesArray& Iterator::variables_array_results() const
{
  if (!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine variables_array_results() "
         << "virtual fn.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }

  return iteratorRep->variables_array_results(); // envelope fwd to letter
}


const Response& Iterator::response_results() const
{
  if(!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Warning: letter class does not redefine response_results() "
         << "virtual fn.\nDefault behavior returns current response.\n";
    return iteratedModel.current_response();
  }

  return iteratorRep->response_results(); // envelope fwd to letter
}


const ResponseArray& Iterator::response_array_results() const
{
  if(!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine response_array_results() "
	 << "virtual fn.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }

  return iteratorRep->response_array_results(); // envelope fwd to letter
}


bool Iterator::accepts_multiple_points() const
{
  if (iteratorRep) // envelope fwd to letter
    return iteratorRep->accepts_multiple_points();
  else // default for letter lacking virtual fn redefinition
    return false;
}


bool Iterator::returns_multiple_points() const
{
  if (iteratorRep) // envelope fwd to letter
    return iteratorRep->returns_multiple_points();
  else // default for letter lacking virtual fn redefinition
    return false;
}


void Iterator::initial_points(const VariablesArray& pts)
{
  if(iteratorRep) // envelope fwd to letter
    iteratorRep->initial_points(pts);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine initial_points virtual fn.\n"
	 << "No default defined at base class." << endl;
    abort_handler(-1);
  }
}


const VariablesArray& Iterator::initial_points() const
{
  if(!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine initial_points "
            "virtual fn.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }

  return iteratorRep->initial_points(); // envelope fwd to letter
}


void Iterator::response_results_active_set(const ActiveSet& set)
{
  if (iteratorRep)
    iteratorRep->response_results_active_set(set);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: Letter lacking redefinition of virtual response_results_"
         << "active_set() function.\nNo default defined at base class." << endl;
    abort_handler(-1);
  }
}


/** This is a convenience function for encapsulating graphics
    initialization operations.  It does not require a strategyRep
    forward since it is only used by letter objects. */
void Iterator::
initialize_graphics(bool graph_2d, bool tabular_data,
		    const String& tabular_file)
{
  if (iteratorRep)
    iteratorRep->initialize_graphics(graph_2d, tabular_data, tabular_file);
  else { // no redefinition of virtual fn., use default initialization
    extern Graphics dakota_graphics; // defined in ParallelLibrary.C
    if (graph_2d)     // initialize the 2D plots
      dakota_graphics.create_plots_2d(
	iteratedModel.current_variables(), iteratedModel.current_response());
    if (tabular_data) // initialize the tabular data file
      dakota_graphics.create_tabular_datastream(
	iteratedModel.current_variables(), iteratedModel.current_response(),
	tabular_file);
    if (graph_2d || tabular_data) // turn out automatic graphics logging
      iteratedModel.auto_graphics(true);
  }
}


/** This virtual function provides additional iterator-specific final results
    outputs beyond the function evaluation summary printed in post_run(). */
void Iterator::print_results(ostream& s)
{
  if (iteratorRep)
    iteratorRep->print_results(s); // envelope fwd to letter
  // else default base class output is nothing additional beyond the fn
  // evaluation summary printed in post_run()
}


void Iterator::
sampling_reset(int min_samples, int rec_samples, bool all_data_flag, 
	       bool stats_flag)
{
  if (iteratorRep) // envelope fwd to letter
    iteratorRep->sampling_reset(min_samples, rec_samples, all_data_flag,
				stats_flag);
  else { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine sampling_reset() virtual "
         << "fn.\nThis iterator does not support sampling." << endl;
    abort_handler(-1);
  }
}


const String& Iterator::sampling_scheme() const
{
  if (!iteratorRep) { // letter lacking redefinition of virtual fn.!
    Cerr << "Error: letter class does not redefine sampling_scheme() virtual "
         << "fn.\nThis iterator does not support sampling." << endl;
    abort_handler(-1);
  }

  return iteratorRep->sampling_scheme(); // envelope fwd to letter
}


String Iterator::uses_method() const
{
  if (iteratorRep) // envelope fwd to letter
    return iteratorRep->uses_method();
  else // default definition (letter lacking redefinition of virtual fn.)
    return String(); // null: no enabling iterator for this iterator
}


void Iterator::method_recourse()
{
  if (iteratorRep) // envelope fwd to letter
    iteratorRep->method_recourse();
  else { // default definition (letter lacking redefinition of virtual fn.)
    Cerr << "Error: no method recourse defined for detected method conflict.\n"
	 << "       Please revise method selections." << endl;
    abort_handler(-1);
  }
}


const VariablesArray& Iterator::all_variables() const
{
  if (!iteratorRep) { // letter lacking redefinition of virtual fn.
    Cerr << "Error: letter class does not redefine all_variables() virtual fn."
         << "\n       This iterator does not support variables histories."
	 << endl;
    abort_handler(-1);
  }

  return iteratorRep->all_variables(); // envelope fwd to letter
}


const ResponseArray& Iterator::all_responses() const
{
  if (!iteratorRep) { // letter lacking redefinition of virtual fn.
    Cerr << "Error: letter class does not redefine all_responses() virtual fn."
         << "\n       This iterator does not support response histories."<<endl;
    abort_handler(-1);
  }

  return iteratorRep->all_responses(); // envelope fwd to letter
}

} // namespace Dakota
