#!/bin/csh -f

# This is the base test script for Dakota.  The script is written to run as
# a csh script because that preserves csh environment variables needed for 
# making dakota
#
# The script can be started at the command line or by using crontab.

# --- external variables used ---
# $1 root directory for builds ($DAKOTA_ROOT)
# $2 (optional) tarball tag (default votd)
# $TEST_DIR      (environment.csh)
# $SNLSYSTEM     (environment.csh)
# $TEST_PARALLEL (environment.csh)
# -------------------------------

# check command syntax
if ( $#argv < 1 ) then
  echo "Usage: $0 DAKOTA_ROOT [tar_tag]"
  echo "*** Assumes presence of tarball in DAKOTA_ROOT/nightly/tarballs"
  echo "    (and version is as given by tar_tag (default votd))"
  exit
endif

echo "Running $0 on `uname -n` at `date`"

# set up the shell variables (local) and environment variables (global)
setenv DAKOTA_ROOT $1
setenv TEST_SQA  $DAKOTA_ROOT/Dakota/test/sqa
source $TEST_SQA/environment.csh
# fully-qualified TEST_DIR
setenv TEST_DIR_FQ $DAKOTA_ROOT/$TEST_DIR

# to find Kerberos tickets in the Mac cron environment on dakmacppc
if ( `uname` == 'Darwin'  && $SNLSYSTEM == 'csri' ) then
  setenv KRB5CCNAME "FILE:/Users/dakota/krb5cache"
endif

# optional argument
if ( $#argv > 1 ) then
  set tar_tag = $2
else
  set tar_tag = "votd"
endif

# static parameters
set distro = 15       # extract all four binaries
set distro_tests = 8  # for now, only run tests on internal

# setup directories and record build host
cd $DAKOTA_ROOT
if ( ! -e $TEST_DIR ) then
  mkdir -p $TEST_DIR
endif
if ( ! -e $TEST_DIR/bin_tars ) then
  mkdir $TEST_DIR/bin_tars
endif

# cleanup Dakota directory and extract nightly tarball
cd $TEST_DIR
uname -n > buildhost
if ( -e Dakota ) then
  echo "Removing directory $TEST_DIR_FQ/Dakota"
  \rm -rf Dakota
  echo "Removing previous tarball"
  rm -f Dakota_${tar_tag}_int_full.src.tar.gz 
endif

# DAKOTA tarballs are currently extracted nightly onto SCICO in
#    /usr/sharelan/dakota/nightly/tarballs
setenv DAKOTA_TARS /usr/sharelan/dakota/nightly/tarballs
if ( $SNLSYSTEM == 'scico' ) then
  if ( -e $DAKOTA_TARS/Dakota_${tar_tag}_int_full.src.tar.gz ) then
    echo "${0}: Extracting tarball"
    gunzip -c $DAKOTA_TARS/Dakota_${tar_tag}_int_full.src.tar.gz | tar xf -
  else
    echo "File $DAKOTA_TARS/Dakota_${tar_tag}_int_full.src.tar.gz"
    echo "does not exist; exiting."
    exit
  endif
else
  # on clusters, scp the file (could scp in both cases...not yet)
  # on clusters, there is no scp2, so use scp; should probably be a variable
  scp -Bp sass9000:$DAKOTA_TARS/Dakota_${tar_tag}_int_full.src.tar.gz .
  if ( -e Dakota_${tar_tag}_int_full.src.tar.gz ) then
    echo "${0}: Extracting tarball"
    gunzip -c Dakota_${tar_tag}_int_full.src.tar.gz | tar xf -
  else
    echo "File Dakota_${tar_tag}_int_full.src.tar.gz does not exist; exiting."
    exit
  endif
endif
setenv DAKOTA $TEST_DIR_FQ/Dakota


# -------------------------
# full internal for testing
# -------------------------
if ( $distro & 8 ) then

  set distro_name = int_full
  set prefix = $TEST_DIR_FQ/bin_tars/$distro_name
  echo "Building for $distro_name with target $prefix"
  # cleanup configure and make out and error files; diff out files
  rm -f $TEST_DIR_FQ/${distro_name}_*_dakota.*
  rm -f $TEST_DIR_FQ/${distro_name}_dakota_*.out
  if ( -e $prefix ) then
    \rm -rf $prefix
  endif
  mkdir -p $prefix


  # for full internal, we configure around failed packages and proceed
  # regardless of failures
  cd $DAKOTA
  set make_script_opt = "--with-plugin --with-nlpql --with-psuade --prefix=$prefix"
  # for now build docs on Linux for full internal only
  if ( $TEST_PLATFORM == "linux64"  && $SNLSYSTEM != "redstorm" ) then
    set make_script_opt = "$make_script_opt --enable-docs"
  endif
  setenv DAK_BUILD_ERROR 0
  $TEST_SQA/extract_bin.csh $DAKOTA $tar_tag $distro_name $make_script_opt

  # perform first phase of email generation to create an interim report 
  # on configure/build status.  Once the testing step completes, this
  # will later be overwritten by test results below, but will provide a 
  # build report in case the testing gets hung or isn't finished by the
  # time pass 2 of send_base_mail is done.
  $TEST_SQA/send_base_mail.csh $DAKOTA_ROOT development.sandia.gov 1

  # exit if there were make errors or there is no executable
  if ( ! -e make_dakota.err && -x src/dakota ) then
    if ( $distro_tests & 8 ) then
      cd $prefix/test
      # run serial Dakota test suite on non-cross-compiler environments
      if ( $USE_QSUB == 'yes' ) then
        echo "Submit serial tests to batch queue on `uname -n` at `date`"
        qsub -V $TEST_SQA/dakota_serial_test.qsub > jobid
          # Wait until the batch job finishes before continuing.
          # We won't wait forever, since the batch job has a time limit.
          # Maybe we can think of prettier logic to accomplish this.
          set myjob  = `tail -1 jobid`
          while ( `qstat | grep $myjob | wc -l` != 0)
            sleep 300
          end
          rm -f jobid

      else if ( $USE_SLURM == 'yes' ) then
          echo "Submit serial tests to batch queue on `uname -n` at `date`"
          sbatch -D `pwd` $TEST_SQA/dakota_serial_test.submission >&! jobid
          # Wait until the batch job finishes before continuing.
          # We won't wait forever, since the batch job has a time limit.
          # Maybe we can think of prettier logic to accomplish this.
          set myjob  = `cut -c 29- jobid`
          while ( `squeue | grep $myjob | wc -l` != 0)
            sleep 300
          end
          rm -f jobid

      else
        $TEST_SQA/dakota_test.perl
      endif

      # run parallel Dakota test suite (Note: Linux requires "kinit -f")
      if ( $TEST_PARALLEL == 'yes' ) then
        if ( $USE_QSUB == 'yes' ) then
          echo "Submit parallel tests to batch queue on `uname -n` at `date`"
          qsub -V $TEST_SQA/dakota_parallel_test.qsub > jobid
          # Wait until the batch job finishes before continuing.
          # We won't wait forever, since the batch job has a time limit.
          # Maybe we can think of prettier logic to accomplish this.
          set myjob  = `tail -1 jobid`
          while ( `qstat | grep $myjob | wc -l` != 0)
            sleep 300
          end
          rm -f jobid
        else if ( $USE_SLURM == 'yes' ) then
          echo "Submit parallel tests to batch queue on `uname -n` at `date`"
          echo "DEBUGGING:  Don't run parallel tests this time."
        else
          $TEST_SQA/dakota_test.perl parallel
        endif
      endif
      cp -fp dakota_diffs.out $TEST_DIR_FQ/${distro_name}_dakota_diffs.out
      cp -fp dakota_pdiffs.out $TEST_DIR_FQ/${distro_name}_dakota_pdiffs.out
    endif
  else
    echo "DAKOTA build failed, not running tests."
  endif

  # perform first phase of email generation (2nd to be done in separate cron)
  $TEST_SQA/send_base_mail.csh $DAKOTA_ROOT development.sandia.gov 1

  # don't build the remainder if there were any problems with full make,
  # including any failed packages
  cd $DAKOTA
  if ( -e reconfig_dakota.err || -e config_dakota.err || \
       -e make_dakota.err                                  ) then
    echo "Error in DAKOTA int_full build; not performing other builds."
    exit
  endif

endif


# -------------------
# internal supervised
# -------------------
if ( $distro & 4 ) then

  set distro_name = int_supervised
  set prefix = $TEST_DIR_FQ/bin_tars/$distro_name
  echo "Building for $distro_name with target $prefix"
  # cleanup configure and make out and error files; diff out files
  rm -f $TEST_DIR_FQ/${distro_name}_*_dakota.*
  rm -f $TEST_DIR_FQ/${distro_name}_dakota_*.out
  if ( -e $prefix ) then
    \rm -rf $prefix
  endif
  mkdir -p $prefix

  cd $DAKOTA
  set make_script_opt = "NOPKG_RECONFIG --without-psuade --without-nlpql --prefix=$prefix"
  # if previous configure, only make clean in src
  if ( -e config.log ) then
    set make_script_opt = "DAKOTA_RECONFIG $make_script_opt"
  endif
  $TEST_SQA/extract_bin.csh $DAKOTA $tar_tag $distro_name $make_script_opt

endif

# ------------
# internal web
# ------------
if ( $distro & 2 ) then

  set distro_name = int_web
  set prefix = $TEST_DIR_FQ/bin_tars/$distro_name
  echo "Building for $distro_name with target $prefix"
  # cleanup configure and make out and error files; diff out files
  rm -f $TEST_DIR_FQ/${distro_name}_*_dakota.*
  rm -f $TEST_DIR_FQ/${distro_name}_dakota_*.out
  if ( -e $prefix ) then
    \rm -rf $prefix
  endif
  mkdir -p $prefix

  cd $DAKOTA
  set make_script_opt = "NOPKG_RECONFIG --without-psuade --without-nlpql --without-dot --prefix=$prefix"
  # if previous configure, only make clean in src
  if ( -e config.log ) then
    set make_script_opt = "DAKOTA_RECONFIG $make_script_opt"
  endif
  $TEST_SQA/extract_bin.csh $DAKOTA $tar_tag $distro_name $make_script_opt

endif

# ------------
# external web
# ------------
if ( $distro & 1 ) then

  set distro_name = ext_web
  set prefix = $TEST_DIR_FQ/bin_tars/$distro_name
  echo "Building for $distro_name with target $prefix"
  # cleanup configure and make out and error files; diff out files
  rm -f $TEST_DIR_FQ/${distro_name}_*_dakota.*
  rm -f $TEST_DIR_FQ/${distro_name}_dakota_*.out
  if ( -e $prefix ) then
    \rm -rf $prefix
  endif
  mkdir -p $prefix

  cd $DAKOTA
  set make_script_opt = "NOPKG_RECONFIG --without-psuade --without-nlpql --without-dot --without-npsol --prefix=$prefix"
  # if previous configure, only make clean in src
  if ( -e config.log ) then
    set make_script_opt = "DAKOTA_RECONFIG $make_script_opt"
  endif
  $TEST_SQA/extract_bin.csh $DAKOTA $tar_tag $distro_name $make_script_opt

endif

# distribute to external web (1), internal web(2), and SCICO internal (4)
$TEST_SQA/create_votd.csh 7 votd >&! $TEST_DIR_FQ/create_votd.log
