/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA) Managed Front End

    CONTENTS:

        Definition of class MParameterDatabase.

    NOTES:

        See notes under Class Definition section of this file.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Wed Feb 08 16:29:50 2006 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the definition of the MParameterDatabase class.
 */




/*
================================================================================
Prevent Multiple Inclusions
================================================================================
*/
#ifndef JEGA_FRONTEND_MANAGED_MPARAMETERDATABASE_HPP
#define JEGA_FRONTEND_MANAGED_MPARAMETERDATABASE_HPP

#pragma once


/*
================================================================================
Includes
================================================================================
*/
#include <MConfig.hpp>







/*
================================================================================
Pre-Namespace Forward Declares
================================================================================
*/
namespace JEGA
{
    namespace Utilities
    {
        class ParameterDatabase;
    }
}






/*
================================================================================
Namespace Using Directives - DISCOURAGED!!
================================================================================
*/
#pragma managed
#using <mscorlib.dll>








/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace FrontEnd {
        namespace Managed {





/*
================================================================================
In-Namespace Forward Declares
================================================================================
*/







/*
================================================================================
In-Namespace File Scope Typedefs
================================================================================
*/








/*
================================================================================
Class Definition
================================================================================
*/
/**
 * \brief A managed version of the ParameterDatabase interface for use in the
 *        managed front end.
 *
 * This interface only requires the "Add" methods but also requires that a
 * manifestation to a JEGA::Utilities::ParameterDatabase be possible for
 * extraction of parameters from within JEGA core.
 */
MANAGED_INTERFACE(public, MParameterDatabase) :
    public System::IDisposable
{
    /*
    ===========================================================================
    Public Methods
    ===========================================================================
    */
    public:

        /**
         * \brief Override to manifest this managed parameter database as a
         *        JEGA::Utilities::ParameterDatabase object.
         *
         * This is necessary so that the data pushed into this keep can be
         * accessed by JEGA core. This is typically simple because typically,
         * managed databases will internally use
         * JEGA::Utilities::ParameterDatabase objects.
         *
         * \return The JEGA::Utilities::ParameterDatabase representation of
         *         this managed database.
         */
        virtual
        JEGA::Utilities::ParameterDatabase&
        Manifest(
            ) = 0;

        /// Maps the supplied integer value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other integral value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddIntegralParam(
            System::String MOH tag,
            int value
            ) = 0;

        /// Maps the supplied double value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other double value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddDoubleParam(
            System::String MOH tag,
            double value
            ) = 0;

        /// Maps the supplied size_t value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other size_t value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddSizeTypeParam(
            System::String MOH tag,
            std::size_t value
            ) = 0;

        /// Maps the supplied boolean value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other boolean value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddBooleanParam(
            System::String MOH tag,
            bool value
            ) = 0;

        /// Maps the supplied string value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other string value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddStringParam(
            System::String MOH tag,
            System::String MOH value
            ) = 0;

        /// Maps the supplied vector of doubles value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other vector of doubles value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddDoubleVectorParam(
            System::String MOH tag,
            DoubleVector MOH value
            ) = 0;

        /// Maps the supplied vector of ints value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other vector of ints value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddIntVectorParam(
            System::String MOH tag,
            IntVector MOH value
            ) = 0;

        /// Maps the supplied matrix of doubles value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other matrix of doubles value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddDoubleMatrixParam(
            System::String MOH tag,
            DoubleMatrix MOH value
            ) = 0;

        /// Maps the supplied vector of strings value to the supplied tag.
        /**
         * The mapping will only succeed if \a tag is not already the key for
         * some other vector of strings value.
         *
         * \param tag The key to which the requested value is to be mapped.
         * \param value The value that is to be mapped by key.
         * \return True of the mapping succeeds and false otherwise.
         */
        virtual
        bool
        AddStringVectorParam(
            System::String MOH tag,
            StringVector MOH value
            ) = 0;

}; // class MParameterDatabase



/*
================================================================================
End Namespace
================================================================================
*/
        } // namespace Managed
    } // namespace FrontEnd
} // namespace JEGA








/*
================================================================================
Include Inlined Functions File
================================================================================
*/
// Not using an Inlined Functions File.



/*
================================================================================
End of Multiple Inclusion Check
================================================================================
*/
#endif // JEGA_FRONTEND_MANAGED_MPARAMETERDATABASE_HPP
