/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class ConstraintInfo.

    NOTES:

        See notes of ConstraintInfo.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Mon Jun 09 16:16:24 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the ConstraintInfo class.
 */



/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <utilities/include/Math.hpp>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/Logging.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/ConstraintInfo.hpp>
#include <../Utilities/include/ConstraintTypeBase.hpp>
#include <../Utilities/include/ConstraintNatureBase.hpp>
#include <../Utilities/include/InequalityConstraintType.hpp>




/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;
using namespace JEGA::Logging;
using namespace eddy::utilities;






/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {








/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/
void
ConstraintInfo::SetType(
   ConstraintTypeBase* type
   )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(type != 0x0);

    delete _type;
    _type = (type == 0x0) ? new InequalityConstraintType(*this) : type;

} // ConstraintInfo::SetType





/*
================================================================================
Accessors
================================================================================
*/







/*
================================================================================
Public Methods
================================================================================
*/

double
ConstraintInfo::Which(
   const Design& des
   ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return des.GetConstraint(GetNumber());

} // ConstraintInfo::Which(Design)

double
ConstraintInfo::GetViolationAmount(
    const Design& of
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetViolationAmount(Which(of));

} // ConstraintInfo::GetViolationAmount

double
ConstraintInfo::GetViolationPercentage(
    const Design& of
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetViolationPercentage(Which(of));

} // ConstraintInfo::GetViolationPercentage

double
ConstraintInfo::GetViolationPercentage(
   double of
   ) const
{
    EDDY_FUNC_DEBUGSCOPE

    // This will be computed using the violation of "of" and the average
    // violation value.
	double viol = Math::Abs(GetViolationAmount(of));

    // A simple case is if viol is 0.0 in which case the return is an obvious
    // 0% violation.
    if(viol == 0.0) return 0.0;

    // Now the only exception is if there are no violations yet recorded.
    // If that is the case, we will return 100%.
    if(_numViolations == 0) return 100.0;

    // now get the average violation amount thus far recorded.
    double avg = GetAverageViolation();

    EDDY_ASSERT(avg != 0.0);

    // we will now compute a percentage based on viol and avg.
    return viol/avg * 100.0;

} // ConstraintInfo::GetViolationPercentage

double
ConstraintInfo::GetViolationAmount(
    double of
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetViolationAmount(of);

} // ConstraintInfo::GetViolationAmount

const Design*
ConstraintInfo::GetPreferredDesign(
    const Design& des1,
    const Design& des2
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    double val1 = GetViolationAmount(des1);
    double val2 = GetViolationAmount(des2);

    return (val1 < val2 ) ? &des1 :
           (val1 > val2 ) ? &des2 :
           0;

} // ConstraintInfo::GetPreferredDesign

double
ConstraintInfo::GetPreferredAmount(
    const Design& des1,
    const Design& des2
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetPreferredAmount(des1, des2);

} // ConstraintInfo::GetPreferredAmount

string
ConstraintInfo::GetTypeString() const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().ToString();
}

string
ConstraintInfo::GetNatureString() const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetNature().ToString();
}

bool
ConstraintInfo::SetNature(
    ConstraintNatureBase* nature
    )
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().SetNature(nature);

} // ConstraintInfo::SetNature

ConstraintNatureBase&
ConstraintInfo::GetNature() const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().GetNature();
}

void
ConstraintInfo::RecordViolation(
    const Design& of
    )
{
    EDDY_FUNC_DEBUGSCOPE
    RecordViolation(Which(of));

} // ConstraintInfo::RecordViolation

void
ConstraintInfo::RecordViolation(
    double rawValue
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // start by getting the magnitude of the violation of the raw value.
    double viol = Math::Abs(this->GetViolationAmount(rawValue));

    // now, if that violation is > 0, use it.  Otherwise, ignore it.
    if(viol > 0.0)
    {
        ++_numViolations;
        _totalViolation += viol;

		JEGALOG_II_G(ldebug(), this,
			ostream_entry(
				ldebug(), this->GetLabel() + ": recorded violation value of "
				) << viol << " average violation is now "
				  << this->GetAverageViolation()
			)

    }

} // ConstraintInfo::RecordViolation

double
ConstraintInfo::GetAverageViolation(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return (_numViolations == 0) ? 0.0 : (_totalViolation / _numViolations);

} // ConstraintInfo::GetAverageViolation

bool
ConstraintInfo::EvaluateConstraint(
    Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return GetType().EvaluateConstraint(des);
}



/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/






/*
================================================================================
Private Methods
================================================================================
*/








/*
================================================================================
Structors
================================================================================
*/

ConstraintInfo::ConstraintInfo(
    DesignTarget& target
    ) :
        InfoBase(target),
        _type(0x0),
        _totalViolation(0.0),
        _numViolations(0)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = new InequalityConstraintType(*this);

} // ConstraintInfo::ConstraintInfo

ConstraintInfo::ConstraintInfo(
    const ConstraintInfo& copy,
    DesignTarget& target
    ) :
        InfoBase(copy, target),
        _type(0x0),
        _totalViolation(copy._totalViolation),
        _numViolations(copy._numViolations)
{
    EDDY_FUNC_DEBUGSCOPE
    _type = copy.GetType().Clone(*this);

} // ConstraintInfo::ConstraintInfo

ConstraintInfo::ConstraintInfo(
    const ConstraintInfo& copy
    ) :
        InfoBase(copy),
        _type(0x0),
        _totalViolation(copy._totalViolation),
        _numViolations(copy._numViolations)
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(copy._type != 0x0);

    _type = copy.GetType().Clone(*this);

} // ConstraintInfo::ConstraintInfo

ConstraintInfo::~ConstraintInfo(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    delete _type;
}








/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
