/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:	 NonDEvidence
//- Description: Class for the Evidence theory methods within DAKOTA/UQ
//- Owner:	 Laura Swiler
//- Checked by:
//- Version:

#ifndef NOND_EVIDENCE_H
#define NOND_EVIDENCE_H

#include "data_types.h"
#include "DakotaNonD.H"
#include "DakotaStrategy.H"
#include "DakotaModel.H"
#include "DakotaIterator.H"
#include "NonDLHSSampling.H"

namespace Dakota {

/// Class for the Dempster-Shafer Evidence Theory methods within DAKOTA/UQ

/** The NonDEvidence class implements the propagation of epistemic uncertainty
    using Dempster-Shafer theory of evidence. In this approach, one assigns 
    a set of basic probability assignments (BPA) to intervals defined 
    for the uncertain variables.  Input interval combinations are calculated,
    along with their BPA.  Currently, the response function is evaluated at 
    a set of sample points, then a response surface is constructed which is 
    sampled extensively to find the minimum and maximum within each input 
    interval cell, corresponding to the belief and plausibility within that 
    cell, respectively.  This data is then aggregated to calculate cumulative 
    distribution functions for belief and plausibility. */

class NonDEvidence: public NonD
{
public:

  //
  //- Heading: Constructors and destructor
  //

  NonDEvidence(Model& model); ///< constructor
  ~NonDEvidence();            ///< destructor

  //
  //- Heading: Member functions
  //

  /// performs an epistemic uncertainty propagation using  
  /// Dempster-Shafer evidence theory methods which solve 
  /// for cumulative distribution functions of belief and plausibility
  void quantify_uncertainty(); // pure virtual, called by run_iterator

  /// print the cumulative distribution functions for belief and plausibility
  void print_results(ostream& s);

protected: 

  /// initialize finalStatistics for belief/plausibility results sets
  void initialize_final_statistics();

  /// method for computing belief and plausibility values for response levels 
  /// or vice-versa
  void compute_statistics();

private:

  //
  //- Heading: Convenience functions
  //

  /// convenience function for encapsulating the calculation of 
  /// basic probability assignments for input interval combinations
  void calculate_basic_prob_intervals();


  /// convenience function for encapsulating the determination of 
  /// maximum and minimum values within each input interval combination (cell).
  void calculate_maxmin_per_interval(const size_t& func_num);

  /// convenience function for determining the cumulative distribution 
  /// functions of belief and plausbility, based on the max and mins 
  /// per interval cell
  void calculate_cum_belief_plaus(const size_t& func_num);
 
  //
  //- Heading: Data
  //

  Iterator lhsSampler; ///< the LHS sampler instance

  const int originalSeed; ///< the user seed specification (default is 0)
  int numSamples; ///< the number of samples used in the surrogate

  //NonDLHSSampling init_samples;
  // These are variables and arrays that must be declared here prior
  // to calling the F77 DDS code.
  //
  /// Size variable for DDS arrays.
  /** NV = number of interval variables */
  int NV;
  /// Size variable for DDS arrays
  /** NCMB = number of cell combinations */
  int NCMB;
  /// Size variable for DDS arrays
  /** MAXINTVLS = maximum number of intervals per individual interval var */
  int MAXINTVLS;
  /// Temporary output variable
  /** Y = current output to be placed in cell*/
  Real Y;
 
  ///Internal DDS array.
  /** Basic Probability Assignments*/
  Real *BPA;
  ///Internal DDS array.
  /** Minimum ends of intervals.*/
  Real *VMIN;
  ///Internal DDS array.
  /** Maximum ends of intervals.*/
  Real *VMAX;
  ///Internal DDS array.
  /** Basic Probability Combinations.*/
  Real *BPAC;
  ///Internal DDS Array.
  /** Minimum per cell combination.*/
  Real *CMIN;
  ///Internal DDS Array.
  /** Maximum per cell combination.*/
  Real *CMAX;
  ///Internal DDS Array.
  /** X per cell combination.*/
  Real *X;
  ///Internal DDS array.
  /** Number of intervals per interval variable*/
  int *NI;
  ///Internal DDS array.
  /** Sort order for combinations*/
  int *IP;
  ///Internal DDS array.
  /** Sort order for belief values*/
  int *IPBEL;
  ///Internal DDS array.
  /** Sort order for belief values*/
  int *IPPLA;

  /// Storage array to hold CCBF values
  RealVectorArray cc_bel_fn;
  /// Storage array to hold CCPF values
  RealVectorArray cc_plaus_fn;
  /// Storage array to hold CCB response values
  RealVectorArray cc_bel_val;
  /// Storage array to hold CCP response values
  RealVectorArray cc_plaus_val;

  /// Storage array to hold variables
  VariablesArray all_vars;
  /// Storage array to hold responses
  ResponseArray all_responses;

};

} // namespace Dakota

#endif
