/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- class:       Dakota::List<T>
//- Description: Inherits from either the RW list class or the STL list class 
//- 		 Extends the base list class to add Dakota specific methods.
//-
//- Assumptions: 
//-              
//- Owner:       Mario Alleva
//- Checked by:  Mike Eldred
//- Version: $Id

#ifndef DAKOTA_LIST_H
#define DAKOTA_LIST_H

#include "template_defs.h"
#include "global_defs.h"
#include "MPIPackBuffer.H"


namespace Dakota {

/// Template class for the Dakota bookkeeping list

/** The List is the common list class for Dakota. It inherits 
    from either the RW list class or the STL list class.  
    Extends the base list class to add Dakota specific methods
    Builds upon the previously existing DakotaValList class  */

template<class T>
class List: public DAKOTA_BASE_LIST<T>
{
public:

  //
  //- Heading: Constructors, destructor, assignment operator
  //

  List();                 ///< Default constructor
  List(const List<T>& a); ///< Copy constructor
  ~List();                ///< Destructor

#ifdef HAVE_MEMBER_TEMPLATES
  /// Range constructor (member template)
  template <class InputIter>
  List(InputIter first, InputIter last);
#endif // HAVE_MEMBER_TEMPLATES

  /// assignment operator
  List<T>& operator=(const List<T>& a);

  //
  //- Heading: Member functions 
  //

  /// Writes a List to an output stream
  void write(ostream& s) const;

  /// Reads a List from an MPIUnpackBuffer after an MPI receive
  void read(MPIUnpackBuffer& s);

  /// Writes a List to a MPIPackBuffer prior to an MPI send
  void write(MPIPackBuffer& s) const;

  // Functions needed by DAKOTA and not supported by STL lists:

  /// Returns the number of items that are currently in the list
  size_t entries() const;

  /// Removes and returns the first item in the list
  T get();

  /// Removes and returns the item at the specified index
  T removeAt(size_t index);

  /// Removes the specified item from the list
  bool remove(const T& a);

  /// Adds the item a to the end of the list
  void insert(const T& a);
    
  /// Returns TRUE if list contains object a, returns FALSE otherwise
  bool contains(const T& a) const;

  /// Returns TRUE if the list contains an object that the test function
  /// finds and sets k to this object
  bool find(bool (*test_fn)(const T&, const void*), const void* test_fn_data,
	    T& found_item) const;
  /// Returns an iterator pointing to an object that the test function finds
  typename List<T>::iterator find(bool (*test_fn)(const T&, const void*),
				  const void* test_fn_data);

  /// Returns the index of object that the test function finds
  size_t index(bool (*test_fn)(const T&, const void*),
	       const void* test_fn_data) const;
  /// Returns the index of the object
  size_t index(const T& a) const;

  /// Returns the number of items in the list that satisfy the test function
  size_t count(bool (*test_fn)(const T&, const void*),
	       const void* test_fn_data) const;
  /// Returns the number of items in the list equal to object a
  size_t count(const T& a) const;

  //
  //- Heading: operators 
  //

  /// Returns the object at index i (can use as lvalue)
  T& operator[](size_t i);
  /// Returns the object at index i, const (can't use as lvalue)
  const T& operator[](size_t i) const;
};


/** Internal functor to mimic the RW find and index functions
    using the STL find_if() method.  The class holds a pointer to
    the test function and the search value.  */
template <class T> 
class FunctionCompare 
{
private:
  /// Pointer to test function. 
  bool (*test_fn)(const T&, void*);
  /// Holds the value to search for.
  void* search_val;

public:
  /// Constructor that defines the pointer to function and search value. 
  FunctionCompare(bool (*func)(const T&, void*), void* v)
    { test_fn = func; search_val = v; }

  /// The operator() must be defined.  Calls the function test_fn 
  bool operator()(T t) const
    { return test_fn(t, search_val); }
};

template <class T> 
inline List<T>::List()
{}


template <class T> 
inline List<T>::List(const List<T>& a)
{ 
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_LIST<T>::operator=(a); // base class assignment operator
}


template <class T> 
inline List<T>::~List()
{}


#ifdef HAVE_MEMBER_TEMPLATES
template <class T> template <class InputIter>
List<T>::List(InputIter first, InputIter last):
  DAKOTA_BASE_LIST<T>(first, last)
{}
#endif // HAVE_MEMBER_TEMPLATES


template <class T> 
inline List<T>& List<T>::operator=(const List<T>& a)
{
  if (this != &a) // verify that these are different objects
    DAKOTA_BASE_LIST<T>::operator=(a); // base class assignment operator
  return *this;
}


template <class T>
void List<T>::write(ostream& s) const
{
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++)
    s << "                     " << *cit << '\n';
}


/// global ostream insertion operator for List
template <class T>
inline ostream& operator<<(ostream& s, const List<T>& data)
{
  data.write(s);
  return s;
}


template <class T>
void List<T>::read(MPIUnpackBuffer& s)
{
  DAKOTA_BASE_LIST<T>::clear();
  size_t len;
  s >> len;
  for (size_t i=0; i<len; i++){
    T data; // fresh allocation needed in case T is ref-counted
    s >> data; 
    DAKOTA_BASE_LIST<T>::push_back(data);
  }
}


template <class T>
void List<T>::write(MPIPackBuffer& s) const
{
  size_t len = DAKOTA_BASE_LIST<T>::size();
  s << len;
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++)
    s << *cit;
}


/// global MPIUnpackBuffer extraction operator for List
template <class T>
inline MPIUnpackBuffer& operator>>(MPIUnpackBuffer& s, List<T>& data)
{
  data.read(s);
  return s;
}


/// global MPIPackBuffer insertion operator for List
template <class T>
inline MPIPackBuffer& operator<<(MPIPackBuffer& s, const List<T>& data)
{
  data.write(s);
  return s;
}


template <class T> 
inline size_t List<T>::entries() const
{ return DAKOTA_BASE_LIST<T>::size(); }


/** Insert item at end of list, calls list::push_back() method. */
template <class T> 
inline void List<T>::insert(const T& a)
{ DAKOTA_BASE_LIST<T>::push_back(a); }


/** Remove and return item from front of list.  Returns the object
    pointed to by the list::begin() iterator. It also deletes the
    first node by calling the list::pop_front() method.  Note: get()
    is not the same as list::front() since the latter would return the
    1st item but would not delete it. */
template <class T> 
T List<T>::get()
{
  T tempT(*DAKOTA_BASE_LIST<T>::begin());
  DAKOTA_BASE_LIST<T>::pop_front();
  return tempT;
}


/** Removes the item at the index specified.  Uses the STL advance()
    function to step to the appropriate position in the list and then
    calls the list::erase() method. */
template <class T> 
T List<T>::removeAt(size_t index)
{
  if (index >= DAKOTA_BASE_LIST<T>::size()) {
    Cerr << "Error: index passed to List<T>::remoteAt() is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  typename List<T>::iterator it = DAKOTA_BASE_LIST<T>::begin();
  advance(it, index);
  T tempT(*it);
  DAKOTA_BASE_LIST<T>::erase(it);
  return tempT;
}


/** Removes the first instance matching object a from the list (and
    therefore differs from the STL list::remove() which removes all
    instances).  Uses the STL find() algorithm to find the object and
    the list::erase() method to perform the remove.*/
template <class T> 
bool List<T>::remove(const T& a)
{
  typename List<T>::iterator it
    = std::find(DAKOTA_BASE_LIST<T>::begin(), DAKOTA_BASE_LIST<T>::end(), a);
  if (it != DAKOTA_BASE_LIST<T>::end()) {
    DAKOTA_BASE_LIST<T>::erase(it);
    return true;
  }
  else
    return false;
}


/** Returns the index of the first item in the list which matches the
    object a.  Uses a single list traversal to both locate the object
    and return its index (generic algorithms would require two loop
    traversals). */
template <class T> 
size_t List<T>::index(const T& a) const
{
  // Use a single list traversal to be as efficient as possible
  size_t cntr = 0;
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++, cntr++)
    if (*cit == a)
      return cntr;
  return _NPOS;

  // While using find() and distance() as below is cleaner, it is less
  // efficient since it must perform 2 list traversals.
  //List<T>::const_iterator cit = std::find(begin(), end(), a);
  //return (cit != end()) ? distance(begin(), cit) : _NPOS;
  // Note: TFLOPS & Solaris WS 5.0 use an older version of the distance()
  // algorithm with the following usage:  distance(begin(), cit, dist);
}


/** Uses the STL count() algorithm to return the number of occurences
    of the specified object. */
template <class T> 
inline size_t List<T>::count(const T& a) const
{
#ifndef HAVE_STD_COUNT
  // These platforms use an older version of the count() algorithm
  size_t cnt = 0;
  std::count(DAKOTA_BASE_LIST<T>::begin(), DAKOTA_BASE_LIST<T>::end(), a, cnt);
  return cnt;
#else
  // All other platforms use the ANSI C++ version of the count() algorithm
  return std::count(DAKOTA_BASE_LIST<T>::begin(),
		    DAKOTA_BASE_LIST<T>::end(), a);
#endif
}


/** Uses the STL find() algorithm to locate the first instance of
    object a.  Returns true if an instance is found. */
template <class T>
inline bool List<T>::contains(const T& a) const
{
  return (std::find(DAKOTA_BASE_LIST<T>::begin(), DAKOTA_BASE_LIST<T>::end(), a)
	  != DAKOTA_BASE_LIST<T>::end()) ? true : false;
}


/** Returns item at position i of the list by stepping through the
    list using forward or reverse STL iterators (depending on which
    end of the list is closer to the desired item).  Once the object
    is found, it returns the value pointed to by the iterator.

    This functionality is inefficient in 0->len loop-based list
    traversals and is being replaced by iterator-based list traversals
    in the main DAKOTA code. For isolated look-ups of a particular
    index, however, this approach is acceptable. */
template <class T>
T& List<T>::operator[](size_t i)
{
#if defined(TFLOPS) || defined(SOLARIS)
  // size() does not require a list traversal for RW STL (Sun, TFLOP)
  size_t len = DAKOTA_BASE_LIST<T>::size(); // returns a class attribute
  if (i >= len) {
    Cerr << "Error: index passed to List<T>::operator[] is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  if (i < len - i) { // traverse list in fwd direction
    typename List<T>::iterator it = DAKOTA_BASE_LIST<T>::begin();
    advance(it, i);
    return *it; // return value at position i
  }
  else { // traverse list in reverse direction
    typename List<T>::reverse_iterator rit = DAKOTA_BASE_LIST<T>::rbegin();
    advance(rit, len-1-i);
    return *rit; // return value at position i
  }
#else
  // when size() requires a list traversal, the following is more efficient
  typename List<T>::iterator it = DAKOTA_BASE_LIST<T>::begin();
  //advance(cit, i); // not all implementations of advance stop at end()
  while (i-- && it != DAKOTA_BASE_LIST<T>::end())
    it++;
  if (it == DAKOTA_BASE_LIST<T>::end()) {
    Cerr << "Error: index passed to List<T>::operator[] is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  return *it; // return value at position i
#endif
}


/** Returns const item at position i of the list by stepping through
    the list using forward or reverse STL iterators (depending on
    which end of the list is closer to the desired item).  Once the
    object is found it returns the value pointed to by the iterator.

    This functionality is inefficient in 0->len loop-based list
    traversals and is being replaced by iterator-based list traversals
    in the main DAKOTA code. For isolated look-ups of a particular
    index, however, this approach is acceptable. */
template <class T>
const T& List<T>::operator[](size_t i) const
{
#if defined(TFLOPS) || defined(SOLARIS)
  // size() does not require a list traversal for RW STL (Sun, TFLOP)
  size_t len = DAKOTA_BASE_LIST<T>::size(); // returns a class attribute
  if (i >= len) {
    Cerr << "Error: index passed to List<T>::operator[] is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  if (i < len - i) { // traverse list in fwd direction
    typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
    advance(cit, i);
    return *cit; // return value at position i
  }
  else { // traverse list in reverse direction
    typename List<T>::const_reverse_iterator crit
      = DAKOTA_BASE_LIST<T>::rbegin();
    advance(crit, len-1-i);
    return *crit; // return value at position i
  }
#else
  // when size() requires a list traversal, the following is more efficient
  typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
  //advance(cit, i); // not all implementations of advance stop at end()
  while (i-- && cit != DAKOTA_BASE_LIST<T>::end())
    cit++;
  if (cit == DAKOTA_BASE_LIST<T>::end()) {
    Cerr << "Error: index passed to List<T>::operator[] is out of bounds."
	 << endl;
    abort_handler(-1);
  }
  return *cit; // return value at position i
#endif
}


/** Find the first item in the list which satisfies the test function.
    Sets k if the object is found. */
template <class T>
bool List<T>::find(bool (*test_fn)(const T&, const void*),
		   const void* test_fn_data, T& found_item) const
{
  // a hand-coded list traversal is faster than the Functor approach
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++) {
    if (test_fn(*cit, test_fn_data)) {
      found_item = *cit;
      return true;
    }
  }
  return false;

  // while only a single list traversal is needed with find_if() in this case,
  // it is still a fair amount slower, presumably due to Functor overhead.
  //List<T>::const_iterator cit
  //  = find_if(begin(), end(), FunctionCompare<T>(test_fn, test_fn_data));
  //if (cit != end()) { // value found: set to value at cit and return true
  //  found_item = *cit;
  //  return true;
  //}
  //else // value not found
  //  return false;
}


/** Find the first item in the list which satisfies the test function
    and return an iterator pointing to it. */
template <class T>
typename List<T>::iterator List<T>::
find(bool (*test_fn)(const T&, const void*), const void* test_fn_data)
{
  // a hand-coded list traversal is faster than the Functor approach
  for (typename List<T>::iterator it = DAKOTA_BASE_LIST<T>::begin();
       it != DAKOTA_BASE_LIST<T>::end(); it++)
    if (test_fn(*it, test_fn_data))
      return it;
  return DAKOTA_BASE_LIST<T>::end();

  // while only a single list traversal is needed with find_if() in this case,
  // it is still a fair amount slower, presumably due to Functor overhead.
  //return find_if(begin(), end(), FunctionCompare<T>(test_fn, test_fn_data));
}


/** Returns the index of the first item in the list which satisfies
    the test function.  Uses a single list traversal to both locate
    the object and return its index (generic algorithms would require
    two loop traversals). */
template <class T>
size_t List<T>::
index(bool (*test_fn)(const T&, const void*), const void* test_fn_data) const
{
  // Use a single list traversal to be as efficient as possible
  size_t cntr = 0;
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++, cntr++)
    if (test_fn(*cit, test_fn_data))
      return cntr;
  return _NPOS;

  // While using find_if() and distance() as below is cleaner, it is less
  // efficient since it must perform 2 list traversals.
  //List<T>::const_iterator cit
  //  = find_if(begin(), end(), FunctionCompare<T>(test_fn, test_fn_data));
  //return (cit != end()) ? distance(begin(), cit) : _NPOS;
  // Note: TFLOPS & Solaris WS 5.0 use an older version of the distance()
  // algorithm with the following usage:  distance(begin(), cit, dist);
}


template <class T>
size_t List<T>::
count(bool (*test_fn)(const T&, const void*), const void* test_fn_data) const
{
  size_t cntr = 0;
  for (typename List<T>::const_iterator cit = DAKOTA_BASE_LIST<T>::begin();
       cit != DAKOTA_BASE_LIST<T>::end(); cit++)
    if (test_fn(*cit, test_fn_data))
      cntr++;
  return cntr;
}

} // namespace Dakota

#endif //DAKOTA_LIST_H
