/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Implementation of class Design.

    NOTES:

        See notes of Design.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Thu May 15 11:04:05 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the implementation of the Design class.
 */




/*
================================================================================
Includes
================================================================================
*/
// JEGAConfig.hpp should be the first include in all JEGA files.
#include <../Utilities/include/JEGAConfig.hpp>

#include <algorithm>
#include <../Utilities/include/Design.hpp>
#include <../Utilities/include/DesignTarget.hpp>
#include <utilities/include/EDDY_DebugScope.hpp>
#include <../Utilities/include/DesignVariableInfo.hpp>



/*
================================================================================
Namespace Using Directives
================================================================================
*/
using namespace std;




/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {



/*
================================================================================
Static Member Data Definitions
================================================================================
*/








/*
================================================================================
Mutators
================================================================================
*/








/*
================================================================================
Accessors
================================================================================
*/








/*
================================================================================
Public Methods
================================================================================
*/
const Design&
Design::operator = (
    const Design& right
    )
{
    EDDY_FUNC_DEBUGSCOPE

    if(&right == this) return *this;

    // The targets must be the same for two Designs to equate
    EDDY_ASSERT(&_target == &right._target);

    // Allocation should already have taken place so
    // equate this to rhs.

    // equate the variables and responses.
    ::memcpy(_variables, right._variables, GetNDV() * sizeof(double));
    ::memcpy(_objectives, right._objectives, GetNOF() * sizeof(double));
    ::memcpy(_constraints, right._constraints, GetNCN() * sizeof(double));

    // equate the attributes.
    _attributes = right._attributes;

    // Tag the two Designs as clones.
    Design::TagAsClones(*this, const_cast<Design&>(right));

    return *this;
}

eddy::utilities::uint8_t
Design::TagAsClones(
    Design& des1,
    Design& des2
    )
{
    EDDY_FUNC_DEBUGSCOPE
    EDDY_ASSERT(&des1 != &des2);

    // Check for trivial abort conditions.
    if(&des1 == &des2) return 0;

    // These pointers will be used to iterate the clone lists.
    Design* pdes1 = &des1;
    Design* pdes2 = &des2;

    if(des1.IsCloned())
    {
        if(des2.IsCloned())
        {
            // both are clones already, check to see
            // if they are already known to be clones
            // of one another.  If not, tag'em.
            if(!des1.HasInCloneList(des2))
            {
                // find beginning of des1's list
                for(; pdes1->_pClone!=0x0; pdes1=pdes1->_pClone);
                // find end of des2's list
                for(; pdes2->_nClone!=0x0; pdes2=pdes2->_nClone);
                //  merge the lists.
                pdes2->_nClone = pdes1;
                pdes1->_pClone = pdes2;
            }

            return 0;
        }
        else // if(!des2.IsCloned())
        {
            // find the end of des1's clone list.
            for(; pdes1->_nClone!=0x0; pdes1=pdes1->_nClone);
            // put des2 at the end.
            pdes1->_nClone = pdes2;
            pdes2->_pClone = pdes1;
            return 1;
        }
    }

    if(des2.IsCloned())
    {
        // find the end of des2's list.
        for(; pdes2->_nClone!=0x0; pdes2=pdes2->_nClone);
        // put des1 at the end.
        pdes2->_nClone = pdes1;
        pdes1->_pClone = pdes2;
        return 1;
    }

    // neither of these guys were clones before, start a new list.
    des1._nClone = pdes2;
    des2._pClone = pdes1;
    return 2;
}

void
Design::RemoveAsClone(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    // typical linked list removal.
    if(_pClone != 0x0)
    {
        if(_nClone != 0x0) _nClone->_pClone = _pClone;
        _pClone->_nClone = _nClone;
    }
    else if(_nClone != 0x0) _nClone->_pClone = 0x0;

    _pClone = _nClone = 0x0;
}

bool
Design::HasInCloneList(
    const Design& des
    ) const
{
    EDDY_FUNC_DEBUGSCOPE

    for(const Design* cdes = _pClone; cdes!=0x0; cdes=cdes->_pClone)
        if(cdes == &des) return true;

    for(const Design* cdes = _nClone; cdes!=0x0; cdes=cdes->_nClone)
        if(cdes == &des) return true;

    return false;
}

eddy::utilities::uint64_t
Design::CountClones(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    eddy::utilities::uint64_t ret = 0;

    // count the nclones.
    for(const Design* clone=_nClone; clone!=0x0; clone=clone->_nClone) ++ret;

    // now the pclones
    for(const Design* clone=_pClone; clone!=0x0; clone=clone->_pClone) ++ret;

    return ret;
}

void
Design::SetVariableValue(
    size_t num,
    double val
    )
{
    EDDY_FUNC_DEBUGSCOPE
    DesignVariableInfoVector::size_type cnum =
        static_cast<DesignVariableInfoVector::size_type>(num);
    const DesignVariableInfoVector& dvinfos = _target.GetDesignVariableInfos();
    _variables[num] = dvinfos.at(cnum)->GetDoubleRepOf(val);
}

double
Design::GetVariableValue(
    size_t num
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    DesignVariableInfoVector::size_type cnum =
        static_cast<DesignVariableInfoVector::size_type>(num);
    const DesignVariableInfoVector& dvinfos = _target.GetDesignVariableInfos();
    return dvinfos.at(cnum)->GetValueOf(_variables[num]);
}


size_t
Design::GetNDV(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return _target.GetNDV();

} // Design::GetNDV

size_t
Design::GetNOF(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return _target.GetNOF();

} // Design::GetNOF

size_t
Design::GetNCN(
    ) const
{
    EDDY_FUNC_DEBUGSCOPE
    return _target.GetNCN();

} // Design::GetNCN






/*
================================================================================
Subclass Visible Methods
================================================================================
*/








/*
================================================================================
Subclass Overridable Methods
================================================================================
*/








/*
================================================================================
Private Methods
================================================================================
*/


void
Design::DoAllocation(
    )
{
    EDDY_FUNC_DEBUGSCOPE

    _variables = new double[_target.GetNDV()];
    _objectives = new double[_target.GetNOF()];
    _constraints = new double[_target.GetNCN()];
}





/*
================================================================================
Structors
================================================================================
*/
Design::Design(
    DesignTarget& target
    ) :
        _variables(0x0),
        _objectives(0x0),
        _constraints(0x0),
        _attributes(0),
        _target(target),
        _nClone(0x0),
        _pClone(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    DoAllocation();
}

Design::Design(
    const Design& copy
    ) :
        _variables(0x0),
        _objectives(0x0),
        _constraints(0x0),
        _attributes(0),
        _target(copy._target),
        _nClone(0x0),
        _pClone(0x0)
{
    EDDY_FUNC_DEBUGSCOPE
    DoAllocation();
    this->operator =(copy);
}

Design::~Design(
    )
{
    EDDY_FUNC_DEBUGSCOPE
    RemoveAsClone();
    delete [] _variables;
    delete [] _objectives;
    delete [] _constraints;
}







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
