/*
================================================================================
    PROJECT:

        John Eddy's Genetic Algorithms (JEGA)

    CONTENTS:

        Inline methods of class DesignGroup.

    NOTES:

        See notes of DesignGroup.hpp.

    PROGRAMMERS:

        John Eddy (jpeddy@sandia.gov) (JE)

    ORGANIZATION:

        Sandia National Laboratories

    COPYRIGHT:

        See the LICENSE file in the top level JEGA directory.

    VERSION:

        1.0.0

    CHANGES:

        Fri May 16 12:05:31 2003 - Original Version (JE)

================================================================================
*/




/*
================================================================================
Document This File
================================================================================
*/
/** \file
 * \brief Contains the inline methods of the DesignGroup class.
 */




/*
================================================================================
Includes
================================================================================
*/




/*
================================================================================
Namespace Using Directives
================================================================================
*/





/*
================================================================================
Begin Namespace
================================================================================
*/
namespace JEGA {
    namespace Utilities {







/*
================================================================================
Inline Mutators
================================================================================
*/


inline
void
DesignGroup::SetTitle(
    const std::string& title
    )
{
    _title.assign(title);
}








/*
================================================================================
Inline Accessors
================================================================================
*/
inline
DesignTarget&
DesignGroup::GetDesignTarget(
    ) const
{
    return _target;
}

inline
const DesignGroup::DVSortContainer&
DesignGroup::GetDVSortContainer(
    ) const
{
    return _dvSort;
}

inline
const DesignGroup::OFSortContainer&
DesignGroup::GetOFSortContainer(
    ) const
{
    return _ofSort;
}

inline
const std::string&
DesignGroup::GetTitle(
    ) const
{
    return _title;
}






/*
================================================================================
Inline Public Methods
================================================================================
*/
template <typename Cont>
void
DesignGroup::AbsorbDesigns(
   const Cont& other
   )
{
    AbsorbDesigns(other.begin(), other.end());
}

template <typename It>
void
DesignGroup::AbsorbDesigns(
   It start,
   const It& end
   )
{
    for(; start!=end; ++start) Insert(*start);
}

inline
bool
DesignGroup::IsOFSortCurrent(
    ) const
{
    return SizeOF() == SizeDV();
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::GetSize(
    ) const
{
    return _dvSort.size();
}

inline
DesignGroup::DVSortContainer::const_iterator
DesignGroup::BeginDV(
    ) const
{
    return _dvSort.begin();
}

inline
DesignGroup::DVSortContainer::iterator
DesignGroup::BeginDV(
    )
{
    return _dvSort.begin();
}

inline
DesignGroup::DVSortContainer::const_iterator
DesignGroup::EndDV(
    ) const
{
    return _dvSort.end();
}

inline
DesignGroup::DVSortContainer::iterator
DesignGroup::EndDV(
    )
{
    return _dvSort.end();
}

inline
DesignGroup::OFSortContainer::const_iterator
DesignGroup::BeginOF(
    ) const
{
    return _ofSort.begin();
}

inline
DesignGroup::OFSortContainer::iterator
DesignGroup::BeginOF(
    )
{
    return _ofSort.begin();
}

inline
DesignGroup::OFSortContainer::const_iterator
DesignGroup::EndOF(
    ) const
{
    return _ofSort.end();
}

inline
DesignGroup::OFSortContainer::iterator
DesignGroup::EndOF(
    )
{
    return _ofSort.end();
}

inline
DesignGroup::OFSortContainer::size_type
DesignGroup::SizeOF(
    ) const
{
    return _ofSort.size();
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::SizeDV(
    ) const
{
    return _dvSort.size();
}

inline
bool
DesignGroup::IsEmpty(
    ) const
{
    return _dvSort.empty() && _ofSort.empty();
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushCloneDesigns(
    )
{
    return FlushIfTrue(&Design::IsCloned);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushEvaluatedDesigns(
    )
{
    return FlushIfTrue(&Design::IsEvaluated);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushFeasibleDesigns(
    )
{
    return FlushIfTrue(&Design::IsFeasible);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushIllconditionedDesigns(
    )
{
    return FlushIfTrue(&Design::IsIllconditioned);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushNonCloneDesigns(
    )
{
    return FlushIfFalse(&Design::IsCloned);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushNonEvaluatedDesigns(
    )
{
    return FlushIfFalse(&Design::IsEvaluated);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushNonIllconditionedDesigns(
    )
{
    return FlushIfFalse(&Design::IsIllconditioned);
}

inline
DesignGroup::DVSortContainer::size_type
DesignGroup::FlushNonFeasibleDesigns(
    )
{
    return FlushIfFalse(&Design::IsFeasible);
}

inline
bool
DesignGroup::AllDesignsAreFeasible(
    ) const
{
    return CountFeasible() == GetSize();
}

inline
bool
DesignGroup::ContainsDesign(
   const Design& des
   ) const
{
    return _dvSort.find_exact(const_cast<Design*>(&des)) != _dvSort.end();
}

/*
================================================================================
Inline Subclass Visible Methods
================================================================================
*/


inline
void
DesignGroup::ClearOFContainer(
    )
{
    _ofSort.clear();
}







/*
================================================================================
Inline Private Methods
================================================================================
*/







/*
================================================================================
Inline Structors
================================================================================
*/
template<typename DesCont>
DesignGroup::DesignGroup(
    DesignTarget& target,
    const DesCont& designs
    ) :
        _target(target),
        _dvSort(),
        _ofSort()
{
    typename DesCont::const_iterator it(designs.begin());
    const typename DesCont::const_iterator e(designs.end());

    for(; it!=e; ++it) Insert(*it);
}




/*
================================================================================
File Scope Utility Class Inline Methods
================================================================================
*/







/*
================================================================================
End Namespace
================================================================================
*/
    } // namespace Utilities
} // namespace JEGA
