/*  _______________________________________________________________________

    DAKOTA: Design Analysis Kit for Optimization and Terascale Applications
    Copyright (c) 2006, Sandia National Laboratories.
    This software is distributed under the GNU General Public License.
    For more information, see the README file in the top Dakota directory.
    _______________________________________________________________________ */

//- Class:       FSUDesignCompExp
//- Description: Implementation code for the FSUDesignCompExp class
//- Owner:       Laura Swiler, Sandia National Laboratories

#include "system_defs.h"
#include "fsu.H"
#include "FSUDesignCompExp.H"
#include "ProblemDescDB.H"
#ifdef DAKOTA_DDACE
#include "Distribution.h"
#elif defined(DAKOTA_UTILIB)
#include <utilib/seconds.h>
#include "template_defs.h"
#endif

static const char rcsId[]="@(#) $Id: FSUDesignCompExp.C 4510 2007-09-14 04:03:24Z briadam $";


namespace Dakota {

/** This constructor is called for a standard iterator built with data from
    probDescDB. */
FSUDesignCompExp::FSUDesignCompExp(Model& model): PStudyDACE(model),
  samplesSpec(probDescDB.get_int("method.samples")), numSamples(samplesSpec),
  allDataFlag(false), numDACERuns(0),
  latinizeFlag(probDescDB.get_bool("method.latinize")),
  volQualityFlag(probDescDB.get_bool("method.quality_metrics")),
  varBasedDecompFlag(probDescDB.get_bool("method.variance_based_decomp"))
{
  if (methodName == "fsu_cvt") {
    // CVT inputs
    originalSeed =  probDescDB.get_int("method.random_seed");
    varyPattern  = !probDescDB.get_bool("method.fixed_seed");
    numCVTTrials =  probDescDB.get_int("method.fsu_cvt.num_trials");

    // Map sample_type string to trialType integer
    const String& trial_type = probDescDB.get_string("method.trial_type");
    if (trial_type == "grid")
      trialType = 2;
    else if (trial_type == "halton")
      trialType = 1;
    else
      trialType = -1; // default is "random"
  }
  else if (methodName == "fsu_halton" || methodName == "fsu_hammersley") {
    // QMC inputs
    sequenceStart =  probDescDB.get_div("method.fsu_quasi_mc.sequenceStart");
    sequenceLeap  =  probDescDB.get_div("method.fsu_quasi_mc.sequenceLeap");
    primeBase     =  probDescDB.get_div("method.fsu_quasi_mc.primeBase");
    varyPattern   = !probDescDB.get_bool("method.fsu_quasi_mc.fixed_sequence");
    // perform error checks and initialize defaults
    if (sequenceStart.empty()) {
      sequenceStart.reshape(numContinuousVars);
      sequenceStart = 0;
    }
    else if (sequenceStart.length() != numContinuousVars) {
      Cerr << "\nError: wrong number of sequence_start inputs.\n";
      abort_handler(-1);
    }
    if (sequenceLeap.empty()) {
      sequenceLeap.reshape(numContinuousVars);
      sequenceLeap = 1;
    }
    else if (sequenceLeap.length() != numContinuousVars) {
      Cerr << "\nError: wrong number of sequence_leap inputs.\n";
      abort_handler(-1);
    }
    if (primeBase.empty()) {
      primeBase.reshape(numContinuousVars);
      if (methodName == "fsu_halton") 
	for (size_t i=0; i<numContinuousVars; i++)
	  primeBase[i] = prime(i+1);
      else { // fsu_hammersley
	primeBase[0] = -numSamples;
	for (size_t i=1; i<numContinuousVars; i++)
	  primeBase[i] = prime(i);
      }
    }
    else if (methodName == "fsu_halton") {
      if (primeBase.length() != numContinuousVars) {
	Cerr << "\nError: wrong number of prime_base inputs.\n";
	abort_handler(-1);
      }
    }
    else { // fsu_hammersley
      if (primeBase.length() != numContinuousVars-1) {
	Cerr << "\nError: wrong number of prime_base inputs.\n";
	abort_handler(-1);
      }
      // modify user input
      primeBase.reshape(numContinuousVars);
      for (size_t i=numContinuousVars-1; i>0; i--)
	primeBase[i] = primeBase[i-1]; // move each input back one position
      primeBase[0] = -numSamples;
      //Cout << primeBase;
    }
  }
  else {
    Cerr << "Error: FSU DACE method \"" << methodName << "\" is not an option."
	 << endl;
    abort_handler(-1);
  }

  if (numSamples) // samples is optional (default = 0)
    maxConcurrency *= numSamples;
}


/** This alternate constructor is used for instantiations on-the-fly,
    using only the incoming data.  No problem description database
    queries are used. */
FSUDesignCompExp::
FSUDesignCompExp(Model& model, int samples, int seed,
		 const String& sampling_method):
  PStudyDACE(NoDBBaseConstructor(), model), samplesSpec(samples),
  numSamples(samples), allDataFlag(true), numDACERuns(0), latinizeFlag(false),
  volQualityFlag(false), varBasedDecompFlag(false), varyPattern(true)
{
  methodName = sampling_method;

  if (methodName == "fsu_cvt") {
    // CVT inputs and defaults
    originalSeed = seed;
    numCVTTrials = 10000;
    trialType    = -1; // default is "random"
  }
  else if (methodName == "fsu_halton" || methodName == "fsu_hammersley") {
    // QMC inputs and defaults
    // initialize defaults
    sequenceStart.reshape(numContinuousVars);
    sequenceStart = 0;
    sequenceLeap.reshape(numContinuousVars);
    sequenceLeap = 1;
    primeBase.reshape(numContinuousVars);
    if (methodName == "fsu_halton") 
      for (size_t i=0; i<numContinuousVars; i++)
	primeBase[i] = prime(i+1);
    else { // fsu_hammersley
      primeBase[0] = -numSamples;
      for (size_t i=1; i<numContinuousVars; i++)
	primeBase[i] = prime(i);
    }
  }
  else {
    Cerr << "Error: FSU DACE method \"" << methodName << "\" is not an option."
	 << endl;
    abort_handler(-1);
  }

  if (numSamples) // samples is optional (default = 0)
    maxConcurrency *= numSamples;
}


FSUDesignCompExp::~FSUDesignCompExp() { }


void FSUDesignCompExp::extract_trends()
{
  // if VBD has been selected, evaluate a series of parameter sets
  // (each of the size specified by the user) in order to compute VBD metrics.
  if (varBasedDecompFlag)
    var_based_decomp(numContinuousVars, numSamples);
  // if VBD has not been selected, evaluate a single parameter set of the size
  // specified by the user
  else {
    // obtain a set of samples for evaluation
    get_parameter_sets(iteratedModel);
    // evaluate the sample set to obtain the corresponding set of results
    evaluate_parameter_sets(iteratedModel, allDataFlag, true);
  }
}


void FSUDesignCompExp::get_parameter_sets(const Model& model)
{
  // error check on input parameters
  enforce_input_rules();

  // keep track of number of DACE executions for this object
  numDACERuns++;

  // Get bounded region and check that (1) the lengths of bounds arrays are 
  // consistent with numContinuousVars, and (2) the bounds are not default 
  // bounds (upper/lower = +/-DBL_MAX) since this results in Infinity in the 
  // sample_points returned.  Discrepancies can occur in the case of uncertain
  // variables, since they do not currently have global bounds specifications.
  // It would be nice to detect this and automatically delete any uncertain
  // variables (from numContinuousVars & local_vars).
  const RealVector& c_l_bnds = model.continuous_lower_bounds();
  const RealVector& c_u_bnds = model.continuous_upper_bounds();
  if (c_l_bnds.length() != numContinuousVars || 
      c_u_bnds.length() != numContinuousVars) {
    Cerr << "\nError: Mismatch in number of active variables and length of"
         << "\n       bounds arrays in FSUDesignCompExp." << endl;
    abort_handler(-1);
  }
  size_t i, j;
  RealVector c_bnds_range(numContinuousVars);
  for (i=0; i<numContinuousVars; i++) {
    if (c_l_bnds[i] <= -DBL_MAX/2. || c_u_bnds[i] >= DBL_MAX/2.) {
      Cerr << "\nError: FSUDesignCompExp requires specification of variable "
	   << "bounds for all active variables." << endl;
      abort_handler(-1);
    }
    c_bnds_range[i] = c_u_bnds[i] - c_l_bnds[i];
  }

  double* sample_points = new double [numContinuousVars*numSamples];

  if( methodName == "fsu_halton") {
    int qmc_step = (varyPattern) ? (numDACERuns-1)*numSamples+1 : 1;
    fsu_halton(numContinuousVars, numSamples, qmc_step, sequenceStart,
	       sequenceLeap, primeBase, sample_points);
  }
  else if( methodName == "fsu_hammersley") {
    int qmc_step = (varyPattern) ? (numDACERuns-1)*numSamples+1 : 1;
    fsu_hammersley(numContinuousVars, numSamples, qmc_step, sequenceStart,
		   sequenceLeap, primeBase, sample_points);
  }
  else if( methodName == "fsu_cvt") {

    // # of initialization sampes = # samples
    // some initialization types (grid) may truncate a pattern if the number
    // of samples is not a power of the number of variables

    // We no longer allow the user to specify an initialization 
    // type although the FSU CVT code allows it.  The reason is that
    // the method is fairly insensitive to initialization type and we wanted 
    // to simplify the interface as much as possible.  If it needs to 
    // be added later, it can be done using the "initialization_type" 
    // keyword and allowing it to be one of four options:  init_grid, 
    // init_halton, init_uniform, and init_random. 
    int init_type = 0; // random initialization is giving problems with seed 
                       // handling; current default is now uniform 

    // Trial type can be one of four values: random, grid, halton, uniform
    // random  = system rand() (to be replaced w/ GSL)
    // uniform = John's personal RNG (simple + portable/reproducible)
    // --> for simplicity, uniform has been removed.

    // trial_type --> could hardwire to random as well

    // batch_size is no longer part of the spec; we now use:
    int batch_size = min(10000, numCVTTrials);
    if (numCVTTrials < numSamples)
      numCVTTrials = numSamples *10;

    // assign default maxIterations (DataMethod default is zero)
    if (!maxIterations)
      maxIterations = 30;

    // Set seed value for input to CVT.  A user-specified seed gives you
    // repeatable behavior but no specification gives you random behavior (seed
    // generated from a system clock).  For the cases where extract_trends() may
    // be called multiple times for the same DACE object (e.g., SBO), a
    // deterministic sequence of seed values is used (unless fixed_seed has been
    // specified).  This renders the study repeatable but the sampling pattern
    // varies from one run to the next.
    if (numDACERuns == 1) { // set initial seed
      if (originalSeed)    // user seed specification: repeatable behavior
	randomSeed = originalSeed;
      else                 // no user specification: random behavior
	// Generate initial seed from a system clock.  NOTE: the system clock
	// is not reused on subsequent invocations since (1) clock granularity
	// can be too coarse (can repeat on subsequent runs for inexpensive test
	// fns) and (2) seed progression can be highly structured, which could
	// induce correlation between sample sets.  Instead, the clock-generated
	// case uses the same seed progression procedure below as the
	// user-specified case.  This has the additional benefit that a random
	// run can be recreated by specifying the clock-generated seed in the
	// input file.
#ifdef DAKOTA_DDACE
	randomSeed = 1 + DistributionBase::timeSeed(); // microsecs, time of day
#elif defined(DAKOTA_UTILIB)
        randomSeed = 1 + (int)CurrentTime(); // secs, time of day
#else
        randomSeed = 1 + (int)clock(); // last resort: clock ticks, exec time
#endif
    }
    else if (varyPattern) { // define sequence of seed values for numLHSRuns > 1
      // It would be preferable to call srand() only once and then call rand()
      // for each execution (the intended usage model), but possible interaction
      // with other uses of rand() in ANN, SGOPT, APPS, etc. is a concern. E.g.,
      // an srand(clock()) executed elsewhere would ruin the repeatability of
      // the NonDSampling seed sequence.  The only way to insure isolation is to
      // reseed each time.  Any induced correlation should be inconsequential
      // for the intended use.
      srand(randomSeed);
      randomSeed = 1 + rand(); // from 1 to RANDMAX+1
    }
    Cout << "\nFSU DACE method = " << methodName << " Samples = " << numSamples;
    if (varyPattern && numDACERuns > 1)
      Cout << " Seed (deterministic sequence) = " << randomSeed << '\n';
    else if (originalSeed){
      randomSeed = originalSeed;
      Cout << " Seed (user-specified) = " << randomSeed << '\n';}
    else
      Cout << " Seed (system-generated) = " << randomSeed << '\n';

    int* p_seed = &randomSeed;
    int* diag_num_iter = new int; // CVT returns actual number of iterations

    // Now generate the array of samples
    fsu_cvt(numContinuousVars, numSamples, batch_size, init_type, trialType,
	    numCVTTrials, maxIterations, p_seed, sample_points,
	    diag_num_iter);

    p_seed = NULL;
    delete diag_num_iter;
  }

  if (latinizeFlag)
    fsu_latinize(numContinuousVars, numSamples, sample_points);

  if (volQualityFlag)
    volumetric_quality(numContinuousVars, numSamples, sample_points);

  // Convert from sample_points on [0,1] to allVariables on [lower,upper]
  if (allVariables.length() != numSamples)
    allVariables.reshape(numSamples);
  const pair<short,short>& view = iteratedModel.current_variables().view();
  RealVector c_vars(numContinuousVars);
  size_t cntr = 0;
  for (i=0; i<numSamples; i++) {
    if (allVariables[i].is_null())
      allVariables[i] = Variables(view); // minimal instantiate-on-the-fly
    for (j=0; j<numContinuousVars; j++)
      c_vars[j] = c_l_bnds[j] + sample_points[cntr++] * c_bnds_range[j];
    allVariables[i].continuous_variables(c_vars);
  }

  delete [] sample_points;
}


/** Users may input a variety of quantities, but this function must
    enforce any restrictions imposed by the sampling algorithms. */
void FSUDesignCompExp::enforce_input_rules()
{
  // error check
  if (numSamples <= 0) {
    Cerr << "Error: number of DACE samples must be greater than zero." << endl;
    abort_handler(-1);
  }

  if (methodName == "fsu_cvt") {
    // no input rules yet
  }
  else { // QMC
    // check for duplication in primeBase sequence
    IntVector tempPrime = primeBase;
    std::sort(tempPrime.begin(), tempPrime.end());
    bool primeEqual = false;
    for (size_t i=0; i<numContinuousVars-1; i++)
      if (tempPrime[i] == tempPrime[i+1])
	primeEqual = true;
    if (primeEqual) {
      Cerr << "\nError: please specify unique prime numbers for Quasi-MC "
	   << "methods." << endl;
      abort_handler(-1);
    }
  }
}

} // namespace Dakota
