/*!\file: sealevelrise_core.cpp
 * \brief: core of the SLR solution 
 */ 

#include "./cores.h"
#include "../toolkits/toolkits.h"
#include "../classes/classes.h"
#include "../shared/shared.h"
#include "../modules/modules.h"
#include "../solutionsequences/solutionsequences.h"

void slrconvergence(bool* pconverged, Vector<IssmDouble>* Sg,Vector<IssmDouble>* Sg_old,IssmDouble eps_rel,IssmDouble eps_abs);

void sealevelrise_core(FemModel* femmodel){

	Vector<IssmDouble> *Sg    = NULL;
	Vector<IssmDouble> *Sg_old    = NULL;
	Vector<IssmDouble> *Sgi    = NULL; //ice convolution
	Vector<IssmDouble> *Sgi_old    = NULL; 
	Vector<IssmDouble> *Sgo    = NULL; //ocean convolution
	Vector<IssmDouble> *Sgo_old    = NULL; 

	/*parameters: */
	int count;
	bool save_results;
	int  gsize;
	int  configuration_type;
	bool spherical=true;
	bool converged=true;
	int max_nonlinear_iterations;
	IssmDouble           eps_rel;
	IssmDouble           eps_abs;
	IssmDouble          *x    = NULL;
	IssmDouble          *y    = NULL;
	IssmDouble          *z    = NULL;
	IssmDouble           eustatic;


	/*Recover some parameters: */
	femmodel->parameters->FindParam(&save_results,SaveResultsEnum);
	femmodel->parameters->FindParam(&configuration_type,ConfigurationTypeEnum);
	femmodel->parameters->FindParam(&max_nonlinear_iterations,SealevelriseMaxiterEnum);
	femmodel->parameters->FindParam(&eps_rel,SealevelriseReltolEnum);
	femmodel->parameters->FindParam(&eps_abs,SealevelriseAbstolEnum);

	if(VerboseSolution()) _printf0_("   computing sea level rise\n");

	/*Call on core computations: */
	femmodel->SetCurrentConfiguration(SealevelriseAnalysisEnum);

	/*first, recover lat,long and radius vectors from vertices: */
	VertexCoordinatesx(&x,&y,&z,femmodel->vertices); //no need for z coordinate

	/*Figure out size of g-set deflection vector and allocate solution vector: */
	gsize      = femmodel->nodes->NumberOfDofs(configuration_type,GsetEnum);
	
	/*Initialize:*/
	Sg = new Vector<IssmDouble>(gsize);
	Sg->Assemble();
	Sg_old = new Vector<IssmDouble>(gsize);
	Sg_old->Assemble();

	Sgi = new Vector<IssmDouble>(gsize);
	Sgi->Assemble();
	Sgi_old = new Vector<IssmDouble>(gsize);
	Sgi_old->Assemble();

	Sgo = new Vector<IssmDouble>(gsize);
	Sgo->Assemble();
	Sgo_old = new Vector<IssmDouble>(gsize);
	Sgo_old->Assemble();

	count=1;
	converged=false;
	
	/*Start loop: */
	for(;;){

		//save pointer to old sea level rise
		delete Sgi_old; Sgi_old=Sgi; 
		delete Sgo_old; Sgo_old=Sgo; 
		delete Sg_old; Sg_old=Sg; 

		/*Initialize solution vector: */
		Sg = new Vector<IssmDouble>(gsize); Sg->Assemble();
		Sgi = new Vector<IssmDouble>(gsize); Sgi->Assemble();
		Sgo = new Vector<IssmDouble>(gsize); Sgo->Assemble();

		/*call the main module: */
        femmodel->Sealevelrise(Sgi,Sgo, &eustatic, Sg_old, Sgi_old, Sgo_old,x, y, z);
		
		/*assemble solution vectors: */
		Sgi->Assemble();
		Sgo->Assemble();
		
		/*Sg is the sum of the ice and ocean convolutions + eustatic component: (eq 4 in Farrel and Clark)*/
		Sgi->Copy(Sg); Sg->AXPY(Sgo,1); 
		Sg->Shift(eustatic);

		/*convergence criterion:*/
		slrconvergence(&converged,Sg,Sg_old,eps_rel,eps_abs);

		/*Increase count: */
		count++;
		if(converged==true){
			break;
		}
		if(count>=max_nonlinear_iterations){
			_printf0_("   maximum number of nonlinear iterations (" << max_nonlinear_iterations << ") exceeded\n"); 
			converged=true;
			break;
		}	
		
	}
	if(VerboseConvergence()) _printf0_("\n   total number of iterations: " << count-1 << "\n");

	InputUpdateFromVectorx(femmodel,Sg,SealevelriseSEnum,VertexSIdEnum);

	if(save_results){
		if(VerboseSolution()) _printf0_("   saving results\n");
		int outputs[1] = {SealevelriseSEnum};
		femmodel->RequestedOutputsx(&femmodel->results,&outputs[0],1);
	}
	xDelete<IssmDouble>(x);
	xDelete<IssmDouble>(y);
	xDelete<IssmDouble>(z);
	delete Sg_old;
	delete Sg;
}

void slrconvergence(bool* pconverged, Vector<IssmDouble>* Sg,Vector<IssmDouble>* Sg_old,IssmDouble eps_rel,IssmDouble eps_abs){ /*{{{*/
	
	bool converged=true;
	IssmDouble ndS,nS; 
	Vector<IssmDouble> *dSg    = NULL;

	//compute norm(du) and norm(u) if requested
	dSg=Sg_old->Duplicate(); Sg_old->Copy(dSg); dSg->AYPX(Sg,-1.0);
	ndS=dSg->Norm(NORM_TWO); 
	
	if(!xIsNan<IssmDouble>(eps_rel)){
		nS=Sg_old->Norm(NORM_TWO);
	}

	if (xIsNan<IssmDouble>(ndS) || xIsNan<IssmDouble>(nS)) _error_("convergence criterion is NaN!");

	//clean up
	delete dSg;

	//print
	if(!xIsNan<IssmDouble>(eps_rel)){
		if((ndS/nS)<eps_rel){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " < " << eps_rel*100 << " %\n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)/norm(S)" << ndS/nS*100 << " > " << eps_rel*100 << " %\n");
			converged=false;
		}
	}
	if(!xIsNan<IssmDouble>(eps_abs)){
		if(ndS<eps_abs){
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)" << ndS << " < " << eps_abs << " \n");
		}
		else{ 
			if(VerboseConvergence()) _printf0_(setw(50) << left << "      convergence criterion: norm(dS)" << ndS << " > " << eps_abs << " \n");
			converged=false;
		}
	}

	/*assign output*/
	*pconverged=converged;

} /*}}}*/
