/*This is where we have all our webgl relevant functionality for the plotting routines: */
//{{{ GL Initialization
function initWebGL(canvas,options) { //{{{
	gl = null;

	try {
		// Try to grab the standard context. If it fails, fallback to experimental.
		gl = canvas.getContext("webgl") || canvas.getContext("experimental-webgl");
	}
	catch(e) {}

	// If we don't have a GL context, give up now
	if (!gl) {
		alert("Unable to initialize WebGL. Your browser may not support it.");
		gl = null;
	}
		
	// Enable depth testing
	gl.enable(gl.DEPTH_TEST);
	// Near things obscure far things
	gl.depthFunc(gl.LEQUAL);
	// Clear the color as well as the depth buffer.
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
	// Enable color blending/overlay
	gl.enable(gl.BLEND);

	// Allocate arrays equal to maximium number of attributes used by any one shader
	gl.enableVertexAttribArray(0);
	gl.enableVertexAttribArray(1);

	// Load shaders and store them in gl object
	gl.shaders = loadShaders(gl);
	


	// Add context state variables
	//TODO:Group variables in objects for organization and naming
	canvas.zoomBounds = options.getfieldvalue('zoombounds',[0,-.52]);
	canvas.zoomFactor = clamp(options.getfieldvalue('zoomfactor',canvas.zoomBounds[1]), canvas.zoomBounds[1], canvas.zoomBounds[0]);
	canvas.zoomLast = canvas.zoomFactor;
	canvas.cameraMatrix = mat4.create();
	canvas.translation = options.getfieldvalue('centeroffset',[0,0,0.25]);
	canvas.rotationAzimuthBounds = options.getfieldvalue('azimuthbounds',[0,360]);
	canvas.rotationElevationBounds = options.getfieldvalue('elevationbounds',[-180,180]);
	canvas.rotationDefault = options.getfieldvalue('view',[0,90]); //0 azimuth - up is north, 90 elevation - looking straight down
	canvas.rotation = canvas.rotationDefault;
	canvas.controlsensitivity = 1;

	if (options.getfieldvalue('2d','off') == 'on') {
		canvas.rotationAzimuthBounds = [0,0];
		canvas.rotationElevationBounds = [90,90];
		canvas.rotationDefault = [0,90];
		canvas.rotation = canvas.rotationDefault;
	}
	
	// Add event listeners for canvas
	canvas.addEventListener("mousewheel", function (ev) {onZoom(ev,canvas,options)}, false);
	canvas.addEventListener("DOMMouseScroll", function (ev) {onZoom(ev,canvas,options)}, false);

	var mc = new Hammer.Manager(canvas);

    mc.add(new Hammer.Pan({threshold:0, pointers:0}));
    mc.add(new Hammer.Pinch({threshold:0})).recognizeWith(mc.get('pan'));

    mc.on("panstart panmove", function (ev) {onPan(ev,canvas,options);});
    mc.on("pinchstart pinchmove", function (ev) {onPinch(ev,canvas,options);});
	//mc.on("mousewheel DOMMouseScroll", function (ev) {onZoom(ev,canvas,options);});
	
	return gl;
} //}}}
function initBuffers(gl,arrays) { //{{{	
	var bufferArray = [];
	for (var i = 0; i < arrays.length; i++) {
		bufferArray[i] = gl.createBuffer();	
		bufferArray[i].itemSize = arrays[i].itemSize;
		bufferArray[i].numItems = arrays[i].length/bufferArray[i].itemSize;
		
		if (bufferArray[i].itemSize > 1) {
			gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(arrays[i]), gl.STATIC_DRAW);
		}
		else {
			//TODO: identify index buffers uniquely (by name)
			gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, new Uint16Array(arrays[i]), gl.STATIC_DRAW);
		}
	}	
	return bufferArray;
} //}}}
function initTexture(gl,imageSource) { //{{{
	var texture = gl.createTexture();
	texture.image = new Image();
	texture.isLoaded = false;
	texture.image.onload = function () {
		handleLoadedTexture(gl,texture);
	}
	texture.image.src = imageSource;
	return texture;
} //}}}
function handleLoadedTexture(gl,texture) { //{{{
	gl.activeTexture(gl.TEXTURE0);
	gl.bindTexture(gl.TEXTURE_2D, texture);
	gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
	gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, texture.image);
	gl.generateMipmap(gl.TEXTURE_2D);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST_MIPMAP_LINEAR);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
	gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
	gl.bindTexture(gl.TEXTURE_2D, null);
	texture.isLoaded = true;
} //}}}
function Node(gl,options) { //{{{
	//Returns a Node object that contains default display states for webgl object
	return {buffers:[],
		shader:gl.shaders["colored"]["program"],
		draw:null,
		hideOcean:false,
		level:0,
		useIndexBuffer:true,
		useOrthographic:false,
		alpha:1.0,
		disableDepthTest:false, 
		enableCullFace:false,
		cullFace:gl.FRONT,
		drawMode:gl.TRIANGLES,
		texture:null,
		translation:vec3.create(),
		rotation:vec3.fromValues(-90, 0, 0),
		scale:vec3.fromValues(1, 1, 1),
		modelMatrix:mat4.create(),
		shaderName:"colored",
		overlay:false,
		drawOrder:0,
		maskEnabled:false,
		maskHeight:150.0,
		maskColor:vec4.fromValues(0.0, 0.0, 1.0, 1.0),
		enabled:true,
	};
} //}}}
function recalculateModelMatrix(node) { //{{{
	//TODO: move to 0,0,0, rotate,move back to normal space, then apply transform
	var modelMatrix = mat4.create();

	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, node["scale"]);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, node["translation"]); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);
	
	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, radians(node["rotation"][2]), [0.0, 0.0, 1.0]);
	mat4.multiply(modelMatrix, zRotationMatrix, modelMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, radians(node["rotation"][1]), [0.0, 1.0, 0.0]);
	mat4.multiply(modelMatrix, yRotationMatrix, modelMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, radians(node["rotation"][0]), [1.0, 0.0, 0.0]);
	mat4.multiply(modelMatrix, xRotationMatrix, modelMatrix);

	return modelMatrix;
} //}}}
function radians (degrees) { //{{{
  return degrees * Math.PI / 180;
} //}}}
function degrees (radians) { //{{{
  return radians * 180 / Math.PI;
} //}}}
function clamp(value, min, max) { //{{{
	return Math.max(min, Math.min(value, max));
} //}}}
function recoverview(canvasid,defaultview) { //{{{
	var canvas  = document.getElementById(canvasid);
	if (canvas && canvas.hasOwnProperty("rotation")) {
		return canvas.rotation;
	}
	return defaultview;
} //}}}
function recovercenteroffset(canvasid,defaultcenter) { //{{{
	var canvas  = document.getElementById(canvasid);
	if (canvas && canvas.hasOwnProperty("translation")) {
		return canvas.translation;
	}
	return defaultcenter;
} //}}}
//}}}
//{{{ Shader Loading
function loadShaders(gl) { //{{{
	var shaderNames = ["colored", "unlit_textured"];
	shaders = {};
	shaders["colored"] = {loaded:false, vsh:{}, fsh:{}};
	shaders["colored"]["vsh"]["string"] = 
		['attribute vec3 aVertexPosition;',
		'attribute vec4 aVertexColor;',
		'',
		'uniform mat4 uMVPMatrix;',
		'uniform float uAlpha;',
		'',
		'varying vec4 vColor;',
		'',
		'void main(void) {',
		'	gl_Position = uMVPMatrix * vec4(aVertexPosition.xyz, 1.0);',
		'	vColor = vec4(aVertexColor.xyz, uAlpha);',
		'}'].join('\n');
	shaders["colored"]["fsh"]["string"] =
		['precision mediump float;',
		'',
		'varying vec4 vColor;',
		'',
		'void main(void) {',
		'	gl_FragColor = vColor;',
		'}'].join('\n');
	shaders["unlit_textured"] = {loaded:false, vsh:{}, fsh:{}};
	shaders["unlit_textured"]["vsh"]["string"] = 
		['attribute vec3 aVertexPosition;',
		'attribute vec2 aTextureCoord;',
		'',
		'uniform mat4 uMVPMatrix;',
		'',
		'varying vec2 vTextureCoord;',
		'varying float vZCoord;',
		'',
		'void main(void) {',
		'	gl_Position = uMVPMatrix * vec4(aVertexPosition.xyz, 1.0);',
		'	vTextureCoord = aTextureCoord;',
		'	vZCoord = aVertexPosition.z;',
		'}'].join('\n');
	shaders["unlit_textured"]["fsh"]["string"] =
		['precision mediump float;',
		'',
		'varying vec2 vTextureCoord;',
		'varying float vZCoord;',
		'',
		'uniform sampler2D uColorSampler;',
		'uniform float uAlpha;',
		'uniform bool uMaskEnabled;',
		'uniform float uMaskHeight;',
		'uniform vec4 uMaskColor;',
		'',
		'void main(void) {',
		'	if (uMaskEnabled && (vZCoord < uMaskHeight)) {',
		'		gl_FragColor = vec4(uMaskColor.xyz, uAlpha);',
		'	}',
		'	else {',
		'		gl_FragColor = vec4(texture2D(uColorSampler, vec2(vTextureCoord.s, vTextureCoord.t)).rgb, uAlpha);',
		'	}',
		'}'].join('\n');
	shaderNames.forEach(function(shaderName){
		shaders[shaderName]["vsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["vsh"]["string"], "vsh");
		shaders[shaderName]["fsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["fsh"]["string"], "fsh");

		shaders[shaderName]["program"] = gl.createProgram();
		gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["vsh"]["shader"]);
		gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["fsh"]["shader"]);
		gl.linkProgram(shaders[shaderName]["program"]);

		if (!gl.getProgramParameter(shaders[shaderName]["program"], gl.LINK_STATUS)) {
			alert("Could not initialise shaders");
		}

		var vshStringArray = shaders[shaderName]["vsh"]["string"].split("\n");
		var fshStringArray = shaders[shaderName]["fsh"]["string"].split("\n");
		var line = "";
		var property = "";
		for (var i = 0; i < vshStringArray.length; i++) {
			line = vshStringArray[i];
			if (line.search("attribute") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getAttribLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("uniform") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("void main") != -1) {
				break;
			}
		}
		for (var i = 0; i < fshStringArray.length; i++) {
			line = fshStringArray[i];
			if (line.search("uniform") != -1) {
				property = nameFromLine(line);
				shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
			}
			else if (line.search("void main") != -1) {
				break;
			}
		}
		shaders[shaderName]["loaded"] = true;
	});
	return shaders;
} //}}}
function getShaderByString(gl,str,type) { //{{{
	var shader;
	if (type == "fsh") {
		shader = gl.createShader(gl.FRAGMENT_SHADER);
	}
	else if (type == "vsh") {
		shader = gl.createShader(gl.VERTEX_SHADER);
	}
	else {
		return null;
	}
	
	gl.shaderSource(shader, str);
	gl.compileShader(shader);

	if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {	
		alert(gl.getShaderInfoLog(shader));
		return null;
	}

	return shader;
} //}}}
function nameFromLine(line) { //{{{
	//returns lowerCamelCase property name from shader line
	var fullName = line.split(" ")[2];
	return fullName.slice(0, fullName.search(";"));
} //}}}
//}}}
//{{{ Interface Functions
function onPan(ev,canvas,options) { //{{{
	ev.preventDefault();
	if (ev.type == 'panstart') {
		canvas.lastDeltaX = 0;
		canvas.lastDeltaY = 0;
	}
	if (ev.srcEvent.shiftKey || ev.pointers.length == 2) {
		var deltaX = (canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth * canvas.zoomFactor * 2;
		var deltaY = (canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight * canvas.zoomFactor * 2;
		
		canvas.translation[0] += Math.cos(radians(canvas.rotation[0])) * deltaX - Math.sin(radians(canvas.rotation[0])) * deltaY;
		canvas.translation[1] += Math.sin(radians(canvas.rotation[0])) * deltaX + Math.cos(radians(canvas.rotation[0])) * deltaY;
	}
	else {
		canvas.rotation[0] += degrees((canvas.lastDeltaX - ev.deltaX) / canvas.clientWidth * canvas.zoomFactor * 2);
		canvas.rotation[1] += degrees((canvas.lastDeltaY - ev.deltaY) / canvas.clientHeight * canvas.zoomFactor * 2);
		
		if (canvas.rotation[0] > 360) {canvas.rotation[0] -= 360};
		if (canvas.rotation[0] < 0) {canvas.rotation[0] += 360};
		if (canvas.rotation[1] > 180) {canvas.rotation[1] -= 360};
		if (canvas.rotation[1] < -180) {canvas.rotation[1] += 360};
		
		canvas.rotation[0] = clamp(canvas.rotation[0], canvas.rotationAzimuthBounds[0], canvas.rotationAzimuthBounds[1]);
		canvas.rotation[1] = clamp(canvas.rotation[1], canvas.rotationElevationBounds[0], canvas.rotationElevationBounds[1])
	}
	canvas.lastDeltaX = ev.deltaX;
	canvas.lastDeltaY = ev.deltaY;
	
	if (options.getfieldvalue('displayview','off') == 'on') console.log(canvas.rotation);
} //}}}
function onPinch(ev,canvas,options) { //{{{
	ev.preventDefault();
	if (ev.type == 'pinchstart') {
		canvas.zoomLast = canvas.zoomFactor;
	}
	else {
		canvas.zoomFactor = clamp(ev.scale * canvas.zoomLast, canvas.zoomBounds[1], canvas.zoomBounds[0]);
		if (options.getfieldvalue('displayzoom','off') == 'on') console.log(canvas.zoomFactor);
	}
} //}}}
function onZoom(ev,canvas,options) { //{{{
	ev.preventDefault();
	var delta = 1/10 * clamp(ev.scale || ev.wheelDelta || -ev.detail, -1, 1);
	canvas.zoomFactor = clamp(canvas.zoomFactor - delta * canvas.zoomFactor, canvas.zoomBounds[1], canvas.zoomBounds[0]);
	if (options.getfieldvalue('displayzoom','off') == 'on') console.log(canvas.zoomFactor);
} //}}}
//}}}
//{{{ Drawing Functions
function updateCameraMatrix(canvas) { //{{{
    //Update view matrix and multiply with projection matrix to get the view-projection (camera) matrix.
	var vMatrix = mat4.create();
	var pMatrix = mat4.create();
	var translateMatrix = mat4.create();
	var rotationMatrix = mat4.create();
	var azimuthRotationMatrix = mat4.create();
	var elevationRotationMatrix = mat4.create();

	mat4.perspective(pMatrix, 90 * Math.PI / 180, canvas.clientWidth / canvas.clientHeight, 0.001, 10000.0);
	
	//Apply worldspace translation
	mat4.translate(translateMatrix, translateMatrix, [canvas.translation[0], canvas.translation[2], canvas.translation[1]]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);
	
	//Calculate rotation around camera focal point about worldspace origin
	mat4.rotate(azimuthRotationMatrix, azimuthRotationMatrix, radians(canvas.rotation[0]), [0, 1, 0]);
	mat4.rotate(elevationRotationMatrix, elevationRotationMatrix, radians(canvas.rotation[1]), [1, 0, 0]);
	mat4.multiply(rotationMatrix, elevationRotationMatrix, azimuthRotationMatrix);
	
	//Apply rotation and scaling transform
	mat4.multiply(vMatrix, rotationMatrix, vMatrix);
	
	//Apply screenspace translation
	mat4.identity(translateMatrix);
	mat4.translate(translateMatrix, translateMatrix, [0.0, 0.0, canvas.zoomFactor]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);

	//Apply projection matrix to get camera matrix
	mat4.multiply(canvas.cameraMatrix, pMatrix, vMatrix);
//	canvas.cameraMatrix = mat4.create();
}//}}}
function drawSceneGraphNode(gl,canvas,node) { //{{{
	if (!node["enabled"]) {
		return;
	}
	if (node["texture"]) {
		if (!node["texture"]["isLoaded"]) {
			return;
		}
	}
	bindAttributes(gl, node["shader"], node["buffers"]);
	var mvpMatrix = mat4.create();
	if (node["useOrthographic"] == true) {
		mat4.ortho(mvpMatrix, -1.0, 1.0, -1.0, 1.0, -1.0, 1.0);
	}
	else {
		mat4.multiply(mvpMatrix, canvas.cameraMatrix, node["modelMatrix"]);
	}
	gl.uniformMatrix4fv(node["shader"]["uMVPMatrix"], false, mvpMatrix);
	gl.uniform1f(node["shader"]["uAlpha"], node["alpha"]);
	gl.uniform1i(node["shader"]["uMaskEnabled"], node["maskEnabled"]);
	gl.uniform1f(node["shader"]["uMaskHeight"], node["maskHeight"]);
	gl.uniform4fv(node["shader"]["uMaskColor"], node["maskColor"]);
	if (node["texture"]) {
		gl.activeTexture(gl.TEXTURE0);
		gl.bindTexture(gl.TEXTURE_2D, node["texture"]);
		gl.uniform1i(node["shader"]["uColorSampler"], 0);	
	}
	//if (node["overlay"] == true) {
	//	gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
	//}
	//else {
	//	gl.blendFunc (gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
	//}
	gl.blendFunc (gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
	if  (node["useIndexBuffer"] == true) {
		gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, node["buffers"][node["buffers"].length - 1]);
		gl.drawElements(node["drawMode"], node["buffers"][node["buffers"].length - 1].numItems, gl.UNSIGNED_SHORT, 0);
	}
	else {
		gl.drawArrays(node["drawMode"], 0, node["buffers"][0].numItems);
	}	
} //}}}
function bindAttributes(gl,shaderProgram,bufferArray) { //{{{
	gl.useProgram(shaderProgram);
	var arrayNumber = 0;
	for (var propertyName in shaderProgram) {
		if (propertyName[0] == "a") {
			if (bufferArray[arrayNumber].itemSize > 1) {
				gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[arrayNumber]);
				gl.vertexAttribPointer(shaderProgram[propertyName], bufferArray[arrayNumber].itemSize, gl.FLOAT, false, 0, 0);
				arrayNumber++;
			}
		}
	}
} //}}}
function draw(gl,options,canvas,nodes) { //{{{
	// Ensure canvas and gl viewport sizes are the same
	var displayWidth  = canvas.clientWidth;
	var displayHeight = canvas.clientHeight;
	if (canvas.width  != displayWidth || canvas.height != displayHeight) {
		canvas.width  = displayWidth;
		canvas.height = displayHeight;
		gl.viewport(0, 0, canvas.width, canvas.height);
	}
	
	// Set clear color to black, fully opaque
	var backgroundcolor=new RGBColor(options.getfieldvalue('backgroundcolor','lightcyan'));
	if(backgroundcolor.ok){
		gl.clearColor(backgroundcolor.r/255.0, backgroundcolor.g/255.0, backgroundcolor.b/255.0, 1.0);
	}
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out background color for curent canvas ",canvas));
		
	// Skip drawing of new frame if any texture is not yet loaded
	for (var node in nodes) {
		if (nodes[node]["texture"] && !nodes[node]["texture"]["isLoaded"]) {
			window.requestAnimationFrame(function(time) {draw(gl,options,canvas,nodes)});
			return;
		}
	}
	// Else, clear the color as well as the depth buffer for new frame
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

	window.requestAnimationFrame(function(time) {draw(gl,options,canvas,nodes)});
	
	updateCameraMatrix(canvas);
	
	var drawPassNumber = 2;
	for (var i = drawPassNumber - 1; i >= 0; i--) {
		for (var node in nodes) {
			if (nodes[node]["drawOrder"] == i) {
				drawSceneGraphNode(gl, canvas, nodes[node]);
			}
		}
	}
} //}}}
//}}}