%CALVING class definition
%
%   Usage:
%      calving=calving();

classdef calving
	properties (SetAccess=public) 
		stabilization = 0;
		spclevelset   = NaN;
		calvingrate   = NaN;
		meltingrate   = NaN;
	end
	methods
		function self = calving(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					inputstruct=varargin{1};
					list1 = properties('calving');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							self.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			self.spclevelset=project3d(md,'vector',self.spclevelset,'type','node');
			self.calvingrate=project3d(md,'vector',self.calvingrate,'type','node');
			self.meltingrate=project3d(md,'vector',self.meltingrate,'type','node');
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%stabilization = 2 by default
			self.stabilization = 2;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			%Early return
			if (solution~=TransientSolutionEnum() | md.transient.iscalving==0), return; end

			md = checkfield(md,'fieldname','calving.spclevelset','Inf',1,'timeseries',1);
			md = checkfield(md,'fieldname','calving.stabilization','values',[0 1 2]);
			md = checkfield(md,'fieldname','calving.calvingrate(1:md.mesh.numberofvertices,:)','>=',0,'timeseries',1,'NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','calving.meltingrate(1:md.mesh.numberofvertices,:)','>=',0,'timeseries',1,'NaN',1,'Inf',1);
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Calving parameters:'));
			fielddisplay(self,'stabilization','0: no, 1: artificial_diffusivity, 2: streamline upwinding');
			fielddisplay(self,'spclevelset','Levelset constraints (NaN means no constraint)');
			fielddisplay(self,'calvingrate','calving rate at given location [m/a]');
			fielddisplay(self,'meltingrate','melting rate at given location [m/a]');
		end % }}}
		function marshall(self,md,fid) % {{{
			yts=365.0*24.0*3600.0;
			WriteData(fid,'enum',CalvingLawEnum(),'data',DefaultCalvingEnum(),'format','Integer');
			WriteData(fid,'enum',LevelsetStabilizationEnum(),'data',self.stabilization,'format','Integer');
			WriteData(fid,'enum',SpcLevelsetEnum(),'data',self.spclevelset,'format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',self,'fieldname','calvingrate','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'scale',1./yts);
			WriteData(fid,'object',self,'fieldname','meltingrate','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1,'scale',1./yts);
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.calving.stabilization'],self.stabilization);
			writejs1Darray(fid,[modelname '.calving.spclevelset'],self.spclevelset);
			writejs1Darray(fid,[modelname '.calving.calvingrate'],self.calvingrate);
			writejs1Darray(fid,[modelname '.calving.meltingrate'],self.meltingrate);

		end % }}}
	end
end
