%MATICE class definition
%
%   Usage:
%      matice=matice();

classdef matice
	properties (SetAccess=public) 
		rho_ice                    = 0.;
		rho_water                  = 0.;
		rho_freshwater             = 0.;
		mu_water                   = 0.;
		heatcapacity               = 0.;
		latentheat                 = 0.;
		thermalconductivity        = 0.;
		temperateiceconductivity   = 0.;
		meltingpoint               = 0.;
		beta                       = 0.;
		mixed_layer_capacity       = 0.;
		thermal_exchange_velocity  = 0.;
		rheology_B   = NaN;
		rheology_n   = NaN;
		rheology_law = '';

		%gia: 
		lithosphere_shear_modulus  = 0.;
		lithosphere_density        = 0.;
		mantle_shear_modulus       = 0.;
		mantle_density             = 0.;

	end
	methods
		function createxml(self,fid) % {{{
			fprintf(fid, '\n\n');
			fprintf(fid, '<!-- materials -->\n');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="rho_ice" type="',class(self.rho_ice),'" default="',convert2str(self.rho_ice),'">','     <section name="materials" />','     <help> ice density [kg/m^3] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="rho_water" type="',class(self.rho_water),'" default="',convert2str(self.rho_water),'">','     <section name="materials" />','     <help> ocean water density [kg/m^3] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="rho_freshwater" type="',class(self.rho_freshwater),'" default="',convert2str(self.rho_freshwater),'">','     <section name="materials" />','     <help> fresh water density [kg/m^3] </help>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="mu_water" type="',class(self.mu_water),'" default="',convert2str(self.mu_water),'">','     <section name="materials" />','     <help> water viscosity [N s/m^2] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="heatcapacity" type="',class(self.heatcapacity),'" default="',convert2str(self.heatcapacity),'">','     <section name="materials" />','     <help> heat capacity [J/kg/K] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="latentheat" type="',class(self.latentheat),'" default="',convert2str(self.latentheat),'">','     <section name="materials" />','     <help> latent heat of fusion [J/kg] </help>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="thermalconductivity" type="',class(self.thermalconductivity),'" default="',convert2str(self.thermalconductivity),'">','     <section name="materials" />','     <help> ice thermal conductivity [W/m/K] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="temperateiceconductivity" type="',class(self.temperateiceconductivity),'" default="',convert2str(self.temperateiceconductivity),'">','     <section name="materials" />','     <help> temperate ice thermal conductivity [W/m/K] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="meltingpoint" type="',class(self.meltingpoint),'" default="',convert2str(self.meltingpoint),'">','     <section name="materials" />','     <help> melting point of ice at 1atm in K </help>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="beta" type="',class(self.beta),'" default="',convert2str(self.beta),'">','     <section name="materials" />','     <help> rate of change of melting point with pressure [K/Pa] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="mixed_layer_capacity" type="',class(self.mixed_layer_capacity),'" default="',convert2str(self.mixed_layer_capacity),'">','     <section name="materials" />','     <help> mixed layer capacity [W/kg/K] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="thermal_exchange_velocity" type="',class(self.thermal_exchange_velocity),'" default="',convert2str(self.thermal_exchange_velocity),'">','     <section name="materials" />','     <help> thermal exchange velocity [m/s] </help>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="rheology_B" type="',class(self.rheology_B),'" default="',convert2str(self.rheology_B),'">','     <section name="materials" />','     <help> flow law parameter [Pa/s^(1/n)] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="rheology_n" type="',class(self.rheology_n),'" default="',convert2str(self.rheology_n),'">','     <section name="materials" />','     <help> Glens flow law exponent </help>','</parameter>');

			% rheology_law drop-down
			fprintf(fid,'%s\n%s\n%s\n%s\n','<parameter key ="rheology_law" type="alternative" optional="false">','     <section name="materials" />','     <help> law for the temperature dependance of the rheology: "None", "Paterson",  "Arrhenius" or "LliboutryDuval" </help>');
			fprintf(fid,'%s\n','       <option value="None" type="string" default="true"> </option>');
			fprintf(fid,'%s\n','       <option value="Paterson" type="string" default="false"> </option>');
			fprintf(fid,'%s\n','       <option value="Arrhenius" type="string" default="false"> </option>');
			fprintf(fid,'%s\n%s\n','       <option value="LliboutryDuval" type="string" default="false"> </option>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="lithosphere_shear_modulus" type="',class(self.lithosphere_shear_modulus),'" default="',convert2str(self.lithosphere_shear_modulus),'">','     <section name="materials" />','     <help> Lithosphere shear modulus [Pa] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="lithosphere_density" type="',class(self.lithosphere_density),'" default="',convert2str(self.lithosphere_density),'">','     <section name="materials" />','     <help> Lithosphere density [g/cm^-3] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="mantle_shear_modulus" type="',class(self.mantle_shear_modulus),'" default="',convert2str(self.mantle_shear_modulus),'">','     <section name="materials" />','     <help> Mantle shear modulus [Pa] </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','<parameter key ="mantle_density" type="',class(self.mantle_density),'" default="',convert2str(self.mantle_density),'">','     <section name="materials" />','     <help> Mantle density [g/cm^-3] </help>','</parameter>');


		end % }}}
		function self = extrude(self,md) % {{{
			self.rheology_B=project3d(md,'vector',self.rheology_B,'type','node');
			self.rheology_n=project3d(md,'vector',self.rheology_n,'type','element');
		end % }}}
		function self = matice(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					inputstruct=varargin{1};
					list1 = properties('matice');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							self.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%ice density (kg/m^3)
			self.rho_ice=917.;

			%ocean water density (kg/m^3)
			self.rho_water=1023.;

			%fresh water density (kg/m^3)
			self.rho_freshwater=1000.;

			%water viscosity (N.s/m^2)
			self.mu_water=0.001787;  

			%ice heat capacity cp (J/kg/K)
			self.heatcapacity=2093.;

			%ice latent heat of fusion L (J/kg)
			self.latentheat=3.34*10^5;

			%ice thermal conductivity (W/m/K)
			self.thermalconductivity=2.4;
			
			%wet ice thermal conductivity (W/m/K)
			self.temperateiceconductivity=.24;

			%the melting point of ice at 1 atmosphere of pressure in K
			self.meltingpoint=273.15;

			%rate of change of melting point with pressure (K/Pa)
			self.beta=9.8*10^-8;

			%mixed layer (ice-water interface) heat capacity (J/kg/K)
			self.mixed_layer_capacity=3974.;

			%thermal exchange velocity (ice-water interface) (m/s)
			self.thermal_exchange_velocity=1.00*10^-4;

			%Rheology law: what is the temperature dependence of B with T
			%available: none, paterson and arrhenius
			self.rheology_law='Paterson';

			% GIA:
			self.lithosphere_shear_modulus  = 6.7*10^10;  % (Pa)
			self.lithosphere_density        = 3.32;       % (g/cm^-3)
			self.mantle_shear_modulus       = 1.45*10^11; % (Pa)
			self.mantle_density             = 3.34;       % (g/cm^-3)

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			md = checkfield(md,'fieldname','materials.rho_ice','>',0);
			md = checkfield(md,'fieldname','materials.rho_water','>',0);
			md = checkfield(md,'fieldname','materials.rho_freshwater','>',0);
			md = checkfield(md,'fieldname','materials.mu_water','>',0);
			md = checkfield(md,'fieldname','materials.rheology_B','>',0,'timeseries',1,'NaN',1,'Inf',1);
			md = checkfield(md,'fieldname','materials.rheology_n','>',0,'size',[md.mesh.numberofelements 1]);
			md = checkfield(md,'fieldname','materials.rheology_law','values',{'None' 'Cuffey' 'Paterson' 'Arrhenius' 'LliboutryDuval'});

			if ismember(GiaAnalysisEnum(),analyses),
				md = checkfield(md,'fieldname','materials.lithosphere_shear_modulus','>',0,'numel',1);
				md = checkfield(md,'fieldname','materials.lithosphere_density','>',0,'numel',1);
				md = checkfield(md,'fieldname','materials.mantle_shear_modulus','>',0,'numel',1);
				md = checkfield(md,'fieldname','materials.mantle_density','>',0,'numel',1);
			end

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Materials:'));

			fielddisplay(self,'rho_ice','ice density [kg/m^3]');
			fielddisplay(self,'rho_water','ocean water density [kg/m^3]');
			fielddisplay(self,'rho_freshwater','fresh water density [kg/m^3]');
			fielddisplay(self,'mu_water','water viscosity [N s/m^2]');
			fielddisplay(self,'heatcapacity','heat capacity [J/kg/K]');
			fielddisplay(self,'thermalconductivity',['ice thermal conductivity [W/m/K]']);
			fielddisplay(self,'temperateiceconductivity','temperate ice thermal conductivity [W/m/K]');
			fielddisplay(self,'meltingpoint','melting point of ice at 1atm in K');
			fielddisplay(self,'latentheat','latent heat of fusion [J/kg]');
			fielddisplay(self,'beta','rate of change of melting point with pressure [K/Pa]');
			fielddisplay(self,'mixed_layer_capacity','mixed layer capacity [W/kg/K]');
			fielddisplay(self,'thermal_exchange_velocity','thermal exchange velocity [m/s]');
			fielddisplay(self,'rheology_B','flow law parameter [Pa/s^(1/n)]');
			fielddisplay(self,'rheology_n','Glen''s flow law exponent');
			fielddisplay(self,'rheology_law',['law for the temperature dependance of the rheology: ''None'', ''Cuffey'', ''Paterson'', ''Arrhenius'' or ''LliboutryDuval''']);
			fielddisplay(self,'lithosphere_shear_modulus','Lithosphere shear modulus [Pa]');
			fielddisplay(self,'lithosphere_density','Lithosphere density [g/cm^-3]');
			fielddisplay(self,'mantle_shear_modulus','Mantle shear modulus [Pa]');
			fielddisplay(self,'mantle_density','Mantle density [g/cm^-3]');
		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'enum',MaterialsEnum(),'data',MaticeEnum(),'format','Integer');
			WriteData(fid,'object',self,'class','materials','fieldname','rho_ice','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','rho_water','enum',MaterialsRhoSeawaterEnum(),'format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','rho_freshwater','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','mu_water','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','heatcapacity','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','latentheat','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','thermalconductivity','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','temperateiceconductivity','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','meltingpoint','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','beta','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','mixed_layer_capacity','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','thermal_exchange_velocity','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','rheology_B','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',self,'class','materials','fieldname','rheology_n','format','DoubleMat','mattype',2);
			WriteData(fid,'data',StringToEnum(self.rheology_law),'enum',MaterialsRheologyLawEnum(),'format','Integer');

			WriteData(fid,'object',self,'class','materials','fieldname','lithosphere_shear_modulus','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','lithosphere_density','format','Double','scale',10^3);
			WriteData(fid,'object',self,'class','materials','fieldname','mantle_shear_modulus','format','Double');
			WriteData(fid,'object',self,'class','materials','fieldname','mantle_density','format','Double','scale',10^3);
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.materials.rho_ice'],self.rho_ice);
			writejsdouble(fid,[modelname '.materials.rho_water'],self.rho_water);
			writejsdouble(fid,[modelname '.materials.rho_freshwater'],self.rho_freshwater);
			writejsdouble(fid,[modelname '.materials.mu_water'],self.mu_water);
			writejsdouble(fid,[modelname '.materials.heatcapacity'],self.heatcapacity);
			writejsdouble(fid,[modelname '.materials.latentheat'],self.latentheat);
			writejsdouble(fid,[modelname '.materials.thermalconductivity'],self.thermalconductivity);
			writejsdouble(fid,[modelname '.materials.temperateiceconductivity'],self.temperateiceconductivity);
			writejsdouble(fid,[modelname '.materials.meltingpoint'],self.meltingpoint);
			writejsdouble(fid,[modelname '.materials.beta'],self.beta);
			writejsdouble(fid,[modelname '.materials.mixed_layer_capacity'],self.mixed_layer_capacity);
			writejsdouble(fid,[modelname '.materials.thermal_exchange_velocity'],self.thermal_exchange_velocity);
			writejsdouble(fid,[modelname '.materials.mixed_layer_capacity'],self.mixed_layer_capacity);
			writejs1Darray(fid,[modelname '.materials.rheology_B'],self.rheology_B);
			writejs1Darray(fid,[modelname '.materials.rheology_n'],self.rheology_n);
			writejsstring(fid,[modelname '.materials.rheology_law'],self.rheology_law);
			writejsdouble(fid,[modelname '.materials.lithosphere_shear_modulus'],self.lithosphere_shear_modulus);
			writejsdouble(fid,[modelname '.materials.lithosphere_density'],self.lithosphere_density);
			writejsdouble(fid,[modelname '.materials.mantle_shear_modulus'],self.mantle_shear_modulus);
			writejsdouble(fid,[modelname '.materials.mantle_density'],self.mantle_density);

		end % }}}
	end
end
