%TRANSIENT class definition
%
%   Usage:
%      transient=transient();

classdef transient
	properties (SetAccess=public) 
		issmb             = 0;
		ismasstransport   = 0;
		isstressbalance   = 0;
		isthermal         = 0;
		isgroundingline   = 0;
		isgia             = 0;
		isdamageevolution = 0;
		islevelset        = 0;
		iscalving         = 0;
		ishydrology       = 0;
		requested_outputs = {};
	end
	methods
        function createxml(self,fid) % {{{
            fprintf(fid, '<!-- transient -->\n');            
                    
            % transient solution parameters
            fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="transient solution parameters">','<section name="transient" />');                    
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="issmb" type="',class(self.issmb),'" default="',convert2str(self.issmb),'">','     <section name="transient" />','     <help> indicates whether a masstransport solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="ismasstransport" type="',class(self.ismasstransport),'" default="',convert2str(self.ismasstransport),'">','     <section name="transient" />','     <help> indicates whether a masstransport solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isstressbalance" type="',class(self.isstressbalance),'" default="',convert2str(self.isstressbalance),'">','     <section name="transient" />','     <help> indicates whether a stressbalance solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isthermal" type="',class(self.isthermal),'" default="',convert2str(self.isthermal),'">','     <section name="transient" />','     <help> indicates whether a thermal solution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isgroundingline" type="',class(self.isgroundingline),'" default="',convert2str(self.isgroundingline),'">','     <section name="transient" />','     <help> indicates whether a groundingline migration is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isgia" type="',class(self.isgia),'" default="',convert2str(self.isgia),'">','     <section name="transient" />','     <help> indicates whether a postglacial rebound model is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isdamageevolution" type="',class(self.isdamageevolution),'" default="',convert2str(self.isdamageevolution),'">','     <section name="transient" />','     <help> indicates whether damage evolution is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="islevelset" type="',class(self.islevelset),'" default="',convert2str(self.islevelset),'">','     <section name="transient" />','     <help> LEVEL SET DESCRIPTION...  </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="iscalving" type="',class(self.iscalving),'" default="',convert2str(self.iscalving),'">','     <section name="transient" />','     <help> indicates whether calving is used in the transient </help>','  </parameter>');
                fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="requested_outputs" type="',class(self.requested_outputs),'" default="',convert2str(self.requested_outputs),'">','     <section name="transient" />','     <help> list of additional outputs requested </help>','  </parameter>');
            fprintf(fid,'%s\n%s\n','</frame>');    
        
        end % }}}
		function self = transient(varargin) % {{{
			switch nargin
				case 0
					self = setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = deactivateall(self) % {{{

			%full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
			self.issmb           = 0;
			self.ismasstransport = 0;
			self.isstressbalance = 0;
			self.isthermal       = 0;
			self.isgroundingline = 0;
			self.isgia           = 0;
			self.isdamageevolution = 0;
			self.islevelset      = 0;
			self.iscalving       =0;
			self.ishydrology     = 0;

			%default output
			self.requested_outputs={};
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%full analysis: Stressbalance, Masstransport and Thermal but no groundingline migration for now
			self.issmb           = 1;
			self.ismasstransport = 1;
			self.isstressbalance = 1;
			self.isthermal       = 1;
			self.isgroundingline = 0;
			self.isgia           = 0;
			self.isdamageevolution = 0;
			self.islevelset      = 0;
			self.iscalving       = 0;
			self.ishydrology     = 0;

			%default output
			self.requested_outputs={'default'};
		end % }}}
		function list = defaultoutputs(self,md) % {{{
			if(self.issmb)
				list = {'SmbMassBalance'};
			else
				list = {};
			end
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if solution~=TransientSolutionEnum(), return; end

			md = checkfield(md,'fieldname','transient.issmb','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.ismasstransport','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isstressbalance','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isthermal','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgroundingline','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isgia','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.isdamageevolution','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.islevelset','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.iscalving','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.ishydrology','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','transient.requested_outputs','stringrow',1);

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   transient solution parameters:'));

			fielddisplay(self,'issmb','indicates whether a surface mass balance solution is used in the transient');
			fielddisplay(self,'ismasstransport','indicates whether a masstransport solution is used in the transient');
			fielddisplay(self,'isstressbalance','indicates whether a stressbalance solution is used in the transient');
			fielddisplay(self,'isthermal','indicates whether a thermal solution is used in the transient');
			fielddisplay(self,'isgroundingline','indicates whether a groundingline migration is used in the transient');
			fielddisplay(self,'isgia','indicates whether a postglacial rebound model is used in the transient');
			fielddisplay(self,'isdamageevolution','indicates whether damage evolution is used in the transient');
			fielddisplay(self,'islevelset','LEVEL SET DESCRIPTION...');
			fielddisplay(self,'iscalving','indicates whether calving is used in the transient');
			fielddisplay(self,'ishydrology','indicates whether an hydrology model is used');
			fielddisplay(self,'requested_outputs','list of additional outputs requested');

		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'object',self,'fieldname','issmb','format','Boolean');
			WriteData(fid,'object',self,'fieldname','ismasstransport','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isstressbalance','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isthermal','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isgroundingline','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isgia','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isdamageevolution','format','Boolean');
			WriteData(fid,'object',self,'fieldname','ishydrology','format','Boolean');
			WriteData(fid,'object',self,'fieldname','islevelset','format','Boolean');
			WriteData(fid,'object',self,'fieldname','iscalving','format','Boolean');

			%process requested outputs
			outputs = self.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(self,md)]; %add defaults
			end
			WriteData(fid,'data',outputs,'enum',TransientRequestedOutputsEnum,'format','StringArray');
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.trans.issmb'],self.issmb);
			writejsdouble(fid,[modelname '.trans.ismasstransport'],self.ismasstransport);
			writejsdouble(fid,[modelname '.trans.isstressbalance'],self.isstressbalance);
			writejsdouble(fid,[modelname '.trans.isthermal'],self.isthermal);
			writejsdouble(fid,[modelname '.trans.isgroundingline'],self.isgroundingline);
			writejsdouble(fid,[modelname '.trans.isgia'],self.isgia);
			writejsdouble(fid,[modelname '.trans.isdamageevolution'],self.isdamageevolution);
			writejsdouble(fid,[modelname '.trans.islevelset'],self.islevelset);
			writejsdouble(fid,[modelname '.trans.iscalving'],self.iscalving);
			writejsdouble(fid,[modelname '.trans.ishydrology'],self.ishydrology);
			writejscellstring(fid,[modelname '.trans.requested_outputs'],self.requested_outputs);

		end % }}}
	end
end
