%THERMAL class definition
%
%   Usage:
%      thermal=thermal();

classdef thermal
	properties (SetAccess=public) 
		spctemperature    = NaN;
		penalty_threshold = 0;
		stabilization     = 0;
		reltol				= 0;
		maxiter           = 0;
		penalty_lock      = 0;
		penalty_factor    = 0;
		isenthalpy        = 0;
		isdynamicbasalspc = 0;
		requested_outputs = {};
	end
	methods
		function createxml(self,fid) % {{{
			fprintf(fid, '<!-- thermal -->\n');            

			% thermal solution parameters
			fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="Thermal solution parameters">','<section name="thermal" />');                    
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="spctemperature" type="',class(self.spctemperature),'" default="',convert2str(self.spctemperature),'">','     <section name="thermal" />','     <help> temperature constraints (NaN means no constraint) [K] </help>','  </parameter>');

			% penalty_threshold drop-down (0, 1, or 2)
			fprintf(fid,'%s\n%s\n%s\n%s\n','  <parameter key ="penalty_threshold" type="alternative"  optional="false">','     <section name="thermal" />','     <help> 0: no, 1: artificial_diffusivity, 2: SUPG </help>');
			fprintf(fid,'%s\n','       <option value="0" type="string" default="true"> </option>');
			fprintf(fid,'%s\n','       <option value="1" type="string" default="false"> </option>');
			fprintf(fid,'%s\n%s\n','       <option value="2" type="string" default="false"> </option>','</parameter>');

			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="stabilization" type="',class(self.stabilization),'" default="',convert2str(self.stabilization),'">','     <section name="thermal" />','     <help> maximum number of non linear iterations </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="reltol" type="',class(self.reltol),'" default="',convert2str(self.reltol),'">','     <section name="steadystate" />','     <help> relative tolerance criterion [K] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="maxiter" type="',class(self.maxiter),'" default="',convert2str(self.maxiter),'">','     <section name="thermal" />','     <help> stabilize unstable thermal constraints that keep zigzagging after n iteration (default is 0, no stabilization) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="penalty_lock" type="',class(self.penalty_lock),'" default="',convert2str(self.penalty_lock),'">','     <section name="thermal" />','     <help> threshold to declare convergence of thermal solution (default is 0)  </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="penalty_factor" type="',class(self.penalty_factor),'" default="',convert2str(self.penalty_factor),'">','     <section name="thermal" />','     <help> scaling exponent (default is 3) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isenthalpy" type="',class(self.isenthalpy),'" default="',convert2str(self.isenthalpy),'">','     <section name="thermal" />','     <help> use an enthalpy formulation to include temperate ice (default is 0) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="isdynamicbasalspc" type="',class(self.isdynamicbasalspc),'" default="',convert2str(self.isdynamicbasalspc),'">','     <section name="thermal" />','     <help> enable dynamic setting of basal forcing. recommended for enthalpy formulation (default is 0)  </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="requested_outputs" type="',class(self.requested_outputs),'" default="',convert2str(self.requested_outputs),'">','     <section name="thermal" />','     <help> additional outputs requested </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');    

		end % }}}
		function self = extrude(self,md) % {{{
			self.spctemperature=project3d(md,'vector',self.spctemperature,'type','node','layer',md.mesh.numberoflayers,'padding',NaN);
			if (length(md.initialization.temperature)==md.mesh.numberofvertices),
				self.spctemperature=NaN(md.mesh.numberofvertices,1);
				pos=find(md.mesh.vertexonsurface);
				self.spctemperature(pos)=md.initialization.temperature(pos); %impose observed temperature on surface
			end
		end % }}}
		function self = thermal(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function list = defaultoutputs(self,md) % {{{

			if self.isenthalpy,
				list = {'Enthalpy','Temperature','Waterfraction','Watercolumn','BasalforcingsGroundediceMeltingRate'};
			else
				list = {'Temperature','BasalforcingsGroundediceMeltingRate'};
			end

		end % }}}
		function self = setdefaultparameters(self) % {{{

			%Number of unstable constraints acceptable
			self.penalty_threshold=0;

			%Type of stabilization used
			self.stabilization=1;

			%Relative tolerance for the enthalpy convergence
			self.reltol=0.01;

			%Maximum number of iterations
			self.maxiter=100;

			%factor used to compute the values of the penalties: kappa=max(stiffness matrix)*10^penalty_factor
			self.penalty_factor=3;

			%Should we use cold ice (default) or enthalpy formulation
			self.isenthalpy=0;

			%will basal boundary conditions be set dynamically
			self.isdynamicbasalspc=0;

			%default output
			self.requested_outputs={'default'};
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			%Early return
			if (~ismember(ThermalAnalysisEnum(),analyses) & ~ismember(EnthalpyAnalysisEnum(),analyses)) | (solution==TransientSolutionEnum() & md.transient.isthermal==0), return; end

			md = checkfield(md,'fieldname','thermal.stabilization','numel',[1],'values',[0 1 2]);
			md = checkfield(md,'fieldname','thermal.spctemperature','timeseries',1);
			if (ismember(EnthalpyAnalysisEnum(),analyses) & md.thermal.isenthalpy & dimension(md.mesh)==3),
				pos=find(md.thermal.spctemperature(1:md.mesh.numberofvertices,:)~=NaN);
				replicate=repmat(md.geometry.surface-md.mesh.z,1,size(md.thermal.spctemperature,2));
				md = checkfield(md,'fieldname','thermal.spctemperature(find(md.thermal.spctemperature(1:md.mesh.numberofvertices,:)~=NaN))','<=',md.materials.meltingpoint-md.materials.beta*md.materials.rho_ice*md.constants.g*replicate(pos)+10^-5,'message','spctemperature should be less or equal than the adjusted melting point');
				md = checkfield(md,'fieldname','thermal.isenthalpy','numel',[1],'values',[0 1]);
				md = checkfield(md,'fieldname','thermal.isdynamicbasalspc','numel', [1],'values',[0 1]);
				if(md.thermal.isenthalpy)
					if isnan(md.stressbalance.reltol),
						md = checkmessage(md,['for a steadystate computation, thermal.reltol (relative convergence criterion) must be defined!']);
					end 
					md = checkfield(md,'fieldname','thermal.reltol','>',0.,'message','reltol must be larger than zero');
				end
	    end

		 md = checkfield(md,'fieldname','thermal.requested_outputs','stringrow',1);
    end % }}} 
		function disp(self) % {{{
			disp(sprintf('   Thermal solution parameters:'));

			fielddisplay(self,'spctemperature','temperature constraints (NaN means no constraint) [K]');
			fielddisplay(self,'stabilization','0: no, 1: artificial_diffusivity, 2: SUPG');
			fielddisplay(self,'reltol','relative tolerance convergence criterion for enthalpy');
			fielddisplay(self,'maxiter','maximum number of non linear iterations');
			fielddisplay(self,'penalty_lock','stabilize unstable thermal constraints that keep zigzagging after n iteration (default is 0, no stabilization)');
			fielddisplay(self,'penalty_threshold','threshold to declare convergence of thermal solution (default is 0)');
			fielddisplay(self,'penalty_factor','scaling exponent (default is 3)');
			fielddisplay(self,'isenthalpy','use an enthalpy formulation to include temperate ice (default is 0)');
			fielddisplay(self,'isdynamicbasalspc',['enable dynamic setting of basal forcing. required for enthalpy formulation (default is 0)']);
			fielddisplay(self,'requested_outputs','additional outputs requested');

		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'object',self,'fieldname','spctemperature','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
			WriteData(fid,'object',self,'fieldname','penalty_threshold','format','Integer');
			WriteData(fid,'object',self,'fieldname','stabilization','format','Integer');
			WriteData(fid,'object',self,'fieldname','reltol','format','Double');
			WriteData(fid,'object',self,'fieldname','maxiter','format','Integer');
			WriteData(fid,'object',self,'fieldname','penalty_lock','format','Integer');
			WriteData(fid,'object',self,'fieldname','penalty_factor','format','Double');
			WriteData(fid,'object',self,'fieldname','isenthalpy','format','Boolean');
			WriteData(fid,'object',self,'fieldname','isdynamicbasalspc','format','Boolean');

			%process requested outputs
			outputs = self.requested_outputs;
			pos  = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(self,md)]; %add defaults
			end
			WriteData(fid,'data',outputs,'enum',ThermalRequestedOutputsEnum(),'format','StringArray');
        	end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejs1Darray(fid,[modelname '.thermal.spctemperature'],self.spctemperature);
			writejsdouble(fid,[modelname '.thermal.penalty_threshold'],self.penalty_threshold);
			writejsdouble(fid,[modelname '.thermal.stabilization'],self.stabilization);
			writejsdouble(fid,[modelname '.thermal.reltol'],self.reltol);
			writejsdouble(fid,[modelname '.thermal.maxiter'],self.maxiter);
			writejsdouble(fid,[modelname '.thermal.penalty_lock'],self.penalty_lock);
			writejsdouble(fid,[modelname '.thermal.penalty_factor'],self.penalty_factor);
			writejsdouble(fid,[modelname '.thermal.isenthalpy'],self.isenthalpy);
			writejsdouble(fid,[modelname '.thermal.isdynamicbasalspc'],self.isdynamicbasalspc);
			writejscellstring(fid,[modelname '.thermal.requested_outputs'],self.requested_outputs);

		end % }}}
	end
end
