%TIMESTEPPING Class definition
%
%   Usage:
%      timestepping=timestepping();

classdef timestepping
	properties (SetAccess=public) 
		start_time      = 0.;
		final_time      = 0.;
		time_step       = 0.;
		time_adapt      = 0;
		cfl_coefficient = 0.;
		interp_forcings = 1;
	end
	methods
       function createxml(self,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- Timestepping -->');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="start_time" type="',              class(self.start_time),'" default="',                  convert2str(self.start_time),'">',              '     <section name="timestepping" />','     <help> simulation starting time [yr] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="final_time" type="',            class(self.final_time),'" default="',                convert2str(self.final_time),'">',   '     <section name="timestepping" />','     <help> final time to stop the simulation [yr] </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',        '<parameter key ="time_step" type="',      class(self.time_step),'" default="',          convert2str(self.time_step),'">',            '     <section name="timestepping" />','     <help> length of time steps [yr] </help>','</parameter>');
            % time_adapt 0 or 1 drop down
            fprintf(fid,'%s\n%s\n%s\n%s\n','<parameter key ="time_adapt" type="alternative" optional="false">','     <section name="timestepping" />','     <help> use cfl condition to define time step ? (0 or 1)  </help>');
            fprintf(fid, '%s\n',   '       <option value="0" type="string" default="true"></option>');
            fprintf(fid, '%s\n%s\n',   '       <option value="1" type="string" default="false"></option>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="cfl_coefficient" type="',       class(self.cfl_coefficient),'" default="',              convert2str(self.cfl_coefficient),'">',              '     <section name="timestepping" />','     <help> coefficient applied to cfl condition </help>','</parameter>');
            
        end % }}}
        
		function self = timestepping(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%time between 2 time steps
			self.time_step=1./2.;

			%final time
			self.final_time=10.*self.time_step;

			%time adaptation? 
			self.time_adapt=0;
			self.cfl_coefficient=0.5;

			%should we interpolate forcings between timesteps?
			self.interp_forcings=1;
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','timestepping.start_time','numel',[1],'NaN',1);
			md = checkfield(md,'fieldname','timestepping.final_time','numel',[1],'NaN',1);
			md = checkfield(md,'fieldname','timestepping.time_step','numel',[1],'>=',0,'NaN',1);
			md = checkfield(md,'fieldname','timestepping.time_adapt','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','timestepping.cfl_coefficient','numel',[1],'>',0,'<=',1);
			md = checkfield(md,'fieldname','timestepping.interp_forcings','numel',[1],'values',[0 1]);
			if self.final_time-self.start_time<0,
				md = checkmessage(md,'timestepping.final_time should be larger than timestepping.start_time');
			end 
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   timestepping parameters:'));

			unit = 'yr';
			fielddisplay(self,'start_time',['simulation starting time [' unit ']']);
			fielddisplay(self,'final_time',['final time to stop the simulation [' unit ']']);
			fielddisplay(self,'time_step',['length of time steps [' unit ']']);
			fielddisplay(self,'time_adapt','use cfl condition to define time step ? (0 or 1) ');
			fielddisplay(self,'cfl_coefficient','coefficient applied to cfl condition');
			fielddisplay(self,'interp_forcings','interpolate in time between requested forcing values ? (0 or 1)');

		end % }}}
		function marshall(self,md,fid) % {{{

			scale = 365.0*24.0*3600.0;
			WriteData(fid,'object',self,'fieldname','start_time','format','Double','scale',scale);
			WriteData(fid,'object',self,'fieldname','final_time','format','Double','scale',scale);
			WriteData(fid,'object',self,'fieldname','time_step','format','Double','scale',scale);
			WriteData(fid,'object',self,'fieldname','time_adapt','format','Boolean');
			WriteData(fid,'object',self,'fieldname','cfl_coefficient','format','Double');
			WriteData(fid,'object',self,'fieldname','interp_forcings','format','Boolean');
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.timestepping.start_time'],self.start_time);
			writejsdouble(fid,[modelname '.timestepping.final_time'],self.final_time);
			writejsdouble(fid,[modelname '.timestepping.time_step'],self.time_step);
			writejsdouble(fid,[modelname '.timestepping.time_adapt'],self.time_adapt);
			writejsdouble(fid,[modelname '.timestepping.cfl_coefficient'],self.cfl_coefficient);
			writejsdouble(fid,[modelname '.timestepping.interp_forcings'],self.interp_forcings);
			writejsdouble(fid,[modelname '.timestepping.in_years'],self.in_years);

		end % }}}
	end
end
