%SETTINGS class definition
%
%   Usage:
%      settings=settings();

classdef settings
	properties (SetAccess=public) 
		results_on_nodes    = 0;
		io_gather           = 0;
		lowmem              = 0;
		output_frequency    = 0;
		recording_frequency   = 0;
		waitonlock          = 0;
		upload_server       = '';
		upload_path         = '';
		upload_login        = '';
		upload_port         = 0;
		upload_filename     = '';

	end
	methods
        function createxml(self,fid) % {{{
            fprintf(fid, '\n\n');
            fprintf(fid, '%s\n', '<!-- settings -->');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="results_on_nodes" type="',      class(self.results_on_nodes),'" default="',        convert2str(self.results_on_nodes),'">',	'     <section name="settings" />','     <help> results are output for all the nodes of each element </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="io_gather" type="',          class(self.io_gather),'" default="',           convert2str(self.io_gather),'">',     '     <section name="settings" />','     <help> I/O gathering strategy for result outputs (default 1) </help>','</parameter>');
            
            % lowmen drop-down (0 or 1)
            fprintf(fid,'%s\n%s\n%s\n%s\n',	'<parameter key ="lowmem" type="alternative" optional="true">',	'     <section name="settings" />','     <help> is the memory limited ? (0 or 1) </help>');
            fprintf(fid,'%s\n','       <option value="0" type="string" default="true"> </option>');
            fprintf(fid,'%s\n%s\n','       <option value="1" type="string" default="false"> </option>','</parameter>');

            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="output_frequency" type="',          class(self.output_frequency),'" default="',           convert2str(self.output_frequency),'">',	'     <section name="settings" />','     <help> frequency at which results are saved in all solutions with multiple time_steps </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="waitonlock" type="',  	class(self.waitonlock),'" default="',      convert2str(self.waitonlock),'">',     '     <section name="settings" />','     <help> maximum number of minutes to wait for batch results (NaN to deactivate) </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_server" type="',         class(self.upload_server),'" default="',          convert2str(self.upload_server),'">',	'     <section name="settings" />','     <help> server hostname where model should be uploaded </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_path" type="',             class(self.upload_path),'" default="',              convert2str(self.upload_path),'">',     '     <section name="settings" />','     <help> path on server where model should be uploaded </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_login" type="',        class(self.upload_login),'" default="',         convert2str(self.upload_login),'">',	'     <section name="settings" />','     <help> server login </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="upload_port" type="',    	class(self.upload_port),'" default="',      convert2str(self.upload_port),'">',     '     <section name="settings" />','     <help> port login (default is 0) </help>','</parameter>');
            fprintf(fid,'%s%s%s%s%s\n%s\n%s\n',    	'<parameter key ="upload_filename" type="',         class(self.upload_filename),'" default="',          convert2str(self.upload_filename),'">',	'     <section name="settings" />','     <help> unique id generated when uploading the file to server </help>','</parameter>');
             
        end % }}}
		function self = settings(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%are we short in memory ? (0 faster but requires more memory)
			self.lowmem=0;

			%i/o:
			self.io_gather=1;

			%results frequency by default every step
			self.output_frequency=1;

			%checkpoints frequency, by default never: 
			self.recording_frequency=0;

			%this option can be activated to load automatically the results
			%onto the model after a parallel run by waiting for the lock file
			%N minutes that is generated once the solution has converged
			%0 to deactivate
			self.waitonlock=Inf;

			%upload options: 
			upload_port         = 0;

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','settings.results_on_nodes','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.io_gather','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.lowmem','numel',[1],'values',[0 1]);
			md = checkfield(md,'fieldname','settings.output_frequency','numel',[1],'>=',1);
			md = checkfield(md,'fieldname','settings.recording_frequency','numel',[1],'>=',0);
			md = checkfield(md,'fieldname','settings.waitonlock','numel',[1]);

		end % }}}
		function disp(self) % {{{
			disp(sprintf('   general settings parameters:'));

			fielddisplay(self,'results_on_nodes','results are output for all the nodes of each element');
			fielddisplay(self,'io_gather','I/O gathering strategy for result outputs (default 1)');
			fielddisplay(self,'lowmem','is the memory limited ? (0 or 1)');
			fielddisplay(self,'output_frequency','frequency at which results are saved in all solutions with multiple time_steps');
			fielddisplay(self,'recording_frequency','frequency at which the runs are being recorded, allowing for a restart');
			fielddisplay(self,'waitonlock','maximum number of minutes to wait for batch results (NaN to deactivate)');
			fielddisplay(self,'upload_server','server hostname where model should be uploaded');
			fielddisplay(self,'upload_path','path on server where model should be uploaded');
			fielddisplay(self,'upload_login','server login');
			fielddisplay(self,'upload_port','port login (default is 0)');
			fielddisplay(self,'upload_filename','unique id generated when uploading the file to server');

		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'object',self,'fieldname','results_on_nodes','format','Boolean');
			WriteData(fid,'object',self,'fieldname','io_gather','format','Boolean');
			WriteData(fid,'object',self,'fieldname','lowmem','format','Boolean');
			WriteData(fid,'object',self,'fieldname','output_frequency','format','Integer');
			WriteData(fid,'object',self,'fieldname','recording_frequency','format','Integer');
			if self.waitonlock>0,
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',true,'format','Boolean');
			else
				WriteData(fid,'enum',SettingsWaitonlockEnum(),'data',false,'format','Boolean');
			end
		end % }}}
		function savemodeljs(self,fid,modelname) % {{{
		
			writejsdouble(fid,[modelname '.settings.results_on_nodes'],self.results_on_nodes);
			writejsdouble(fid,[modelname '.settings.io_gather'],self.io_gather);
			writejsdouble(fid,[modelname '.settings.lowmem'],self.lowmem);
			writejsdouble(fid,[modelname '.settings.output_frequency'],self.output_frequency);
			writejsdouble(fid,[modelname '.settings.recording_frequency'],self.recording_frequency);
			writejsdouble(fid,[modelname '.settings.waitonlock'],self.waitonlock);
			writejsstring(fid,[modelname '.settings.upload_server'],self.upload_server);
			writejsstring(fid,[modelname '.settings.upload_path'],self.upload_path);
			writejsstring(fid,[modelname '.settings.upload_login'],self.upload_login);
			writejsdouble(fid,[modelname '.settings.upload_port'],self.upload_port);
			writejsstring(fid,[modelname '.settings.upload_filename'],self.upload_filename);

		end % }}}
	end
end
