/*This is where we have all our webgl relevant functionality for the plotting routines: */
//{{{ GL Initialization
function initWebGL(canvas,options) { //{{{
	gl = null;

	try {
		// Try to grab the standard context. If it fails, fallback to experimental.
		gl = canvas.getContext("webgl") || canvas.getContext("experimental-webgl");
	}
	catch(e) {}

	// If we don't have a GL context, give up now
	if (!gl) {
		alert("Unable to initialize WebGL. Your browser may not support it.");
		gl = null;
	}

	// Enable depth testing
	gl.enable(gl.DEPTH_TEST);
	// Near things obscure far things
	gl.depthFunc(gl.LEQUAL);
	// Clear the color as well as the depth buffer.
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

	// Allocate arrays equal to maximium number of attributes used by any one shader
	gl.enableVertexAttribArray(0);
	gl.enableVertexAttribArray(1);

	// Load shaders and store them in gl object
	gl.shaders = loadShaders(gl);

	// Add context state variables
	canvas.zoomFactor = options.getfieldvalue('zoomfactor',-.52);
	canvas.zoomFactorMax = options.getfieldvalue('zoomfactor',-.52);
	canvas.cameraMatrix = mat4.create();

	// Add event listeners for canvas
	if (canvas.addEventListener) {
		// IE9, Chrome, Safari, Opera
		canvas.addEventListener("mousewheel", function (e) {MouseWheelHandler(e,canvas)}, false);
		// Firefox
		canvas.addEventListener("DOMMouseScroll", function (e) {MouseWheelHandler(e,canvas)}, false);
		// Mobile
		canvas.addEventListener("gesturechange", MouseWheelHandler, false);
	}

	return gl;
} //}}}
function initBuffers(gl,arrays) { //{{{	
	var bufferArray = [];
	for (var i = 0; i < arrays.length; i++) {
		bufferArray[i] = gl.createBuffer();	
		bufferArray[i].itemSize = arrays[i].itemSize;
		bufferArray[i].numItems = arrays[i].length/bufferArray[i].itemSize;
		
		if (bufferArray[i].itemSize > 1) {
			gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(arrays[i]), gl.STATIC_DRAW);
		}
		else {
			//TODO: identify index buffers uniquely (by name)
			gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, bufferArray[i]);
			gl.bufferData(gl.ELEMENT_ARRAY_BUFFER, new Uint16Array(arrays[i]), gl.STATIC_DRAW);
		}
	}	
	return bufferArray;
} //}}}
function Node(gl,options) { //{{{
	
	var node;

	node= {buffers:[],
		shader:gl.shaders["colored"]["program"],
		draw:null,
		hideOcean:false,
		level:0,
		useIndexBuffer:true,
		useOrthographic:false,
		transparency:1.0,
		disableDepthTest:false, 
		enableCullFace:false,
		cullFace:gl.FRONT,
		drawMode:gl.TRIANGLES,
		texture:null,
		translation:vec3.create(),
		rotation:vec3.create(),
		scale:vec3.fromValues(1, 1, 1),
		modelMatrix:mat4.create(),
		shaderName:"colored",
	};
	
	return node;

} //}}}
function recalculateModelMatrix(node) { //{{{
	var modelMatrix = mat4.create();

	var scaleMatrix = mat4.create();
	mat4.scale(scaleMatrix, scaleMatrix, node["scale"]);
	mat4.multiply(modelMatrix, scaleMatrix, modelMatrix);

	var zRotationMatrix = mat4.create();	
	mat4.rotate(zRotationMatrix, zRotationMatrix, node["rotation"][2] * Math.PI/180.0, [0.0, 0.0, 1.0]);
	mat4.multiply(modelMatrix, zRotationMatrix, modelMatrix);
	var yRotationMatrix = mat4.create();	
	mat4.rotate(yRotationMatrix, yRotationMatrix, node["rotation"][1] * Math.PI/180.0, [0.0, 1.0, 0.0]);
	mat4.multiply(modelMatrix, yRotationMatrix, modelMatrix);
	var xRotationMatrix = mat4.create();	
	mat4.rotate(xRotationMatrix, xRotationMatrix, node["rotation"][0] * Math.PI/180.0, [1.0, 0.0, 0.0]);
	mat4.multiply(modelMatrix, xRotationMatrix, modelMatrix);

	var translationMatrix = mat4.create();
	mat4.translate(translationMatrix, translationMatrix, node["translation"]); //relative translation
	mat4.multiply(modelMatrix, translationMatrix, modelMatrix);

	return modelMatrix;
} //}}}
function rgb(value, min, max) { //{{{
	
	var normalizedValue;
					
	colorbar=colorbars["rainbow"];

	value = clamp(value, min, max);
	if((max-min)!=0) normalizedValue = (value - min) / (max - min);
	else normalizedValue = value;

	var index = clamp(Math.round(normalizedValue * colorbar.length), 0, colorbar.length - 1);
	return colorbar[index];
} //}}}
function clamp(value, min, max) { //{{{
	return Math.max(min, Math.min(value, max));
} //}}}
//}}}
//{{{ Shader Loading
function loadShaders(gl) { //{{{
	// TODO: Subsitute shaders["colored"] with shaderColored
	shaderName = "colored";
	shaders = {};
	shaders[shaderName] = {loaded:false, vsh:{}, fsh:{}};
	shaders["colored"]["vsh"]["string"] = `
		attribute vec3 aVertexPosition;
		attribute vec4 aVertexColor;

		uniform mat4 uMVPMatrix;
		uniform float uAlpha;

		varying vec4 vColor;

		void main(void) {
			gl_Position = uMVPMatrix * vec4(aVertexPosition.xyz, 1.0);
			vColor = vec4(aVertexColor.xyz, uAlpha);
		}
	`;
	shaders["colored"]["fsh"]["string"] = `
		precision mediump float;

		varying vec4 vColor;

		void main(void) {
			gl_FragColor = vColor;
		}
	`;

	shaders[shaderName]["vsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["vsh"]["string"], "vsh");
	shaders[shaderName]["fsh"]["shader"] = getShaderByString(gl, shaders[shaderName]["fsh"]["string"], "fsh");

	shaders[shaderName]["program"] = gl.createProgram();
	gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["vsh"]["shader"]);
	gl.attachShader(shaders[shaderName]["program"], shaders[shaderName]["fsh"]["shader"]);
	gl.linkProgram(shaders[shaderName]["program"]);

	if (!gl.getProgramParameter(shaders[shaderName]["program"], gl.LINK_STATUS)) {
		alert("Could not initialise shaders");
	}

	var vshStringArray = shaders[shaderName]["vsh"]["string"].split("\n");
	var fshStringArray = shaders[shaderName]["fsh"]["string"].split("\n");
	var line = "";
	var property = "";
	for (var i = 0; i < vshStringArray.length; i++) {
		line = vshStringArray[i];
		if (line.search("attribute") != -1) {
			property = nameFromLine(line);
			shaders[shaderName]["program"][property] = gl.getAttribLocation(shaders[shaderName]["program"], property);
		}
		else if (line.search("uniform") != -1) {
			property = nameFromLine(line);
			shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
		}
		else if (line.search("void main") != -1) {
			break;
		}
	}
	for (var i = 0; i < fshStringArray.length; i++) {
		line = fshStringArray[i];
		if (line.search("uniform") != -1) {
			property = nameFromLine(line);
			shaders[shaderName]["program"][property] = gl.getUniformLocation(shaders[shaderName]["program"], property);
		}
		else if (line.search("void main") != -1) {
			break;
		}
	}
	shaders[shaderName]["loaded"] = true;
	return shaders;
} //}}}
function getShaderByString(gl,str,type) { //{{{
	var shader;
	if (type == "fsh") {
		shader = gl.createShader(gl.FRAGMENT_SHADER);
	}
	else if (type == "vsh") {
		shader = gl.createShader(gl.VERTEX_SHADER);
	}
	else {
		return null;
	}
	
	gl.shaderSource(shader, str);
	gl.compileShader(shader);

	if (!gl.getShaderParameter(shader, gl.COMPILE_STATUS)) {	
		alert(gl.getShaderInfoLog(shader));
		return null;
	}

	return shader;
} //}}}
function nameFromLine(line) { //{{{
	//returns lowerCamelCase property name from shader line
	var fullName = line.split(" ")[2];
	return fullName.slice(0, fullName.search(";"));
} //}}}
//}}}
//{{{ Interface Functions
function MouseWheelHandler(e,canvas) { //{{{
	// prevent scrolling when over canvas
	e.preventDefault();
	var e = window.event || e; // old IE support
	var delta = 1/10 * clamp((e.scale || e.wheelDelta || -e.detail), -1, 1);
	canvas.zoomFactor = clamp(canvas.zoomFactor + -delta * canvas.zoomFactor,canvas.zoomFactorMax,0);
} //}}}
//}}}
//{{{ Drawing Functions
function updateCameraMatrix(canvas) { //{{{
       	//Update view matrix and multiply with projection matrix to get the view-projection (camera) matrix.
	var vMatrix = mat4.create();
	var pMatrix = mat4.create();

	mat4.perspective(pMatrix, 90 * Math.PI / 180, canvas.width / canvas.height, 0.001, 10000.0);

	//Apply screenspace relative translation
	var translateMatrix = mat4.create();
	mat4.translate(translateMatrix, translateMatrix, [0.0, 0.0, canvas.zoomFactor]);
	mat4.multiply(vMatrix, translateMatrix, vMatrix);

	//Apply projection matrix to get camera matrix
	mat4.multiply(canvas.cameraMatrix, pMatrix, vMatrix);
//	canvas.cameraMatrix = mat4.create();
}//}}}
function drawSceneGraphNode(gl,canvas,node) { //{{{
	bindAttributes(gl, node["shader"], node["buffers"]);
	var mvpMatrix = mat4.create();
	if (node["useOrthographic"] == true) {
		mat4.ortho(mvpMatrix, -1.0, 1.0, -1.0, 1.0, -1.0, 1.0);
	}
	else {
		mat4.multiply(mvpMatrix, canvas.cameraMatrix, node["modelMatrix"]);
	}
	gl.uniformMatrix4fv(node["shader"]["uMVPMatrix"], false, mvpMatrix);
	gl.uniform1f(node["shader"]["uAlpha"], node["transparency"]);
	if  (node["useIndexBuffer"]) {
		gl.bindBuffer(gl.ELEMENT_ARRAY_BUFFER, node["buffers"][node["buffers"].length - 1]);
		gl.drawElements(node["drawMode"], node["buffers"][node["buffers"].length - 1].numItems, gl.UNSIGNED_SHORT, 0);
	}
	else {
		gl.drawArrays(node["drawMode"], 0, node["buffers"][0].numItems);
	}	
} //}}}
function bindAttributes(gl,shaderProgram,bufferArray) { //{{{
	gl.useProgram(shaderProgram);
	var arrayNumber = 0;
	for (var propertyName in shaderProgram) {
		if (propertyName[0] == "a") {
			if (bufferArray[arrayNumber].itemSize > 1) {
				gl.bindBuffer(gl.ARRAY_BUFFER, bufferArray[arrayNumber]);
				gl.vertexAttribPointer(shaderProgram[propertyName], bufferArray[arrayNumber].itemSize, gl.FLOAT, false, 0, 0);
				arrayNumber++;
			}
		}
	}
} //}}}
function draw(gl,options,canvas,node) { //{{{

	// Set clear color to black, fully opaque
	var backgroundcolor=new RGBColor(options.getfieldvalue('backgroundcolor','white'));
	if(backgroundcolor.ok){
		gl.clearColor(backgroundcolor.r/255.0, backgroundcolor.g/255.0, backgroundcolor.b/255.0, 1.0);
	}
	else throw Error(sprintf("s%s%s\n","initWebGL error message: cound not find out background color for curent canvas ",canvas));
	
	// Clear the color as well as the depth buffer.
	gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);

	window.requestAnimationFrame(function(time) {draw(gl,options,canvas,node)});
	updateCameraMatrix(canvas);
	drawSceneGraphNode(gl, canvas, node);

} //}}}
//}}}
