/*!\file:  issm_dakota.cpp
 * \brief: ISSM DAKOTA main program
 */ 

#include "./issm.h"

/*Dakota includes: */
#include "ParallelLibrary.hpp"
#include "ProblemDescDB.hpp"
#include "LibraryEnvironment.hpp"
#include "DakotaModel.hpp"
#include "DakotaInterface.hpp"

int main(int argc,char **argv){

	bool parallel=true;
	char* dakota_input_file=NULL;

	/*Define MPI_DEBUG in dakota_global_defs.cpp to cause a hold here*/
	Dakota::mpi_debug_hold();
	
	/*Initialize MPI: */
	ISSM_MPI_Init(&argc, &argv); // initialize MPI

	/*Recover file name for dakota input file:*/
	dakota_input_file=xNew<char>((strlen(argv[2])+strlen(argv[3])+strlen(".qmu.in")+1));
	sprintf(dakota_input_file,"%s%s%s",argv[2],argv[3],".qmu.in");

	Cout << "dakota_input_file: " << dakota_input_file << "\n";
	
	/* Parse input and construct Dakota LibraryEnvironment, performing input data checks*/
	Dakota::ProgramOptions opts;
	opts.input_file(dakota_input_file);

	
	/* Defaults constructs the MPIManager, which assumes COMM_WORLD*/
	Dakota::LibraryEnvironment env(opts);

	if (env.mpi_manager().world_rank() == 0)
		Cout << "Library mode 1: run_dakota_parse()\n";
	
	/* get the list of all models matching the specified model, interface, driver:*/
	Dakota::ModelList filt_models = env.filtered_model_list("single", "direct", "matlab");
	if (filt_models.empty()) {
		Cerr << "Error: no parallel interface plugin performed.  Check compatibility "
			<< "between parallel\n       configuration and selected analysis_driver."
			<< std::endl;
		Dakota::abort_handler(-1);
	}
	
	Dakota::ProblemDescDB& problem_db = env.problem_description_db();
	Dakota::ModelLIter ml_iter;
	size_t model_index = problem_db.get_db_model_node(); // for restoration
	for (ml_iter = filt_models.begin(); ml_iter != filt_models.end(); ++ml_iter) {
		// set DB nodes to input specification for this Model
		problem_db.set_db_model_nodes(ml_iter->model_id());

		Dakota::Interface& model_interface = ml_iter->derived_interface();

		// Parallel case: plug in derived Interface object with an analysisComm.
		// Note: retrieval and passing of analysisComm is necessary only if
		// parallel operations will be performed in the derived constructor.

		// retrieve the currently active analysisComm from the Model.  In the most
		// general case, need an array of Comms to cover all Model configurations.
		const MPI_Comm& analysis_comm = ml_iter->analysis_comm();

		// don't increment ref count since no other envelope shares this letter
		model_interface.assign_rep(new
				SIM::IssmParallelDirectApplicInterface(problem_db, analysis_comm,NULL), false);
	}
	problem_db.set_db_model_nodes(model_index);            // restore

	/* Execute the environment:*/
	env.execute();

	/*Finalize MPI:*/
	ISSM_MPI_Finalize();

	/*Return unix success: */
	return 0; 
}
