/*!\file:  IssmParallelDirectApplicInterface.h. This code is only valid for Dakota versions higher than 6!
 *
 * \brief: derived ParallelDirectApplicInterface class declaration and implementation, taylored to ISSM.
 * This class is registered into the interface database of Dakota, and is used to directly call ISSM cores 
 * from Dakota. 
 *
 * This routine helps running ISSM and Dakota in library mode, for Dakota versions that are >=6, and that fully 
 * support parallelism.  The setup is radically different than from version <6! Now, dakota runs the show more. 
 * The reason is that dakota now controls the parallelism in a master/slave setup, and hands over to ISSM a  bunch 
 * of slave communicators, which are then used to run our simulations. Because ISSM is now ESMF compliant, we can 
 * use these communicators to create separate identical FemModel instances on each slave communicator! This allows 
 * us to scale to large jobs (think 1000's of cpus), which we split into multiple sub-slave communicators, which 
 * run the sampling (or forward different, local reliability, optimization you name it) simulations on each slave. 
 * 
 * This is all bootstraped from the main issm_dakota main, (see c/main directory), which is heavily inspired on the
 * main found in the dakota/src/library_mode.cpp code. We also have to create an ISSM code that registers into the 
 * dakota database, which is capable of running ISSM. This is derived from the Dakota class called 
 * ParallelDirectApplicInterface. 
 */ 
#ifndef _ISSMPARALLELDIRECTAPPLICINTERFACE_
#define _ISSMPARALLELDIRECTAPPLICINTERFACE_

/*Issm Configuration: {{{*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif
/*}}}*/

#if defined(_HAVE_DAKOTA_) && _DAKOTA_MAJOR_ >= 6

/*Dakota include files: {{{*/
#include <DirectApplicInterface.hpp>
/*}}}*/

namespace SIM {
	class IssmParallelDirectApplicInterface: public Dakota::DirectApplicInterface{
      
		public:
			/*these fields are used by core solutions: */
			void *femmodel;
			
			/*Constructors/Destructors{{{*/
			IssmParallelDirectApplicInterface(const Dakota::ProblemDescDB& problem_db, const MPI_Comm& analysis_comm, void* in_femmodel):Dakota::DirectApplicInterface(problem_db){

				#ifdef MPI_DEBUG
				  // For testing purposes, output size/rank of the incoming analysis_comm
				  int rank, size;
				  MPI_Comm_rank(analysis_comm, &rank);
				  MPI_Comm_size(analysis_comm, &size);
				  Cout << "In SIM::ParallelDirectApplicInterface ctor, rank = " << rank
					   << " size = " << size << std::endl;
				 #endif // MPI_DEBUG

				femmodel = in_femmodel;
			}
			~IssmParallelDirectApplicInterface(){
			}
			/*}}}*/
		
		protected:
			
			/// execute an analysis code portion of a direct evaluation invocation
			int derived_map_ac(const Dakota::String& driver){/*{{{*/

				#ifdef MPI_DEBUG
					Cout << "analysis server " << analysisServerId << " invoking " << ac_name
						 << " within SIM::ParallelDirectApplicInterface." << std::endl;
				#endif // MPI_DEBUG

				int i;
				IssmDouble* variables=NULL;
				char** variable_descriptors=NULL;
				char*  variable_descriptor=NULL;
				IssmDouble* responses=NULL;

				/*Before launching analysis, we need to transfer the dakota inputs into Issm 
				 *readable variables: */

				/*First, the variables: */
				variables=xNew<IssmDouble>(numACV);
				for(i=0;i<numACV;i++){
					variables[i]=xC[i];
				}
				/*The descriptors: */
				variable_descriptors=xNew<char*>(numACV);
				for(i=0;i<numACV;i++){
					std::string label=xCLabels[i];
					variable_descriptor=xNew<char>(strlen(label.c_str())+1);
					memcpy(variable_descriptor,label.c_str(),(strlen(label.c_str())+1)*sizeof(char));

					variable_descriptors[i]=variable_descriptor;
				}

				/*Initialize responses: */
				responses=xNewZeroInit<IssmDouble>(numFns);

				/*run core solution: */
				//DakotaSpawnCore(responses,numFns, variables,variable_descriptors,numACV,femmodel);

				/*populate responses: */
				for(i=0;i<numFns;i++){
					fnVals[i]=responses[i];
				}

				/*Free ressources:*/
				xDelete<IssmDouble>(variables);
				for(i=0;i<numACV;i++){
					variable_descriptor=variable_descriptors[i];
					xDelete<char>(variable_descriptor);
				}
				xDelete<char*>(variable_descriptors);
				xDelete<IssmDouble>(responses);

				return 0;
			}/*}}}*/

			/// no-op hides base error; job batching occurs within wait_local_evaluations()
			//void derived_map_asynch(const Dakota::ParamResponsePair& pair){};

			/// evaluate the batch of jobs contained in prp_queue
			//void wait_local_evaluations(Dakota::PRPQueue& prp_queue);

			/// invokes wait_local_evaluations() (no special nowait support)
			//void test_local_evaluations(Dakota::PRPQueue& prp_queue) { wait_local_evaluations(prp_queue); };

			/// no-op hides default run-time error checks at DirectApplicInterface level
			//void set_communicators_checks(int max_eval_concurrency){};

		private:
	};
}
/*}}}*/
#endif
#endif
