%DAMAGEICE class definition
%
%   Usage:
%      damage=damage();

classdef damage
	properties (SetAccess=public)  
		%damage 
		isdamage            = 0;
		D                   = NaN;
		law                 = 0;
		spcdamage           = NaN; 
		max_damage          = 0;
	
		%numerical
		stabilization       = 0;
		maxiter             = 0;
		elementinterp       = '';
		
		%general parameters for evolution law: 
		stress_threshold    = 0;
		kappa               = 0;
		c1                  = 0;
		c2                  = 0;
		c3                  = 0;
		c4                  = 0;
		healing             = 0;
		equiv_stress		  = 0;
		requested_outputs   = {};
	end
	methods
		function createxml(self,fid) % {{{
			fprintf(fid, '\n\n');
			fprintf(fid, '%s\n', '<!-- damage -->');
			fprintf(fid, '%s\n', '<!-- Note: this class depends on different input of law -->');

			%fprintf(fid,'%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="law" type="logical"',           '" default="',                num2str(self.law),'">',   '     <section name="damage" />','     <help> damage law (string) from {"undamaged","pralong"} </help>','</parameter>');
			% drop-down 
			fprintf(fid,'%s%s%s%s%s\n\t%s\n','<parameter key ="law" type="','alternative','" optional="','false','">','<section name="damage" />');

			% law = 'undamage'
			fprintf(fid,'\t%s%s%s%s%s\n\t\t%s\n','<option value="undamage" type="','string','" default="','true','">','<help> law = undamage </help>');
			% footer for option
			fprintf(fid,'\t%s\n%s\n','</option>');

			% law = 'pralong'
			fprintf(fid,'\t%s%s%s%s%s\n\t\t%s\n','<option value="pralong" type="','string','" default="','false','">','<help> law = pralong </help>');

			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',   '<parameter key ="stress_threshold" type="',class(self.stress_threshold),'" default="',num2str(self.stress_threshold),'">','<help> damage stress threshold [Pa] </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',	'<parameter key ="c1" type="', class(self.c1),'" default="',   num2str(self.c1),'">',   '<help> damage parameter 1 </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="c2" type="',          class(self.c2),'" default="',            num2str(self.c2),'">','<help> damage parameter 2 </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="c3" type="',          class(self.c3),'" default="',            num2str(self.c3),'">','<help> damage parameter 3 [W/m^2] </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="c4" type="',            class(self.c4),'" default="',              num2str(self.c4),'">','<help> damage parameter 4 </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="healing" type="', class(self.healing),'" default="',   num2str(self.healing),'">','<help> damage healing parameter 1 </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="equiv_stress" type="',          class(self.equiv_stress),'" default="',convert2str(self.equiv_stress),'">','<help> 0: von Mises </help>','</parameter>');
			fprintf(fid,'\t\t%s%s%s%s%s\n\t\t\t%s\n\t\t%s\n',  	'<parameter key ="requested_outputs" type="',          class(self.requested_outputs),'" default="',convert2str(self.requested_outputs),'">','<help> additional outputs requested </help>','</parameter>');


			% footer for option
			fprintf(fid,'\t%s\n%s\n','</option>');


			% footer for drop-down
			fprintf(fid,'\t%s\n%s\n%s','<help> damage law (string) from {"undamaged","pralong"} </help>','</parameter>');


			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="D" type="',              class(self.D),'" default="',                  num2str(self.D),'">',              '     <section name="damage" />','     <help> damage tensor (scalar) </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="law" type="',            class(self.law),'" default="',                num2str(self.law),'">',   '     <section name="damage" />','     <help> damage law (string) from {"undamaged","pralong"} </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',        '<parameter key ="spcdamage" type="',      class(self.spcdamage),'" default="',          num2str(self.spcdamage),'">',            '     <section name="damage" />','     <help> damage constraints (NaN means no constraint) </help>','</parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',        '<parameter key ="max_damage" type="',     class(self.max_damage),'" default="',         num2str(self.max_damage),'">',            '     <section name="damage" />','     <help> maximum possible damage (0&amp;lt;=max_damage&amp;lt;1) </help>','</parameter>');

			% stabilization (0,1, or 2) drop-down
			fprintf(fid,'%s\n%s\n%s\n%s\n',    '<parameter key ="stabilization" type="alternative" optional="false">','     <section name="damage" />','     <help> 0: no, 1: artificial_diffusivity, 2: SUPG </help>');
			fprintf(fid, '%s\n', '       <option value="0" type="string" default="true"></option>');
			fprintf(fid, '%s\n', '       <option value="1" type="string" default="false"></option>');
			fprintf(fid, '%s\n', '       <option value="2" type="string" default="false"></option>');
			fprintf(fid, '%s\n','</parameter>');

			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n',    '<parameter key ="maxiter" type="',             class(self.maxiter),'" default="',   num2str(self.maxiter),'">',   '     <section name="damage" />','     <help> maximum number of non linear iterations </help>','</parameter>');

		end % }}}
		function self = damage(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					inputstruct=varargin{1};
					list1 = properties('damage');
					list2 = fieldnames(inputstruct);
					for i=1:length(list1)
						fieldname = list1{i};
						if ismember(fieldname,list2),
							self.(fieldname) = inputstruct.(fieldname);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = extrude(self,md) % {{{
			self.D=project3d(md,'vector',self.D,'type','node');
			self.spcdamage=project3d(md,'vector',self.spcdamage,'type','node');
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%damage parameters: 
			self.isdamage=0;
			self.D=0;
			self.law=0;
			
			self.max_damage=1-1e-5; %if damage reaches 1, solve becomes singular, as viscosity becomes nil
		
			%Type of stabilization used
			self.stabilization=4;
			
			%Maximum number of iterations
			self.maxiter=100;

			%finite element interpolation
			self.elementinterp='P1';

			%damage evolution parameters 
			self.stress_threshold=1.3e5;
			self.kappa=2.8;
			self.healing=0;
			self.c1=0;
			self.c2=0;
			self.c3=0;
			self.c4=0;
			self.equiv_stress=0;

			 %output default:
			 self.requested_outputs={'default'};

		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{
			
			md = checkfield(md,'fieldname','damage.isdamage','values',[1,0]);
			if self.isdamage,
				md = checkfield(md,'fieldname','damage.law','numel',[1],'values',[0,1,2]);
				md = checkfield(md,'fieldname','damage.D','>=',0,'<=',self.max_damage,'size',[md.mesh.numberofvertices 1]);
				md = checkfield(md,'fieldname','damage.spcdamage','timeseries',1);
				md = checkfield(md,'fieldname','damage.max_damage','<',1,'>=',0);
				md = checkfield(md,'fieldname','damage.stabilization','numel',[1],'values',[0 1 2 4]);
				md = checkfield(md,'fieldname','damage.maxiter','>=0',0);
				md = checkfield(md,'fieldname','damage.elementinterp','values',{'P1','P2'});
				md = checkfield(md,'fieldname','damage.stress_threshold','>=',0);
				md = checkfield(md,'fieldname','damage.kappa','>',1);
				md = checkfield(md,'fieldname','damage.healing','>=',0);
				md = checkfield(md,'fieldname','damage.c1','>=',0);
				md = checkfield(md,'fieldname','damage.c2','>=',0);
				md = checkfield(md,'fieldname','damage.c3','>=',0);
				md = checkfield(md,'fieldname','damage.c4','>=',0);
				md = checkfield(md,'fieldname','damage.equiv_stress','numel',[1],'values',[0 1]);
				md = checkfield(md,'fieldname','damage.requested_outputs','stringrow',1);
			elseif (self.law~=0),
				if (solution==DamageEvolutionSolutionEnum),
					error('Invalid evolution law (md.damage.law) for a damage solution');
				end
			end
		end % }}}
		function list=defaultoutputs(self,md) % {{{

			if strcmp(domaintype(md.mesh),'2Dhorizontal'),
            list = {'DamageDbar'};
         else
            list = {'DamageD'};
			end
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   Damage:\n'));

			fielddisplay(self,'isdamage','is damage mechanics being used? {true,false}');
			if self.isdamage,
				fielddisplay(self,'law','damage law {''0: analytical'',''1: pralong''}');
				fielddisplay(self,'D','damage tensor (scalar)');
				fielddisplay(self,'spcdamage','damage constraints (NaN means no constraint)');
				fielddisplay(self,'max_damage','maximum possible damage (0<=max_damage<1)');
				
				fielddisplay(self,'stabilization','0: no, 1: artificial_diffusivity, 2: SUPG (not working), 4: flux corrected transport');
				fielddisplay(self,'maxiter','maximum number of non linear iterations');
				fielddisplay(self,'elementinterp','interpolation scheme for finite elements {''P1'',''P2''}');
				fielddisplay(self,'stress_threshold','stress threshold for damage initiation [Pa]');
				fielddisplay(self,'kappa','ductility parameter for stress softening and damage');
				fielddisplay(self,'c1','damage parameter 1');
				fielddisplay(self,'c2','damage parameter 2');
				fielddisplay(self,'c3','damage parameter 3');
				fielddisplay(self,'c4','damage parameter 4');
				fielddisplay(self,'healing','damage healing parameter');
				fielddisplay(self,'equiv_stress','0: von Mises, 1: max principal');
				fielddisplay(self,'requested_outputs','additional outputs requested');
			end

		end % }}}
		function marshall(self,md,fid) % {{{
		
			WriteData(fid,'object',self,'fieldname','isdamage','format','Boolean');
			if self.isdamage,
				WriteData(fid,'object',self,'fieldname','law','format','Integer');
				WriteData(fid,'object',self,'fieldname','D','format','DoubleMat','mattype',1);
				WriteData(fid,'object',self,'fieldname','spcdamage','format','DoubleMat','mattype',1,'timeserieslength',md.mesh.numberofvertices+1);
				WriteData(fid,'object',self,'fieldname','max_damage','format','Double');

				WriteData(fid,'object',self,'fieldname','stabilization','format','Integer');
				WriteData(fid,'object',self,'fieldname','maxiter','format','Integer');
				WriteData(fid,'enum',DamageElementinterpEnum(),'data',StringToEnum(self.elementinterp),'format','Integer');
				WriteData(fid,'object',self,'fieldname','stress_threshold','format','Double');
				WriteData(fid,'object',self,'fieldname','kappa','format','Double');
				WriteData(fid,'object',self,'fieldname','c1','format','Double');
				WriteData(fid,'object',self,'fieldname','c2','format','Double');
				WriteData(fid,'object',self,'fieldname','c3','format','Double');
				WriteData(fid,'object',self,'fieldname','c4','format','Double');
				WriteData(fid,'object',self,'fieldname','healing','format','Double');
				WriteData(fid,'object',self,'fieldname','equiv_stress','format','Integer');
			end

			%process requested outputs
			outputs = self.requested_outputs;
			pos = find(ismember(outputs,'default'));
			if ~isempty(pos),
				outputs(pos) = [];                         %remove 'default' from outputs
				outputs      = [outputs defaultoutputs(self,md)]; %add defaults
			end
			if self.isdamage,
				WriteData(fid,'data',outputs,'enum',DamageEvolutionRequestedOutputsEnum,'format','StringArray');
			end

		end % }}}
	end
end
