%PFE class definition
%
%   Usage:
%      cluster=discover();
%      cluster=discover('np',3);
%      cluster=discover('np',3,'login','username');

classdef discover 
    properties (SetAccess=public)  
		 % {{{
		 name=oshostname();
		 login='';
		 numnodes=20;
		 cpuspernode=8; 
		 port=0;
		 queue='general';
		 time=12*60;
		 processor='west';
		 codepath='';
		 executionpath='';
		 interactive=0;
		 bbftp=0;
		 numstreams=8;
		 hyperthreading=0;
	 end
	 properties (SetAccess=private) 
		 np=20*8;
		 % }}}
	 end
	 methods
		 function cluster=discover(varargin) % {{{

			 %initialize cluster using default settings if provided
			 if (exist('discover_settings')==2), discover_settings; end

			 %use provided options to change fields
			 cluster=AssignObjectFields(pairoptions(varargin{:}),cluster);
		 end
		 %}}}
		 function disp(cluster) % {{{
			 %  display the object
			 disp(sprintf('class ''%s'' object ''%s'' = ',class(cluster),inputname(1)));
			 disp(sprintf('    name: %s',cluster.name));
			 disp(sprintf('    login: %s',cluster.login));
			 disp(sprintf('    port: %i',cluster.port));
			 disp(sprintf('    numnodes: %i',cluster.numnodes));
			 disp(sprintf('    cpuspernode: %i',cluster.cpuspernode));
			 disp(sprintf('    np: %i',cluster.cpuspernode*cluster.numnodes));
			 disp(sprintf('    queue: %s',cluster.queue));
			 disp(sprintf('    time: %i',cluster.time));
			 disp(sprintf('    processor: %s',cluster.processor));
			 disp(sprintf('    codepath: %s',cluster.codepath));
			 disp(sprintf('    executionpath: %s',cluster.executionpath));
			 disp(sprintf('    interactive: %i',cluster.interactive));
			 disp(sprintf('    hyperthreading: %i',cluster.hyperthreading));
		 end
		 %}}}
		 function md = checkconsistency(cluster,md,solution,analyses) % {{{

			 available_queues={'general_long','general','debug'};
			 queue_requirements_time=[24*60 12*60 60];
			 queue_requirements_np=[516 1024 32];

			 QueueRequirements(available_queues,queue_requirements_time,queue_requirements_np,cluster.queue,cluster.np,cluster.time)

			 %now, check cluster.cpuspernode according to processor type
			 if ( strcmpi(cluster.processor,'sand')),
				 if ((cluster.cpuspernode>16 ) | (cluster.cpuspernode<1)),
					 md = checkmessage(md,'cpuspernode should be between 1 and 16 for ''sand'' processors');
				 end
			 elseif strcmpi(cluster.processor,'west'),
				 if ((cluster.cpuspernode>12 ) | (cluster.cpuspernode<1)),
					 md = checkmessage(md,'cpuspernode should be between 1 and 12 for ''west'' processors');
				 end
			 else
				 md = checkmessage(md,'unknown processor type, should be ''sand'' or ''west'' ');
			 end

			 %Miscelaneous
			 if isempty(cluster.login), md = checkmessage(md,'login empty'); end
			 if isempty(cluster.codepath), md = checkmessage(md,'codepath empty'); end
			 if isempty(cluster.executionpath), md = checkmessage(md,'executionpath empty'); end

		 end
		 %}}}
		 function BuildQueueScript(cluster,dirname,modelname,solution,io_gather,isvalgrind,isgprof) % {{{

			 if(isgprof),    disp('gprof not supported by cluster, ignoring...'); end

			 %compute number of processors
			 cluster.np=cluster.numnodes*cluster.cpuspernode;

			 %write queuing script 
			 fid=fopen([modelname '.queue'],'w');
			 fprintf(fid,'#!/bin/bash\n');
%			 fprintf(fid,'#PBS -N %s\n',modelname);
			 fprintf(fid,'#PBS -l select=%i:mpiprocs=%i:proc=%s\n',cluster.numnodes,cluster.cpuspernode,cluster.processor);
			 fprintf(fid,'#PBS -l walltime=%i:00\n',cluster.time); %SLURM walltime is in minutes:
			 if ~strcmp(cluster.queue,'general'),
				 fprintf(fid,'#PBS -q %s \n',cluster.queue);
			 end
			 fprintf(fid,'#PBS -W group_list=s1010\n');
			 fprintf(fid,'#PBS -m e\n');
			 fprintf(fid,'#PBS -o %s.outlog \n',[cluster.executionpath '/' dirname '/' modelname]);
			 fprintf(fid,'#PBS -e %s.errlog \n\n',[cluster.executionpath '/' dirname '/' modelname]);
			 fprintf(fid,'. /usr/share/modules/init/bash\n\n');
			 fprintf(fid,'module purge\n');
			 fprintf(fid,'module load comp/intel-11.1.038\n');
			 fprintf(fid,'module load mpi/impi-4.0.3.008\n');
			 fprintf(fid,'module load lib/mkl-10.1.2.024\n\n');
			 fprintf(fid,'export PATH="$PATH:."\n\n');
			 fprintf(fid,'export MPI_GROUP_MAX=64\n\n');
			 fprintf(fid,'cd %s/%s/\n\n',cluster.executionpath,dirname);
			 fprintf(fid,'mpirun -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
			 if ~io_gather, %concatenate the output files:
				 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
			 end
			 fclose(fid);

			 %in interactive mode, create a run file, and errlog and outlog file
			 if cluster.interactive,
				 fid=fopen([modelname '.run'],'w');
				 if ~isvalgrind,
					 fprintf(fid,'mpirun -np %i %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
				 else
					 fprintf(fid,'mpirun -np %i valgrind --leak-check=full %s/issm.exe %s %s %s\n',cluster.np,cluster.codepath,EnumToString(solution),[cluster.executionpath '/' dirname],modelname);
				 end
				 if ~io_gather, %concatenate the output files:
					 fprintf(fid,'cat %s.outbin.* > %s.outbin',modelname,modelname);
				 end
				 fclose(fid);
				 fid=fopen([modelname '.errlog'],'w');
				 fclose(fid);
				 fid=fopen([modelname '.outlog'],'w');
				 fclose(fid);
			 end
		 end %}}}
		 function UploadQueueJob(cluster,modelname,dirname,filelist)% {{{

			 %compress the files into one zip.
			 compressstring=['tar -zcf ' dirname '.tar.gz '];
			 for i=1:numel(filelist),
				 compressstring = [compressstring ' ' filelist{i}];
			 end
			 if cluster.interactive,
				 compressstring = [compressstring ' ' modelname '.run '  modelname '.errlog ' modelname '.outlog '];
			 end
			 system(compressstring);

			 disp('uploading input file and queueing script');
			 if cluster.interactive,
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive)];
			 else 
				 directory=cluster.executionpath;
			 end

			 if ~cluster.bbftp,
				 issmscpout(cluster.name,directory,cluster.login,cluster.port,{[dirname '.tar.gz']});
			 else
				 issmbbftpout(cluster.name,directory,cluster.login,cluster.port,cluster.numstreams,{[dirname '.tar.gz']});
			 end

		 end
		 %}}}
		 function LaunchQueueJob(cluster,modelname,dirname,filelist,restart)% {{{

			 if ~cluster.interactive, 
				 if ~isempty(restart)
					 launchcommand=['cd ' cluster.executionpath ' && cd ' dirname ' && qsub ' modelname '.queue '];
				 else
					 launchcommand=['cd ' cluster.executionpath ' && rm -rf ./' dirname ' && mkdir ' dirname ...
						 ' && cd ' dirname ' && mv ../' dirname '.tar.gz ./ && tar -zxf ' dirname '.tar.gz  && qsub ' modelname '.queue '];
				 end
			 else
				 if ~isempty(restart)
					 launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive)];
				 else
					 launchcommand=['cd ' cluster.executionpath '/Interactive' num2str(cluster.interactive) ' && tar -zxf ' dirname '.tar.gz'];
				 end
			 end

			 disp('launching solution sequence on remote cluster');
			 issmssh(cluster.name,cluster.login,cluster.port,launchcommand);
		 end
		 %}}}
		 function Download(cluster,dirname,filelist)% {{{

			 %copy files from cluster to current directory
			 if ~cluster.interactive,
				 directory=[cluster.executionpath '/' dirname '/'];
			 else
				 directory=[cluster.executionpath '/Interactive' num2str(cluster.interactive) '/'];
			 end

			 if ~cluster.bbftp,
				 issmscpin(cluster.name,cluster.login,cluster.port,directory,filelist);
			 else
				 issmbbftpin(cluster.name, cluster.login, cluster.port, cluster.numstreams, directory, filelist);
			 end

		 end %}}}
	end
end
