/*! \file IoModel.cpp
 * \brief  IoModel structure that mirrors the matlab workspace structure. Servers for the serial 
 * and parallel runs.
 */

#ifdef HAVE_CONFIG_H
	#include "config.h"
#else
#error "Cannot compile with HAVE_CONFIG_H symbol! run configure first!"
#endif

#include "../shared/shared.h"
#include "../io/io.h"
#include "../include/globals.h"
#include "../include/macros.h"

#include <string.h>
#include "stdio.h"

#include "./IoModel.h"


/*!--------------------------------------------------
	NewIoModel
  --------------------------------------------------*/

IoModel* NewIoModel(void) {
	/*! create a new IoModel object */
	IoModel* iomodel=NULL;

	iomodel=(IoModel*)xmalloc(sizeof(IoModel));

	/*!initialize all pointers to 0: */
	iomodel->name=NULL;
	iomodel->repository=NULL;
	iomodel->meshtype=NULL;
	iomodel->analysis_type=0;
	iomodel->sub_analysis_type=0;
	iomodel->qmu_analysis=0;
	iomodel->control_analysis=0;
	iomodel->solverstring=NULL;
	iomodel->numberofresponses=0;
	iomodel->numberofvariables=0;
	iomodel->qmu_npart=0; 
	iomodel->numberofelements=0;
	iomodel->numberofnodes=0;
	iomodel->x=NULL; 
	iomodel->y=NULL;
	iomodel->z=NULL;
	iomodel->elements=NULL;
	iomodel->elements_type=NULL;
	iomodel->numberofnodes2d=0;
	iomodel->elements2d=NULL;
	iomodel->deadgrids=NULL;
	iomodel->numlayers=0;
	iomodel->uppernodes=NULL;
	iomodel->gridonhutter=NULL;
	iomodel->gridonmacayeal=NULL;
	iomodel->gridonpattyn=NULL;
	
	iomodel->vx_obs=NULL;
	iomodel->vy_obs=NULL;
	iomodel->vx=NULL;
	iomodel->vy=NULL;
	iomodel->vz=NULL;
	iomodel->pressure=NULL;
	iomodel->temperature=NULL;
	iomodel->melting=NULL;
	iomodel->geothermalflux=NULL;
	iomodel->elementonbed=NULL;
	iomodel->elementonsurface=NULL;
	iomodel->gridonbed=NULL;
	iomodel->gridonsurface=NULL;
	iomodel->gridonstokes=NULL;
	iomodel->borderstokes=NULL;
	iomodel->thickness=NULL;
	iomodel->surface=NULL;
	iomodel->bed=NULL;
	iomodel->elementoniceshelf=NULL;
	iomodel->elementonwater=NULL;
	iomodel->gridonicesheet=NULL;
	iomodel->gridoniceshelf=NULL;

	iomodel->drag_type=0;
	iomodel->drag=NULL;
	iomodel->p=NULL;
	iomodel->q=NULL;
	
	
	iomodel->numberofpressureloads=0;
	iomodel->numberofpressureloads_stokes=0;
	iomodel->pressureload=NULL;
	iomodel->pressureload_stokes=NULL;
	iomodel-> spcvelocity=NULL;
	iomodel-> spctemperature=NULL;
	iomodel-> spcthickness=NULL;
	
	/*!materials: */
	iomodel->rho_water=0;
	iomodel->rho_ice=0;
	iomodel->g=0;
	iomodel->n=NULL;
	iomodel->B=NULL;

	/*!control methods: */
	iomodel->control_type=NULL;

	/*!solution parameters: */
	iomodel->fit=NULL;
	iomodel->meanvel=0;
	iomodel->epsvel=0;
	iomodel->artificial_diffusivity=0;
	iomodel->nsteps=0;
	iomodel->tolx=0;
	iomodel->maxiter=NULL;
	iomodel->mincontrolconstraint=0;
	iomodel->maxcontrolconstraint=0;
	iomodel->debug=0;
	iomodel->plot=0;
	iomodel->eps_res=0;
	iomodel->eps_rel=0;
	iomodel->eps_abs=0;
	iomodel->dt=0;
	iomodel->ndt=0;
	iomodel->penalty_offset=0;
	iomodel->penalty_melting=0;
	iomodel->penalty_lock=0;
	iomodel->sparsity=0;
	iomodel->connectivity=0;
	iomodel->lowmem=0;
	iomodel->optscal=NULL;
	iomodel->yts=0;
	iomodel->viscosity_overshoot=0;
	iomodel->stokesreconditioning=0;
	iomodel->waitonlock=0;

	/*!thermal parameters: */
	iomodel->beta=0;
	iomodel->meltingpoint=0;
	iomodel->latentheat=0;
	iomodel->heatcapacity=0;
	iomodel->thermalconductivity=0;
	iomodel->min_thermal_constraints=0;
	iomodel->mixed_layer_capacity=0;
	iomodel->thermal_exchange_velocity=0;

	
	iomodel->numrifts=0;
	iomodel->riftinfo=NULL;

	/*!penalties: */
	iomodel->numpenalties=0;
	iomodel->penalties=NULL;
	iomodel->penaltypartitioning=NULL;

	/*!basal: */
	iomodel->accumulation=NULL;

	/*elements type: */
	iomodel->ishutter=0;
	iomodel->ismacayealpattyn=0;
	iomodel->isstokes=0;


	iomodel->epart=NULL;
	iomodel->npart=NULL;
	iomodel->my_grids=NULL;
	iomodel->my_bordergrids=NULL;

	return iomodel;
}


/*!--------------------------------------------------
	DeleteIoModel
  --------------------------------------------------*/

void DeleteIoModel(IoModel** piomodel){

	/*!Recover structure: */
	IoModel* iomodel = *piomodel;
	
	int i;

	/*!Two cases here: 
	 * - serial mode: matlab's memory manager will take care of delete iomodel when returning from Imp. Do nothing here, so as not to confuse 
	 *                the memory manager.
     * - in parallel, anything the io layer does (FetchData) did needs to be erased explicitely in the iomodel.
	 */

	#ifdef _PARALLEL_
	xfree((void**)&iomodel->x);
	xfree((void**)&iomodel->y);
	xfree((void**)&iomodel->z);
	xfree((void**)&iomodel->elements);
	xfree((void**)&iomodel->elements_type);
	xfree((void**)&iomodel->gridonhutter);
	xfree((void**)&iomodel->gridonmacayeal);
	if (strcmp(iomodel->meshtype,"3d")==0){
		xfree((void**)&iomodel->elements2d);
		xfree((void**)&iomodel->deadgrids);
		xfree((void**)&iomodel->uppernodes);
		xfree((void**)&iomodel->gridonpattyn);
	}
	xfree((void**)&iomodel->solverstring);
	xfree((void**)&iomodel->elementonbed);
	xfree((void**)&iomodel->elementonsurface);
	xfree((void**)&iomodel->gridonbed);
	xfree((void**)&iomodel->gridonsurface);
	xfree((void**)&iomodel->gridonstokes);
	xfree((void**)&iomodel->borderstokes);
	xfree((void**)&iomodel->thickness);
	xfree((void**)&iomodel->surface);
	xfree((void**)&iomodel->bed);
	xfree((void**)&iomodel->vx_obs);
	xfree((void**)&iomodel->vy_obs);
	xfree((void**)&iomodel->vx);
	xfree((void**)&iomodel->vy);
	xfree((void**)&iomodel->vz);
	xfree((void**)&iomodel->pressure);
	xfree((void**)&iomodel->temperature);
	xfree((void**)&iomodel->drag);
	xfree((void**)&iomodel->p);
	xfree((void**)&iomodel->q);
	xfree((void**)&iomodel->elementoniceshelf);
	xfree((void**)&iomodel->elementonwater);
	xfree((void**)&iomodel->gridonicesheet);
	xfree((void**)&iomodel->gridoniceshelf);
	xfree((void**)&iomodel->pressureload);
	xfree((void**)&iomodel->pressureload_stokes);
	xfree((void**)&iomodel->spcvelocity);
	xfree((void**)&iomodel->spcthickness);
	xfree((void**)&iomodel->spctemperature);
	xfree((void**)&iomodel->geothermalflux);
	xfree((void**)&iomodel->melting);
	xfree((void**)&iomodel->accumulation);
	xfree((void**)&iomodel->B);
	xfree((void**)&iomodel->n);
	xfree((void**)&iomodel->fit);
	xfree((void**)&iomodel->optscal);
	xfree((void**)&iomodel->maxiter);


	/*!Delete structure fields: */
	xfree((void**)&iomodel->repository);
	xfree((void**)&iomodel->meshtype);
	xfree((void**)&iomodel->name);
	
	xfree((void**)&iomodel->riftinfo);
	
	xfree((void**)&iomodel->penalties);
	xfree((void**)&iomodel->penaltypartitioning);
	
	xfree((void**)&iomodel->control_type);
	
	xfree((void**)&iomodel->epart);
	xfree((void**)&iomodel->npart);
	xfree((void**)&iomodel->my_grids);
	xfree((void**)&iomodel->my_bordergrids);
	
	/*!Delete entire structure: */
	xfree((void**)piomodel);
	#endif
}

/*!--------------------------------------------------
	IoModelInit
  --------------------------------------------------*/

#undef __FUNCT__ 
#define __FUNCT__ "IoModelInit"

int	IoModelInit(IoModel** piomodel,ConstDataHandle iomodel_handle){

	int i,j;
	
	/*output: */
	IoModel* iomodel=NULL;

	/*Allocate iomodel: */
	iomodel=NewIoModel();

	/*In IoModelInit, we get all the data that is not difficult to get, and that is small: */
	IoModelFetchData((void**)&iomodel->name,NULL,NULL,iomodel_handle,"name","String",NULL); 
	IoModelFetchData((void**)&iomodel->analysis_type,NULL,NULL,iomodel_handle,"analysis_type","Integer",NULL); 
	IoModelFetchData((void**)&iomodel->sub_analysis_type,NULL,NULL,iomodel_handle,"sub_analysis_type","Integer",NULL); 
	IoModelFetchData((void**)&iomodel->qmu_analysis,NULL,NULL,iomodel_handle,"qmu_analysis","Integer",NULL); 
	IoModelFetchData((void**)&iomodel->control_analysis,NULL,NULL,iomodel_handle,"control_analysis","Integer",NULL); 
	IoModelFetchData((void**)&iomodel->meshtype,NULL,NULL,iomodel_handle,"type","String",NULL);
	/*!Get numberofelements and numberofnodes: */
	IoModelFetchData((void**)&iomodel->numberofnodes,NULL,NULL,iomodel_handle,"numberofgrids","Integer",NULL);
	IoModelFetchData((void**)&iomodel->numberofelements,NULL,NULL,iomodel_handle,"numberofelements","Integer",NULL);
	/*!In case we are running 3d, we are going to need the collapsed and non-collapsed 2d meshes, from which the 3d mesh was extruded: */
	if (strcmp(iomodel->meshtype,"3d")==0){
	
		/*!Deal with 2d mesh: */
		IoModelFetchData((void**)&iomodel->numberofelements2d,NULL,NULL,iomodel_handle,"numberofelements2d","Integer",NULL);
		IoModelFetchData((void**)&iomodel->numberofnodes2d,NULL,NULL,iomodel_handle,"numberofgrids2d","Integer",NULL);
		IoModelFetchData((void**)&iomodel->numlayers,NULL,NULL,iomodel_handle,"numlayers","Integer",NULL);
	}


	/*elements type: */
	IoModelFetchData((void**)&iomodel->ishutter,NULL,NULL,iomodel_handle,"ishutter","Integer",NULL);
	IoModelFetchData((void**)&iomodel->ismacayealpattyn,NULL,NULL,iomodel_handle,"ismacayealpattyn","Integer",NULL);
	IoModelFetchData((void**)&iomodel->isstokes,NULL,NULL,iomodel_handle,"isstokes","Integer",NULL);

	/*!Get drag_type, drag and p,q: */
	IoModelFetchData((void**)&iomodel->drag_type,NULL,NULL,iomodel_handle,"drag_type","Integer",NULL);

	/*!Get materials: */
	IoModelFetchData((void**)&iomodel->rho_water,NULL,NULL,iomodel_handle,"rho_water","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->rho_ice,NULL,NULL,iomodel_handle,"rho_ice","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->g,NULL,NULL,iomodel_handle,"g","Scalar",NULL);

	/*Get control parameters: */
	IoModelFetchData((void**)&iomodel->control_type,NULL,NULL,iomodel_handle,"control_type","String",NULL); 

	/*!Get solution parameters: */
	IoModelFetchData((void**)&iomodel->yts,NULL,NULL,iomodel_handle,"yts","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->meanvel,NULL,NULL,iomodel_handle,"meanvel","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->epsvel,NULL,NULL,iomodel_handle,"epsvel","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->debug,NULL,NULL,iomodel_handle,"debug","Integer",NULL);
	IoModelFetchData((void**)&iomodel->plot,NULL,NULL,iomodel_handle,"plot","Integer",NULL);
	IoModelFetchData((void**)&iomodel->artificial_diffusivity,NULL,NULL,iomodel_handle,"artificial_diffusivity","Integer",NULL);
	IoModelFetchData((void**)&iomodel->nsteps,NULL,NULL,iomodel_handle,"nsteps","Integer",NULL);
	IoModelFetchData((void**)&iomodel->tolx,NULL,NULL,iomodel_handle,"tolx","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->mincontrolconstraint,NULL,NULL,iomodel_handle,"mincontrolconstraint","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->maxcontrolconstraint,NULL,NULL,iomodel_handle,"maxcontrolconstraint","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->eps_res,NULL,NULL,iomodel_handle,"eps_res","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->eps_rel,NULL,NULL,iomodel_handle,"eps_rel","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->eps_abs,NULL,NULL,iomodel_handle,"eps_abs","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->dt,NULL,NULL,iomodel_handle,"dt","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->ndt,NULL,NULL,iomodel_handle,"ndt","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->penalty_offset,NULL,NULL,iomodel_handle,"penalty_offset","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->penalty_melting,NULL,NULL,iomodel_handle,"penalty_melting","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->penalty_lock,NULL,NULL,iomodel_handle,"penalty_lock","Integer",NULL);
	IoModelFetchData((void**)&iomodel->sparsity,NULL,NULL,iomodel_handle,"sparsity","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->connectivity,NULL,NULL,iomodel_handle,"connectivity","Integer",NULL);
	IoModelFetchData((void**)&iomodel->lowmem,NULL,NULL,iomodel_handle,"lowmem","Integer",NULL);
	IoModelFetchData((void**)&iomodel->solverstring,NULL,NULL,iomodel_handle,"solverstring","String",NULL);
	IoModelFetchData((void**)&iomodel->viscosity_overshoot,NULL,NULL,iomodel_handle,"viscosity_overshoot","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->stokesreconditioning,NULL,NULL,iomodel_handle,"stokesreconditioning","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->waitonlock,NULL,NULL,iomodel_handle,"waitonlock","Integer",NULL);

	/*!Get thermal parameters: */
	IoModelFetchData((void**)&iomodel->beta,NULL,NULL,iomodel_handle,"beta","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->meltingpoint,NULL,NULL,iomodel_handle,"meltingpoint","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->latentheat,NULL,NULL,iomodel_handle,"latentheat","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->heatcapacity,NULL,NULL,iomodel_handle,"heatcapacity","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->thermalconductivity,NULL,NULL,iomodel_handle,"thermalconductivity","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->min_thermal_constraints,NULL,NULL,iomodel_handle,"min_thermal_constraints","Integer",NULL);
	IoModelFetchData((void**)&iomodel->mixed_layer_capacity,NULL,NULL,iomodel_handle,"mixed_layer_capacity","Scalar",NULL);
	IoModelFetchData((void**)&iomodel->thermal_exchange_velocity,NULL,NULL,iomodel_handle,"thermal_exchange_velocity","Scalar",NULL);

	/*qmu: */
	if(iomodel->qmu_analysis){
		IoModelFetchData((void**)&iomodel->numberofvariables,NULL,NULL,iomodel_handle,"numberofvariables","Integer",NULL);
		IoModelFetchData((void**)&iomodel->numberofresponses,NULL,NULL,iomodel_handle,"numberofresponses","Integer",NULL);
		IoModelFetchData((void**)&iomodel->qmu_npart,NULL,NULL,iomodel_handle,"npart","Integer",NULL);
	}

	/*Assign output pointers: */
	*piomodel=iomodel;

	return 1;
}

/*!--------------------------------------------------
	IoModelEcho
  --------------------------------------------------*/
void IoModelEcho(IoModel* iomodel,int which_part,int rank) {

	//which_part  determines what gets echoed, otherwise, we'll get too much output.
	//1-> penalties

	int i,j;

	if(which_part==1 && my_rank==rank && (strcmp(iomodel->meshtype,"3d")==0)){
		printf("IoModel penalties: \n");
		printf("   number of penalties: %i\n",iomodel->numpenalties);
		printf("   grids: \n");

		for(i=0;i<iomodel->numpenalties;i++){
			for(j=0;j<iomodel->numlayers;j++){
				printf("%i ",(int)*(iomodel->penalties+iomodel->numlayers*i+j));
			}
			printf("\n");
		}
	}
	
	return;
}
