%MESH2D class definition
%
%   Usage:
%      mesh2d=mesh2d();

classdef mesh2d
	properties (SetAccess=public) 
		x                           = NaN;
		y                           = NaN;
		elements                    = NaN;
		numberofelements            = 0;
		numberofvertices            = 0;
		numberofedges               = 0;

		lat                         = NaN;
		long                        = NaN;
		epsg                        = 0;

		vertexonboundary            = NaN;

		edges                       = NaN;
		segments                    = NaN;
		segmentmarkers              = NaN;
		vertexconnectivity          = NaN;
		elementconnectivity         = NaN;
		average_vertex_connectivity = 0;

		extractedvertices           = NaN;
		extractedelements           = NaN;
	end
	methods (Static)
		function self = loadobj(self) % {{{
			% This function is directly called by matlab when a model selfect is
			% loaded. Update old properties here

			%2014 Oct. 1st
			if isstruct(self),
				oldself=self;
				%Assign property values from struct
				self=structtoobj(mesh2d(),oldself);
				if isfield(oldself,'hemisphere'),
					disp('md.mesh.hemisphere has been automatically converted to EPSG code');
					if strcmpi(oldself.hemisphere,'n'),
						self.epsg=3413;
					else
						self.epsg=3031;
					end
				end
			end

		end% }}}
	end
	methods
		function self = mesh2d(varargin) % {{{
			switch nargin
				case 0
					self=setdefaultparameters(self);
				case 1
					self=mesh2d();
					object=varargin{1};
					fields=fieldnames(object);
					for i=1:length(fields)
						field=fields{i};
						if ismember(field,properties('mesh2d')),
							self.(field)=object.(field);
						end
					end
				otherwise
					error('constructor not supported');
			end
		end % }}}
		function self = setdefaultparameters(self) % {{{

			%the connectivity is the averaged number of nodes linked to a
			%given node through an edge. This connectivity is used to initially
			%allocate memory to the stiffness matrix. A value of 16 seems to
			%give a good memory/time ration. This value can be checked in
			%trunk/test/Miscellaneous/runme.m
			self.average_vertex_connectivity=25;
		end % }}}
		function md = checkconsistency(self,md,solution,analyses) % {{{

			md = checkfield(md,'fieldname','mesh.x','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','mesh.y','NaN',1,'size',[md.mesh.numberofvertices 1]);
			md = checkfield(md,'fieldname','mesh.elements','NaN',1,'>',0,'values',1:md.mesh.numberofvertices);
			md = checkfield(md,'fieldname','mesh.elements','size',[md.mesh.numberofelements 3]);
			if any(~ismember(1:md.mesh.numberofvertices,sort(unique(md.mesh.elements(:)))));
				md = checkmessage(md,'orphan nodes have been found. Check the mesh outline');
			end
			md = checkfield(md,'fieldname','mesh.numberofelements','>',0);
			md = checkfield(md,'fieldname','mesh.numberofvertices','>',0);
			md = checkfield(md,'fieldname','mesh.average_vertex_connectivity','>=',9,'message','''mesh.average_vertex_connectivity'' should be at least 9 in 2d');

			switch(solution),
				case ThermalSolutionEnum(),
					md = checkmessage(md,'thermal not supported for 2d mesh');
			end
		end % }}}
		function disp(self) % {{{
			disp(sprintf('   2D tria Mesh (horizontal):')); 

			disp(sprintf('\n      Elements and vertices:'));
			fielddisplay(self,'numberofelements','number of elements');
			fielddisplay(self,'numberofvertices','number of vertices');
			fielddisplay(self,'elements','vertex indices of the mesh elements');
			fielddisplay(self,'x','vertices x coordinate [m]');
			fielddisplay(self,'y','vertices y coordinate [m]');
			fielddisplay(self,'edges','edges of the 2d mesh (vertex1 vertex2 element1 element2)');
			fielddisplay(self,'numberofedges','number of edges of the 2d mesh');

			disp(sprintf('\n      Properties:'));
			fielddisplay(self,'vertexonboundary','vertices on the boundary of the domain flag list');
			fielddisplay(self,'segments','edges on domain boundary (vertex1 vertex2 element)');
			fielddisplay(self,'segmentmarkers','number associated to each segment');
			fielddisplay(self,'vertexconnectivity','list of vertices connected to vertex_i');
			fielddisplay(self,'elementconnectivity','list of vertices connected to element_i');
			fielddisplay(self,'average_vertex_connectivity','average number of vertices connected to one vertex');

			disp(sprintf('\n      Extracted model:'));
			fielddisplay(self,'extractedvertices','vertices extracted from the model');
			fielddisplay(self,'extractedelements','elements extracted from the model');

			disp(sprintf('\n      Projection:'));
			fielddisplay(self,'lat','vertices latitude [degrees]');
			fielddisplay(self,'long','vertices longitude [degrees]');
			fielddisplay(self,'epsg','EPSG code (ex: 3413 for UPS Greenland, 3031 for UPS Antarctica)');
		end % }}}
		function createxml(self,fid) % {{{
			fprintf(fid, '<!-- 2D tria Mesh (horizontal) -->\n');

			%elements and vertices
			fprintf(fid,'%s\n%s\n%s\n','<frame key="1" label="Elements and vertices">','<section name="mesh" />');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','  <parameter key ="numberofelements" type="',class(self.numberofelements),'" default="',convert2str(self.numberofelements),'">','     <section name="mesh" />','     <help> number of elements </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','  <parameter key ="numberofvertices" type="',class(self.numberofvertices),'" default="',convert2str(self.numberofvertices),'">','     <section name="mesh" />','     <help> number of vertices </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n','  <parameter key ="elements" type="',class(self.elements),'" default="',convert2str(self.elements),'">','     <section name="mesh" />','     <help> vertex indices of the mesh elements </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="x" type="',class(self.x),'" default="',convert2str(self.x),'">','     <section name="mesh" />','     <help> vertices x coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="y" type="',class(self.y),'" default="',convert2str(self.y),'">','     <section name="mesh" />','     <help> vertices y coordinate [m] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="edges" type="',class(self.edges),'" default="',convert2str(self.edges),'">','     <section name="mesh" />','     <help> edges of the 2d mesh (vertex1 vertex2 element1 element2) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="numberofedges" type="',class(self.numberofedges),'" default="',convert2str(self.numberofedges),'">','     <section name="mesh" />','     <help> number of edges of the 2d mesh </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

			% properties
			fprintf(fid,'%s\n%s\n%s\n','<frame key="2" label="Properties">','<section name="mesh" />');             
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexonboundary" type="',class(self.vertexonboundary),'" default="',convert2str(self.vertexonboundary),'">','     <section name="mesh" />','     <help> vertices on the boundary of the domain flag list </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="segments" type="',class(self.segments),'" default="',convert2str(self.segments),'">','     <section name="mesh" />','     <help> edges on domain boundary (vertex1 vertex2 element) </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="segmentmarkers" type="',class(self.segmentmarkers),'" default="',convert2str(self.segmentmarkers),'">','     <section name="mesh" />','     <help> number associated to each segment </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="vertexconnectivity" type="',class(self.vertexconnectivity),'" default="',convert2str(self.vertexconnectivity),'">','     <section name="mesh" />','     <help> list of vertices connected to vertex_i </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="elementconnectivity" type="',class(self.elementconnectivity),'" default="',convert2str(self.elementconnectivity),'">','     <section name="mesh" />','     <help> list of vertices connected to element_i </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="average_vertex_connectivity" type="',class(self.average_vertex_connectivity),'" default="',convert2str(self.average_vertex_connectivity),'">','     <section name="mesh" />','     <help> average number of vertices connected to one vertex </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

			%extracted model
			fprintf(fid,'%s\n%s\n%s\n','<frame key="3" label="Extracted Model">','<section name="mesh" />'); 
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="extractedvertices" type="',class(self.extractedvertices),'" default="',convert2str(self.extractedvertices),'">','     <section name="mesh" />','     <help> vertices extracted from the model </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="extractedelements" type="',class(self.extractedelements),'" default="',convert2str(self.extractedelements),'">','     <section name="mesh" />','     <help> elements extracted from the model </help>','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

			%projection
			fprintf(fid,'%s\n%s\n%s\n','<frame key="4" label="Projection">','<section name="mesh" />'); 
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="lat" type="',class(self.lat),'" default="',convert2str(self.lat),'">','     <section name="mesh" />','     <help> vertices latitude [degrees] </help>','  </parameter>');
			fprintf(fid,'%s%s%s%s%s\n%s\n%s\n%s\n','  <parameter key ="long" type="',class(self.long),'" default="',convert2str(self.long),'">','     <section name="mesh" />','     <help> verticies longitude [degrees] </help>','  </parameter>');
			% choice (epsg) 'n' or 's'
			fprintf(fid,'%s\n%s\n%s\n','  <parameter key ="epsg" type="alternative" optional="false">','     <section name="mesh" />','     <help> Indicate epsg ''n'' or ''s'' </help>');
			fprintf(fid,'%s\n','       <option value="n" type="string" default="true"> </option>');
			fprintf(fid,'%s\n','       <option value="s" type="string" default="false"> </option>');
			fprintf(fid,'%s\n','  </parameter>');
			fprintf(fid,'%s\n%s\n','</frame>');

		end % }}}
		function marshall(self,md,fid) % {{{
			WriteData(fid,'enum',DomainTypeEnum(),'data',StringToEnum(['Domain' domaintype(self)]),'format','Integer');
			WriteData(fid,'enum',DomainDimensionEnum(),'data',dimension(self),'format','Integer');
			WriteData(fid,'enum',MeshElementtypeEnum(),'data',StringToEnum(elementtype(self)),'format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','x','format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','y','format','DoubleMat','mattype',1);
			WriteData(fid,'enum',MeshZEnum(),'data',zeros(self.numberofvertices,1),'format','DoubleMat','mattype',1);
			WriteData(fid,'object',self,'class','mesh','fieldname','elements','format','DoubleMat','mattype',2);
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofelements','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','numberofvertices','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','average_vertex_connectivity','format','Integer');
			WriteData(fid,'object',self,'class','mesh','fieldname','vertexonboundary','format','DoubleMat','mattype',1);
		end % }}}
		function t = domaintype(self) % {{{
			t = '2Dhorizontal';
		end % }}}
		function d = dimension(self) % {{{
			d = 2;
		end % }}}
		function s = elementtype(self) % {{{
			s = 'Tria';
		end % }}}
	end
end
